# coding=utf-8
from . import common

import os
import logging

log = logging.getLogger("fetchers")


class IYPFetcher(object):
    def query_accounts(self):
        """
        :rtype: typing.Iterable[str, dict, dict]
        """
        raise NotImplementedError

    def query_segments(self):
        """
        :rtype: typing.Iterable[str, dict]
        """


class YPFetcher(IYPFetcher):
    def __init__(self, client):
        """

        :type client: yp.client.YpClient
        """
        self.client = client

    def query_accounts(self):
        processed = 0

        query_limit = 500
        last_account = None
        while True:
            if last_account is None:
                filter_ = None
            else:
                filter_ = "([/meta/id])>'{}'".format(last_account)
            account_chunk = self.client.select_objects(
                "account",
                selectors=["/meta/id", "/spec", "/status"],
                limit=query_limit,
                filter=filter_,
            )
            if len(account_chunk) == 0:
                break

            for item in account_chunk:
                yield item
                last_account = item[0]
                processed += 1

            logging.debug("Fetched accounts count: {}".format(processed))

    def query_segments(self):

        bigdata = self.client.select_objects(
            "node_segment",
            selectors=["/meta/id", "/status/total_resources"],
            filter="[/status/total_resources/cpu/capacity] > 0"
        )
        logging.debug("Fetched segments count: {}".format(len(bigdata)))
        return bigdata


class IABCServiceFetcher(object):
    def fetch_all_service_ids(self):
        """

        :rtype: typing.Set[int]
        """
        raise NotImplementedError


class HttpAbcServiceFetcher(IABCServiceFetcher):
    ABC_URL = "https://abc-back.yandex-team.ru/api/v4"

    def __init__(self):
        self.session = common.create_session(os.environ["ABC_OAUTH"])

    def fetch_all_service_ids(self):
        services = set()

        query = "{}/services/?fields=id&state__in=develop,supported,needinfo".format(self.ABC_URL)
        while query:
            results = self.session.get(query).json()

            for service in results["results"]:
                services.add(service["id"])
            print("Read batch of ABC services (Size: {})".format(len(results["results"])))

            query = results.get("next")

        return services
