from . import models
from . import fetchers
from . import reporters

from yp.client import YpClient
from yp.client import find_token

import click

import dataclasses
import logging
import os
import requests
import six
import sys
import typing  # noqa


def setup_custom_logger(name):
    formatter = logging.Formatter(fmt='%(asctime)s %(levelname)-8s %(message)s',
                                  datefmt='%Y-%m-%d %H:%M:%S')
    screen_handler = logging.StreamHandler(stream=sys.stdout)
    screen_handler.setFormatter(formatter)
    logger = logging.getLogger(name)
    logger.handlers = []
    logger.setLevel(logging.DEBUG)
    logger.addHandler(screen_handler)
    return logger


LOGGER = setup_custom_logger('')
requests.packages.urllib3.disable_warnings()


def get_current_cluster():
    if "YP_MONITORING_CLUSTERS" in os.environ:
        return os.environ["YP_MONITORING_CLUSTERS"].split(",")

    print("Host={}".format(os.uname()[1]))
    return [os.uname()[1].split(".")[1]]


@dataclasses.dataclass()
class Bootstrap(object):
    abc: fetchers.IABCServiceFetcher
    juggler_reporter: reporters.IJugglerReporter
    cluster: str
    yp_fetcher: fetchers.IYPFetcher
    solomon_reporter: reporters.ISolomonReporter


def setup_bootstrap(cluster: str, push: bool):
    client = YpClient(
        address=cluster,
        config={
            "token": find_token(),
            "retries":
                {
                    "enable": True,
                    "count": 3,
                    "backoff":
                        {
                            "policy": "constant_time",
                            "constant_time": 1000
                        }
                }
        }
    )

    bs = Bootstrap(
        abc=fetchers.HttpAbcServiceFetcher(),
        juggler_reporter=reporters.HttpJugglerReporter(push),
        cluster=cluster,
        yp_fetcher=fetchers.YPFetcher(client),
        solomon_reporter=reporters.HttpSolomonReporter(push),
    )
    return bs


@dataclasses.dataclass()
class Statistics(object):
    sensors_pushed: int = 0
    accounts_pushed: int = 0

    def __iadd__(self, other):
        self.sensors_pushed += other.sensors_pushed
        self.accounts_pushed += other.accounts_pushed
        return self


def monitor_orphaned_abc_accounts(bootstrap: Bootstrap, yp_abc_accounts: typing.Set[str]):
    live_abc_services = ["abc:service:{}".format(x) for x in bootstrap.abc.fetch_all_service_ids()]
    yp_orphaned_abc_accounts = list(yp_abc_accounts.difference(live_abc_services))

    reporters.report_orphaned_abc_yp_accounts(
        bootstrap.juggler_reporter,
        bootstrap.cluster,
        yp_orphaned_abc_accounts,
    )
    bootstrap.juggler_reporter.notify(
        bootstrap.cluster,
        "OK",
        "OK",
        reporters.JugglerServices.YP_ACCOUNT_MONITORING,
    )


def monitor_hardware_capacity(bootstrap: Bootstrap):
    segments_hardware_capacity = models.find_segments(bootstrap.yp_fetcher, bootstrap.cluster)

    reporters.report_hardware_limits(
        bootstrap.cluster,
        segments_hardware_capacity,
        bootstrap.solomon_reporter,
    )


def process_cluster(bootstrap: Bootstrap):
    yp_abc_accounts = set()
    total_quota = models.PerSegmentVectorizedResources("[total]", {})
    total_usage = models.PerSegmentVectorizedResources("[total]", {})
    total_overdraft = models.PerSegmentVectorizedResources("[total]", {})
    pushed_accounts = set()

    overdraft_reports = []  # type: typing.List[models.AccountSegmentOverdraftReport]
    for account_id, quota, usage in models.find_accounts(bootstrap.yp_fetcher, bootstrap.cluster):
        if account_id.startswith("abc:") and not quota.is_empty():
            yp_abc_accounts.add(account_id)
            # NB: non-abc accounts are ephemeral or special and may overcommit cluster
            total_quota.add_resources_from(quota)
            total_usage.add_resources_from(usage)

        LOGGER.debug("Processing account [{}]".format(account_id))

        for segment in set(six.iterkeys(usage.resources_per_segment)) | set(six.iterkeys(quota.resources_per_segment)):
            segment_usage = usage.resources_per_segment.get(segment, models.TResourceGang({}))
            segment_quota = quota.resources_per_segment.get(segment, models.TResourceGang({}))

            overdraft = segment_usage - segment_quota
            overdraft.drop_by(0)

            if account_id.startswith("abc:"):
                total_overdraft.resources_per_segment.setdefault(segment, models.TResourceGang({}))
                total_overdraft.resources_per_segment[segment] += overdraft

            if not segment_usage.is_empty() or not segment_quota.is_empty():
                overdraft_reports.append(
                    models.AccountSegmentOverdraftReport(
                        account_id,
                        segment,
                        segment_usage,
                        segment_quota,
                        overdraft,
                    )
                )

    for report in overdraft_reports:
        pushed_accounts.add(report.account_id)
        reporters.report_account_overdraft(
            bootstrap.cluster,
            report,
            bootstrap.solomon_reporter,
        )

    for segment in six.iterkeys(total_overdraft.resources_per_segment):
        total_quota.resources_per_segment.setdefault(segment, models.TResourceGang({}))
        total_usage.resources_per_segment.setdefault(segment, models.TResourceGang({}))

    for segment in six.iterkeys(total_quota.resources_per_segment):
        reporters.report_account_overdraft(
            bootstrap.cluster,
            models.AccountSegmentOverdraftReport(
                total_quota.id,
                segment,
                total_usage.resources_per_segment[segment],
                total_quota.resources_per_segment[segment],
                total_overdraft.resources_per_segment[segment],
            ),
            bootstrap.solomon_reporter,
        )

    monitor_hardware_capacity(bootstrap)

    monitor_orphaned_abc_accounts(bootstrap, yp_abc_accounts)

    result = Statistics(
        sensors_pushed=bootstrap.solomon_reporter.get_pushed_sensor_count(),
        accounts_pushed=len(pushed_accounts)
    )
    logging.info("Statistics for {}: {}".format(bootstrap.cluster, dataclasses.asdict(result)))
    return result


@click.command()
@click.option("--push/--no-push", default=True)
def main(push):
    statistics = Statistics()
    for cluster in get_current_cluster():
        LOGGER.info("Processing {}".format(cluster))
        bootstrap = setup_bootstrap(cluster, push)
        cluster_stats = process_cluster(bootstrap)
        statistics += cluster_stats

    logging.info("Statistics: {}".format(dataclasses.asdict(statistics)))
