# coding=utf-8

from . import fetchers

from yp_proto.yp.client.api.proto import data_model_pb2
from google.protobuf import json_format

import dataclasses
import logging
import six
import typing  # noqa


class EResourceKinds(object):
    CPU = "cpu"
    MEMORY = "memory"
    DISK = "disk"
    INTERNET_ADDRESS = "internet_address"
    GPU = "gpu"


NONE_RESOURCE_FLAVOR = "none"


class EResourceMetrics(object):
    CAPACITY = "capacity"
    BANDWIDTH = "bandwidth"


@dataclasses.dataclass(order=True, frozen=True)
class TResource(object):
    kind: str
    flavor: str
    metric: str


class TResourceGang(object):
    def __init__(self, items):
        """
        :type items: typing.Dict[TResource, int]
        """
        self.items = items

    def is_empty(self):
        return all(
            x == 0 for x in six.itervalues(self.items)
        )

    def __sub__(self, other):
        """

        :type other: TResourceGang
        """
        result = {}
        for resource in set(six.iterkeys(self.items)) | set(six.iterkeys(other.items)):
            result[resource] = self.items.get(resource, 0) - other.items.get(resource, 0)

        return TResourceGang(result)

    def __add__(self, other):
        result = {}
        for resource in set(six.iterkeys(self.items)) | set(six.iterkeys(other.items)):
            result[resource] = self.items.get(resource, 0) + other.items.get(resource, 0)

        return TResourceGang(result)

    def drop_by(self, value):
        keys = list(six.iterkeys(self.items))
        for resource in keys:
            if self.items[resource] <= value:
                del self.items[resource]


def parse_resource_totals(resources):
    """
    :type resources: data_model_pb2.TPerSegmentResourceTotals
    :rtype: TResourceGang
    """
    result = {
        TResource(EResourceKinds.CPU, NONE_RESOURCE_FLAVOR, EResourceMetrics.CAPACITY): resources.cpu.capacity,
        TResource(EResourceKinds.MEMORY, NONE_RESOURCE_FLAVOR, EResourceMetrics.CAPACITY): resources.memory.capacity,
        TResource(EResourceKinds.INTERNET_ADDRESS, NONE_RESOURCE_FLAVOR, EResourceMetrics.CAPACITY):
            resources.internet_address.capacity,
    }

    for storage_class, value in resources.disk_per_storage_class.items():
        resource = TResource(EResourceKinds.DISK, storage_class, EResourceMetrics.CAPACITY)
        result[resource] = value.capacity
        resource = TResource(EResourceKinds.DISK, storage_class, EResourceMetrics.BANDWIDTH)
        result[resource] = value.bandwidth

    for gpu_model, gpu_totals in resources.gpu_per_model.items():
        if gpu_model == "0":
            # Seems like junk data yn YP database
            continue
        resource = TResource(EResourceKinds.GPU, gpu_model, EResourceMetrics.CAPACITY)
        result[resource] = gpu_totals.capacity

    return TResourceGang(result)


def parse_per_segment_resource_totals(object_id, segment_to_resources_dict):
    """

    :type object_id: str
    :type segment_to_resources_dict: dict
    :rtype: PerSegmentVectorizedResources
    """
    resources_per_segment = dict()

    for segment, value in six.iteritems(segment_to_resources_dict):
        resource_totals = data_model_pb2.TPerSegmentResourceTotals()
        json_format.ParseDict(value, resource_totals)
        resources_per_segment[segment] = parse_resource_totals(resource_totals)

    return PerSegmentVectorizedResources(object_id, resources_per_segment)


class PerSegmentVectorizedResources(object):
    def __init__(self, object_id, accounts_pers_segments):
        """

        :type object_id: str
        :type accounts_pers_segments: typing.Dict[str, TResourceGang]
        """
        self.id = object_id
        self.resources_per_segment = accounts_pers_segments

    def add_resources_from(self, other):
        """
        :type other: PerSegmentVectorizedResources
        """
        for segment, resource_gang in six.iteritems(other.resources_per_segment):
            self.resources_per_segment.setdefault(segment, TResourceGang({}))
            self.resources_per_segment[segment] += resource_gang

    def is_empty(self):
        return all(x.is_empty() for x in six.itervalues(self.resources_per_segment))


def find_accounts(yp_fetcher, cluster):
    # type: (fetchers.IYPFetcher, str) -> typing.Iterable[typing.Tuple[str, PerSegmentVectorizedResources, PerSegmentVectorizedResources]]

    logging.info("Retrieving accounts {}".format(cluster))
    for account_id, spec, status in yp_fetcher.query_accounts():
        available_per_segment = parse_per_segment_resource_totals(
            account_id,
            spec.get("resource_limits", {}).get("per_segment", {}),
        )
        used_per_segment = parse_per_segment_resource_totals(
            account_id,
            status.get("immediate_resource_usage", {}).get("per_segment", {}),
        )

        yield account_id, available_per_segment, used_per_segment


def find_segments(yp_fetcher, cluster):
    # type: (fetchers.IYPFetcher, str) -> PerSegmentVectorizedResources

    logging.info("Retrieving accounts {}".format(cluster))
    segments_data = {x[0]: x[1] for x in yp_fetcher.query_segments()}
    total_resources = parse_per_segment_resource_totals(
        "[hardware_capacity]",
        segments_data,
    )
    return total_resources


@dataclasses.dataclass
class AccountSegmentOverdraftReport(object):
    account_id: str
    segment: str
    usage: TResourceGang
    quota: TResourceGang
    overdraft: TResourceGang
