# coding=utf-8

from . import common

import logging
import os
import six


class IJugglerReporter(object):
    def notify(self, cluster, status, description, service):
        """
        :type cluster: str
        :type status: str
        :type description: str
        :type service: str
        """
        raise NotImplementedError


class JugglerReporterBase(IJugglerReporter):
    def _format_report(self, cluster, status, description, service):
        logging.info("Notifying cluster {} status={} description={}".format(cluster, status, description))

        notify_data = {
            "source": "yp_account_monitoring",
            "events": [
                {
                    "description": description,
                    "host": "yp-{}.yandex.net".format(cluster),
                    "instance": "",
                    "service": service,
                    "status": status
                }
            ]
        }
        return notify_data

    def _do_notify(self, payload):
        raise NotImplementedError

    def notify(self, cluster, status, description, service):
        payload = self._format_report(cluster, status, description, service)
        self._do_notify(payload)
        logging.info("Notify sent. status={}, data={}".format(status, payload))


class HttpJugglerReporter(JugglerReporterBase):
    def __init__(self, push):
        self._push = push
        if push:
            self._session = common.create_session()
        else:
            self._session = None

    def _do_notify(self, payload):
        if not self._push:
            return
        result = self._session.post("http://juggler-push.search.yandex.net/events", json=payload, timeout=10)
        result.raise_for_status()


class JugglerServices(object):
    YP_ACCOUNT_MONITORING = "yp_account_monitoring"
    YP_ABC_ORPHANED_ACCOUNTS = "yp_abc_orphaned_accounts"


class ISolomonReporter(object):
    def push_sensors(self, project, service, cluster, common_labels, sensors):
        raise NotImplementedError

    def get_pushed_sensor_count(self):
        raise NotImplementedError


class HttpSolomonReporter(ISolomonReporter):
    def __init__(self, push):
        self._push = push
        if push:
            self._session = common.create_session(os.environ["SOLOMON_TOKEN"])
        else:
            self._session = None
        self._sensors_pushed = 0

    def push_sensors(self, project, service, cluster, common_labels, sensors):
        params = dict(project=project, service=service, cluster=cluster)
        data = dict(
            commonLabels=common_labels,
            sensors=sensors,
        )
        if self._push:
            url = "https://api.solomon.search.yandex.net/api/v2/push"
            result = self._session.post(url, params=params, json=data, timeout=60)
            result.raise_for_status()
        self._sensors_pushed += len(sensors)

    def get_pushed_sensor_count(self):
        return self._sensors_pushed


def extend_basic_sensors(sensors, resource_gang, name, extra_labels):
    for resource, value in six.iteritems(resource_gang.items):
        sensors.append(
            {
                "labels": {
                    "sensor": "resource_{}".format(name),
                    "kind": resource.kind,
                    "flavor": resource.flavor,
                    "metric": resource.metric,
                    **extra_labels,
                },
                "value": value,
            },
        )


def report_account_overdraft(cluster, report, solomon_reporter):
    """

    :type cluster: str
    :type report: models.AccountSegmentOverdraftReport
    :type solomon_reporter: reporters.ISolomonReporter
    """

    project = "yp"
    service = "account_monitoring"
    cluster = "yp-{}".format(cluster)
    common_labels = {
        "account": report.account_id,
        "segment": report.segment,
    }
    sensors = list()

    extend_basic_sensors(sensors, report.overdraft, "overdraft", common_labels)
    extend_basic_sensors(sensors, report.usage, "usage", common_labels)
    extend_basic_sensors(sensors, report.quota, "limit", common_labels)

    solomon_reporter.push_sensors(project, service, cluster, {}, sensors)


def report_hardware_limits(cluster, resources, solomon_reporter):
    """

    :type cluster: str
    :type resources: models.PerSegmentVectorizedResources
    :type solomon_reporter: reporters.ISolomonReporter
    """

    project = "yp"
    service = "account_monitoring"
    cluster = "yp-{}".format(cluster)
    sensors = list()
    for segment, resource_gang in six.iteritems(resources.resources_per_segment):
        extend_basic_sensors(sensors, resource_gang, "hardware_capacity", {"segment": segment})

    solomon_reporter.push_sensors(project, service, cluster, {}, sensors)


def report_orphaned_abc_yp_accounts(juggler_reporter, cluster, yp_orphaned_abc_accounts):
    """
    :type juggler_reporter: IJugglerReporter
    :type cluster: str
    :type yp_orphaned_abc_accounts: typing.Any
    """

    if len(yp_orphaned_abc_accounts) == 0:
        juggler_reporter.notify(cluster, "OK", "OK", JugglerServices.YP_ABC_ORPHANED_ACCOUNTS)
    else:
        accounts_str = ",".join(yp_orphaned_abc_accounts[:min(2, len(yp_orphaned_abc_accounts))])
        if len(yp_orphaned_abc_accounts) > 2:
            accounts_str += " and more"

        juggler_reporter.notify(
            cluster,
            "CRIT",
            "Orphaned abc accounts " + accounts_str,
            JugglerServices.YP_ABC_ORPHANED_ACCOUNTS,
        )
