import json
import argparse
import ipaddress
import logging
import os
import requests
import retry
import sys

from lxml import etree as et
from requests.packages.urllib3 import Retry

from sandbox.projects.common.nanny.client import NannyClient


logger = logging.getLogger()
sh = logging.StreamHandler()
sh.setFormatter(logging.Formatter('[%(asctime)s] [%(levelname)s] %(message)s'))
logger.addHandler(sh)


NANNY_URL = 'http://nanny.yandex-team.ru'
RACKTABLES_ZONES_URL = 'https://ro.racktables.yandex.net/export/zones.xml'

YP_DNS_PORT = 9091

SKIP_ZONES = set([
    '158.158.93.in-addr.arpa',
])

YP_CLUSTERS = set([
    'sas-test',
    'man-pre',
    'sas',
    'man',
    'vla',
    'myt',
    'iva',
    'xdc',
])


class GetConfigException(Exception):
    pass


def get_retry_session(retries=5, backoff_factor=2, status_forcelist=(500, 502, 503, 504)):
    session = requests.Session()
    adapter = requests.adapters.HTTPAdapter(max_retries=Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
    ))
    session.mount('http://', adapter)
    return session


@retry.retry(tries=5, delay=2, backoff=2)
def juggler_notify(host, status, description):
    result = get_retry_session().post(
        "http://juggler-push.search.yandex.net/events",
        json={
            "source": "racktables_zones_list_monitoring",
            "events": [
                {
                    "description": description,
                    "host": host,
                    "instance": "",
                    "service": "racktables_zones_list_monitoring",
                    "status": status,
                }
            ]
        },
        timeout=10
    )
    result.raise_for_status()


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--nanny-services', required=True)
    parser.add_argument('--nanny-token')
    parser.add_argument('--verbose', action='store_true')
    args = parser.parse_args(argv)

    args.nanny_services = args.nanny_services.split(',')

    if args.nanny_token is not None:
        os.environ['OAUTH_NANNY'] = args.nanny_token

    return args


@retry.retry(tries=5, delay=2, backoff=2)
def fetch_racktables_zones_xml():
    result = get_retry_session().get(RACKTABLES_ZONES_URL)
    result.raise_for_status()
    return result.content


def common_suffix(lhs, rhs):
    for i in range(min(len(lhs), len(rhs))):
        if lhs[-i - 1] != rhs[-i - 1]:
            return lhs[-i:]


def get_racktables_zones(checks):
    zones_xml = fetch_racktables_zones_xml()
    root = et.fromstring(zones_xml)

    result = []
    for elem in root:
        checks_satisfied = {name: False for name in checks.keys()}
        network = ''
        for prop in elem:
            if prop.tag == 'network':
                network = prop.text
            elif prop.tag in checks:
                if prop.text == checks[prop.tag]:
                    checks_satisfied[prop.tag] = True
        if all(checks_satisfied.values()):
            net = ipaddress.ip_network(unicode(network))
            zone = common_suffix(net.network_address.reverse_pointer, net.broadcast_address.reverse_pointer)
            result.append(zone[1:])
    return result


def get_racktables_qloud_zones():
    return set(get_racktables_zones({'infrastructure': 'qloud'})) - SKIP_ZONES


def get_nanny_service_instance(nanny_client, service):
    return map(lambda instance: instance['container_hostname'],
               nanny_client.get_service_current_instances(service)['result'])


def get_yp_dns_config(instance):
    r = get_retry_session().get('http://{}:{}/config'.format(instance, YP_DNS_PORT))
    r.raise_for_status()
    return r.content


def get_yp_dns_zones(instance):
    try:
        config_raw = get_yp_dns_config(instance)
    except Exception as e:
        raise GetConfigException(e)
    config = json.loads(config_raw)
    return config['Zones']


def format_list(values, max_elems=3):
    result = ', '.join(list(values)[:max_elems])
    if len(values) > max_elems:
        result += ' and {} more'.format(len(values) - max_elems)
    return result


def check_zones_list(nanny_service):
    nanny_client = NannyClient(NANNY_URL, os.environ['OAUTH_NANNY'])
    actual_zones = get_racktables_qloud_zones()

    instances = get_nanny_service_instance(nanny_client, nanny_service)
    get_config_errors = 0
    for instance in instances:
        try:
            try:
                yp_dns_zones = get_yp_dns_zones(instance)
            except GetConfigException as e:
                logger.warn('Failed to get config from {}: {}'.format(instance, e))
                get_config_errors += 1
                continue

            yp_dns_zones_names = set(map(lambda zone: zone['Name'], yp_dns_zones))
            diff = actual_zones - yp_dns_zones_names
            if len(diff) > 0:
                return 'CRIT', 'Missing {} zones in YP.DNS config'.format(', '.join(diff))

            failed_zones = []
            for zone in yp_dns_zones:
                if zone['Name'] not in actual_zones:
                    continue
                if set(zone['YPClusters']) != YP_CLUSTERS:
                    failed_zones.append(zone['Name'])

            if failed_zones:
                logger.error('There are zones with invalid set of clusters. Valid set is {}'.format(YP_CLUSTERS))
                logger.error('List of zones with invalid set of clusters:\n{}'.format('\n'.join(failed_zones)))
                return 'CRIT', '{} zones have invalid set of YP clusters in config: {}'.format(len(failed_zones), format_list(failed_zones))
        except Exception as e:
            return 'CRIT', 'An error occured while checking YP.DNS instance {}:\n{}'.format(instance, e)

    if get_config_errors > 0.5 * len(instances):
        return 'CRIT', 'Failed to get YP.DNS config from {} instances in {}'.format(get_config_errors, nanny_service)

    return 'OK', 'All zones from racktables are in config'


def check_yp_dns_zones_list(args):
    for nanny_service in args.nanny_services:
        logger.info('Check YP.DNS config in service {}'.format(nanny_service))
        status, description = check_zones_list(nanny_service)

        logger.info('Result status: {}'.format(status))
        logger.info('Additional info:\n{}'.format(description))
        juggler_notify(nanny_service, status, description)


def main(argv):
    args = parse_args(argv)

    if args.verbose:
        logger.setLevel(logging.DEBUG)
    else:
        logger.setLevel(logging.INFO)

    try:
        check_yp_dns_zones_list(args)
    except Exception as e:
        logger.exception(e)


if __name__ == '__main__':
    main(sys.argv[1:])
