#!/usr/bin/env python3
# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
import argparse
import os
import itertools

import psycopg2
from email.mime.text import MIMEText
import smtplib
from smtplib import SMTPRecipientsRefused

import sys
from datetime import datetime, timedelta
import time
import requests.packages.urllib3
from nanny_rpc_client import RequestsRpcClient
from nanny_repo import repo_api_stub
from nanny_repo import repo_api_pb2
from yp.client import YpClient, find_token
from yp.common import YpNoSuchObjectError
from yt.wrapper.yson import YsonEntity

requests.packages.urllib3.disable_warnings()


def setup_custom_logger(name):
    formatter = logging.Formatter(fmt='%(asctime)s %(levelname)-8s %(message)s',
                                  datefmt='%Y-%m-%d %H:%M:%S')
    screen_handler = logging.StreamHandler(stream=sys.stdout)
    screen_handler.setFormatter(formatter)
    logger = logging.getLogger(name)
    logger.handlers = []
    logger.setLevel(logging.DEBUG)
    logger.addHandler(screen_handler)
    return logger


LOGGER = setup_custom_logger('binary')


ACCOUNT_MONITORING_START_DATE = datetime(2018, 9, 20)
STARTRACK_QUEUE = None
STARTRACK_SESSION = None
DC_LIST = ["vla", "sas", "man", "iva", "myt", "xdc"]
# DC_LIST = ["myt"]
NANNY_URL = 'http://nanny.yandex-team.ru'
STAFF_URL = 'https://staff-api.yandex-team.ru/v3/'
DEPLOY_RESERVED_USERS = set(["abc:service-scope:2900:8", "robot-vmagent-rtc", "abc:service-scope:2900:16", "robot-drug-deploy"])
STAFF_PERSON_GOOD_EMPLOYEES = {}

MANAGER_STAFF = "arivkin"
MANAGER_CONTACT = "Андрей Ривкин <{staff}@yandex-team.ru>".format(staff=MANAGER_STAFF)
TEST_SERVICES = []

STANDARD_TICKET_MESSAGES = {
    "first": "Коллеги, остается неделя до вывода мощностей из сервиса",
    "second": "Внимание! Завтра мощности будут выведены из-под сервиса, нужно что-то срочно решать",
    "final": "К сожалению, сегодня мощности будут выведены из-под сервиса",
    "two_weeks_left": "Коллеги, остается две недели до вывода мощностей из-под сервиса",
}


class NotificationActions(object):
    def __init__(self, level, timedelta, action):
        self.level = level
        self.offset = timedelta
        self.action = action

    def do(self, can_write, podset, session):
        self.action(podset, can_write, self.level, session)


def create_ticket_and_nanny_notification(service, can_write, notification_level):
    LOGGER.debug("Tracecmd. Want to create_ticket_and_nanny_notification for id={}, ticket={}".format(
        service.podset_id,
        service.ticket))

    if service.ticket is None:
        service.ticket = create_ticket(service, can_write)


def add_comment_and_nanny_notification(service, can_write, text, notification_level):
    if not can_write:
        LOGGER.debug("Tracecmd. Want to add_comment_and_nanny_notification for id={}, ticket={}".format(
            service.podset_id,
            service.ticket))
        return

    notify_add_ticket_comment(service, text)


def make_final_notification(session, service, can_write, text, notification_level):
    if not can_write:
        LOGGER.debug("Tracecmd. Want to make_final_notification for id={}, ticket={}".format(
            service.podset_id,
            service.ticket))
        return

    notify_add_ticket_comment(service, text)
    add_ticket_tag(session, service, "tmp_account_please_delete_me")

    notify_service_email(
        service,
        "YPRES. Please delete service",
        "Please delete service for ticket https://st.yandex-team.ru/{}".format(service.ticket),
        notification_level,
        forced_to="{staff}@yandex-team.ru".format(staff=MANAGER_STAFF))


NOTIFICATION_LEVELS = [
    NotificationActions(1,
                        timedelta(days=7),
                        lambda service, can_write, level, session: send_welcome_email(service, can_write, level)),

    NotificationActions(2,
                        timedelta(days=7),
                        lambda service, can_write, level, session:

                        create_ticket_and_nanny_notification(service, can_write, level) if service.ticket is None
                        else
                        add_comment_and_nanny_notification(service,
                                                           can_write,
                                                           STANDARD_TICKET_MESSAGES["two_weeks_left"],
                                                           level)),

    NotificationActions(3,
                        timedelta(days=7),
                        lambda service, can_write, level, session: add_comment_and_nanny_notification(service,
                                                                                                      can_write,
                                                                                                      STANDARD_TICKET_MESSAGES["first"],
                                                                                                      level)),

    NotificationActions(4,
                        timedelta(days=7),
                        lambda service, can_write, level, session: add_comment_and_nanny_notification(service,
                                                                                                      can_write,
                                                                                                      STANDARD_TICKET_MESSAGES["second"],
                                                                                                      level)),

    NotificationActions(5,
                        timedelta(days=1),
                        lambda service, can_write, level, session: make_final_notification(session,
                                                                                           service,
                                                                                           can_write,
                                                                                           STANDARD_TICKET_MESSAGES["final"],
                                                                                           level)),
]


def get_notification(level):
    return [notification for notification in NOTIFICATION_LEVELS if notification.level == level][0]


def get_nofication_levels():
    return [notification.level for notification in NOTIFICATION_LEVELS]


NANNY_NOTIFICATIONS = {}
for level in get_nofication_levels():
    NANNY_NOTIFICATIONS[level] = []


def find_expected_notification_level(podset_id, pod_creation_time, notification_tuple, current, deadline):
    notification_lifelen = sum([notification.offset for notification in NOTIFICATION_LEVELS], timedelta())

    e_notification_date, e_notification_level = notification_tuple

    pod_creation_time = round_to_day(pod_creation_time)
    deadline = round_to_day(deadline)
    e_notification_date = round_to_day(e_notification_date)

    # deadline was increased. include length of current notification level
    if deadline is None:
        deadline = pod_creation_time + notification_lifelen
    else:
        if deadline < pod_creation_time + notification_lifelen:
            deadline = pod_creation_time + notification_lifelen

    LOGGER.debug(
        "find_expected_notification_level podset_id={} pod_creation_time={} "
        "notification_tuple={} current={} deadline={} notification_lifelen={}".format(
        podset_id,
        pod_creation_time,
        notification_tuple,
        current,
        deadline,
        notification_lifelen))

    current_levels = []
    tm = pod_creation_time
    for (level, offset) in [(notification.level, notification.offset) for notification in NOTIFICATION_LEVELS]:

        if e_notification_level == level:
            tm = e_notification_date
        else:
            tm = tm + offset

        current_levels.append((level, tm))

    proposed_deadline = max(max([border for (level, border) in current_levels]), deadline)
    for (level, border) in current_levels:
        LOGGER.debug("podset_id={} level={} border={}".format(podset_id, level, border))

    LOGGER.debug("proposed deadline={}".format(proposed_deadline))

    e_notification_date, e_notification_level = notification_tuple
    current_levels = []
    tm = proposed_deadline
    for (level, offset) in reversed(
            [(notification.level, notification.offset) for notification in NOTIFICATION_LEVELS]):
        current_levels.append((level, tm))
        tm = tm - offset

    # find deadline without the last notification

    for (level, border) in current_levels:
        LOGGER.debug("podset_id={} level={} border={}".format(podset_id, level, border))

    for (level, border) in current_levels:
        if current >= border:
            if level < e_notification_level:
                level = max(e_notification_level - 1, level)
                return level, proposed_deadline, border

            elif level > e_notification_level:
                level = min(e_notification_level + 1, level)
                preliminary_notification_date = current + timedelta(seconds=1)

                new_level, new_deadline, new_border = find_expected_notification_level(
                    podset_id=podset_id,
                    pod_creation_time=pod_creation_time,
                    notification_tuple=(preliminary_notification_date, level),
                    current=preliminary_notification_date,
                    deadline=proposed_deadline)

                return level, new_deadline, border
            else:
                return level, proposed_deadline, border

    if e_notification_level > 1:
        return e_notification_level - 1, proposed_deadline, pod_creation_time
    else:
        return e_notification_level, proposed_deadline, pod_creation_time


def retry_me(closure, count):
    retries = count

    while retries > 0:
        try:
            retries = retries - 1
            return closure()
        except Exception as err:
            LOGGER.debug(err)
            if retries > 0:
                sys.stderr.write("Retrying, attempt {}\n".format(count - retries))
                time.sleep(1)
                pass
            else:
                ex = sys.exc_info()
                raise ex[0](ex[1]).with_traceback(ex[2])


def check_owner(session, person):
    if person.find(",") >= 0:
        return False

    if person not in STAFF_PERSON_GOOD_EMPLOYEES.keys():
        LOGGER.debug("Verifying person {}".format(person))
        url = '{}persons?login={}&_one=1'.format(STAFF_URL, person)
        response = retry_me(lambda: session.get(url, timeout=10), 10)

        if response.ok:
            employee_info = response.json()

            good_employee = True
            if employee_info["official"]["affiliation"] == "external":
                good_employee = False
            # it's a virtual employee group
            elif len(employee_info["robot_owners"]) > 0:
                good_employee = False
            assert person.find("robot") == -1 or not good_employee

            STAFF_PERSON_GOOD_EMPLOYEES[person] = good_employee
        else:
            STAFF_PERSON_GOOD_EMPLOYEES[person] = False

        LOGGER.debug("Verifying person {} is good {}".format(person, STAFF_PERSON_GOOD_EMPLOYEES[person]))
    return STAFF_PERSON_GOOD_EMPLOYEES[person]


def only_human_beeings(session, owners):
    return [person for person in owners if check_owner(session, person)]


class Group:
    def __init__(self, id, yp_client=None, session=None):
        self.id = id
        self.yp_client = yp_client
        self.session = session

    def resolve(self):
        if self.yp_client is not None:
            return self._resolve_yp(self.id)
        else:
            return self._resolve_staff(self.id)

    STAFF_CACHE = {}

    def _resolve_staff(self, group):
        persons = []
        if group not in Group.STAFF_CACHE.keys():
            if group.isdigit():
                url = '{}groupmembership?group.id={}&_fields=person.login'.format(STAFF_URL, group)
                response = retry_me(lambda: self.session.get(url, timeout=10), 10)
                response.raise_for_status()
                Group.STAFF_CACHE[group] = list(set([item["person"]["login"] for item in response.json()["result"]]))

                if len(Group.STAFF_CACHE[group]) == 0:
                    LOGGER.debug("Group {} has no members".format(group))

                LOGGER.debug("Asking staff for group={} members={}".format(group, Group.STAFF_CACHE[group]))
            else:
                url = '{}groupmembership?group.department.url={}&_fields=person.login'.format(STAFF_URL, group)
                response = retry_me(lambda: self.session.get(url, timeout=10), 10)
                response.raise_for_status()
                Group.STAFF_CACHE[group] = list(set([item["person"]["login"] for item in response.json()["result"]]))
                assert len(Group.STAFF_CACHE[group]) > 0

        persons.extend(Group.STAFF_CACHE[group])

        return list(set(persons))

    def _resolve_yp(self, id):
        LOGGER.debug("Resolving yp group {id}".format(id=id))
        try:
            members = set()
            spec_members = self.yp_client.get_object(
                "group",
                id,
                selectors=["/spec/members"])[0]
            if isinstance(spec_members, YsonEntity):
                return []

            for member in spec_members:
                if member.find(":") >= 0:
                    members = members.union(set(self._resolve_yp(member)))
                else:
                    members.add(member)

            return list(members)
        except YpNoSuchObjectError:
            LOGGER.exception("Group {} is not found".format(id))
            return []


class Service:
    def __init__(self, podset_id, dc, yp_client, yp_xdc_client, session):
        self.podset_id = podset_id
        self.dcs = [dc]
        self.yp_client = yp_client
        self.yp_xdc_client = yp_xdc_client
        self.service_is_empty = False
        self.session = session
        self.account_id = None
        self.podset_power = 0
        self.users = list()
        self.groups = list()
        self.owners = list()
        self.notification_date = None
        self.notification_level = 0
        self.ticket = None
        self.ticket_deadline = None
        self.creation_time = None
        self.service_type = None
        self.stage_id = None
        self.nanny_service_id = None
        self._init()

    def __repr__(self):
        return "[{type}] id={id} notification_level={level}".format(
            type=self.service_type, id=self.podset_id, level=self.notification_level)

    def get_deletion_date(self):
        podset_deadline = self.ticket_deadline
        if podset_deadline is not None:
            return podset_deadline
        else:
            when = max(self.creation_time, ACCOUNT_MONITORING_START_DATE)
            for offset in [notification.offset for notification in NOTIFICATION_LEVELS]:
                when = when + offset
            return when

    def load_from_db(self, notification_level, notification_date, ticket):
        self.notification_level = notification_level
        self.notification_date = notification_date
        self.ticket = ticket

        if self.ticket is not None:
            resp = STARTRACK_SESSION.get("https://st-api.yandex-team.ru/v2/issues/{}".format(self.ticket))
            resp.raise_for_status()
            self.ticket_deadline = datetime.strptime(resp.json()["deadline"], '%Y-%m-%d')

    def is_empty(self):
        return self.service_is_empty or self.podset_power == 0

    def get_account_id(self):
        return self.account_id

    def get_owners(self):
        return self.owners

    def get_service_power(self):
        return self.podset_power

    @staticmethod
    def _get_service_type(labels):
        if labels.get("deploy_engine", "") == "QYP":
            return "QYP"
        elif labels.get("deploy_engine", "") in ["RSC", "MCRSC"]:
            return "DEPLOY"
        elif "nanny_service_id" in labels:
            return "NANNY"
        else:
            return "GENERIC"

    def merge(self, other):
        assert self.podset_id == other.podset_id
        assert self.service_is_empty == other.service_is_empty

        dcs = set(self.dcs)
        for dc in other.dcs:
            dcs.add(dc)
        self.dcs = list(dcs)
        self.podset_power += other.podset_power

        if self.service_is_empty:
            return

        self.users = list(set(list(self.users) + list(other.users)))
        self.groups = list(set(list(self.groups) + list(other.groups)))
        self._build_owners()

    def _build_owners(self):
        owners = only_human_beeings(self.session, self.users)
        if len(owners) > 0:
            self.owners = owners
        else:
            owners = list()
            for group in self.groups:
                owners.extend(group.resolve())
            self.owners = only_human_beeings(self.session, list(set(owners)))

    def _init(self):
        meta = spec = labels = None
        try:
            meta, spec, labels = self.yp_client.get_object(
                "pod_set",
                self.podset_id,
                selectors=["/meta", "/spec", "/labels"])
        except YpNoSuchObjectError as ex:
            self.service_is_empty = True
            LOGGER.exception("Podset {} is not found".format(self.podset_id))
            raise ex

        self.service_type = Service._get_service_type(labels)
        if self.service_type == "QYP":
            self._initialize_qyp(meta, spec, labels)
        elif self.service_type == "DEPLOY" and self.podset_id.find(".") != -1:
            self._initialize_deploy(meta, spec, labels)
        elif self.service_type == "NANNY":
            self._initialize_nanny(meta, spec, labels)
        else:
            self._initialize_generic(meta, spec, labels)

        pods = self.yp_client.select_objects("pod", filter="[/meta/pod_set_id]=\"" + self.podset_id + "\"", selectors=["/meta/id"])
        self.podset_power = len(list(itertools.chain(*pods)))

#        if self.is_empty():
#            return

        self._build_owners()

        self._initialize_common_properties(meta, labels)

    def _initialize_common_properties(self, meta, labels):
        self.creation_time = datetime.utcfromtimestamp(meta["creation_time"] / 1000000)
        self.notification_date = self.creation_time

    def _initialize_qyp(self, meta, spec, labels):
        owners = self.yp_client.get_object(
            "pod",
            self.podset_id,
            selectors=["/annotations/owners"],
        )[0]
        self.users = set(list(owners["logins"]))
        self.groups = [Group(str(group), session=self.session) for group in owners["groups"]]

    def _initialize_deploy(self, meta, spec, labels):
        self.stage_id = self.podset_id.split('.')[0]

        try:
            stage_info = self.yp_xdc_client.get_object("stage", self.stage_id, selectors=["/meta/acl", "/meta/project_id"])
            stage_acl = stage_info[0]
            project_id = stage_info[1]

            acl_writers = [rule["subjects"] for rule in stage_acl
                           if rule["action"] == "allow" and "write" in set(rule["permissions"])
                           and "subjects" in rule]

            project_owner_found = False
            owner_types = ["OWNER", "MAINTAINER", "DEVELOPER"]
            for owner_type in owner_types:
                try:
                    project_owners = self.yp_xdc_client.get_object("group",
                                                                   "deploy:{project_id}.{owner_type}".format(project_id=project_id, owner_type=owner_type),
                                                                   selectors=["/spec/members"])

                    if not all([isinstance(project_owner, YsonEntity) for project_owner in project_owners]):
                        acl_writers.extend(project_owners)
                        project_owner_found = True
                        break

                except YpNoSuchObjectError:
                    self.service_is_empty = True
                    LOGGER.exception("Project {project_id} {owner_type} are not found".format(project_id=project_id, owner_type=owner_type))

            if not project_owner_found:
                LOGGER.exception("Not found any owner for project {project_id}".format(project_id=project_id))

            groups = set()
            users = set()
            for writers in acl_writers:
                if isinstance(writers, YsonEntity):
                    continue

                for writer in writers:
                    if writer not in DEPLOY_RESERVED_USERS:
                        if writer.find(":") > 0:
                            groups.add(writer)
                        else:
                            users.add(writer)

            self.groups = list()
            for group in groups:
                self.groups.append(Group(group, yp_client=self.yp_xdc_client))

            self.users = list(users)
        except YpNoSuchObjectError:
            self.service_is_empty = True
            LOGGER.exception("Stage {} is not found".format(self.stage_id))
            return

        # response = self.yp_xdc_client.get_object_access_allowed_for([{"object_type": "stage",
        #                                                               "object_id": self.stage_id,
        #                                                               "permission": "write"}])
        # self.users = set(response[0]["user_ids"])

    def _initialize_nanny(self, meta, spec, labels):
        LOGGER.debug("Loading nanny info {}".format(self.podset_id))
        self.nanny_service_id = labels["nanny_service_id"]
        response = retry_me(lambda: self.session.get('{}/v2/services/{}'.format(NANNY_URL, self.nanny_service_id), timeout=10), 10)
        if not response.ok:
            self.service_is_empty = True
            return

        service = response.json()
        auth_content = service["auth_attrs"]["content"]

        LOGGER.debug("Loading nanny info {} logins={} groups={}".format(self.podset_id,
                                                                        auth_content["owners"]["logins"],
                                                                        auth_content["owners"]["groups"]))

        self.users = auth_content["owners"]["logins"]

        self.groups = list()
        for group in auth_content["owners"]["groups"]:
            self.groups.append(Group(group, session=self.session))

    def _initialize_generic(self, meta, spec, labels):
        response = self.yp_client.get_object_access_allowed_for([{"object_type": "pod_set",
                                                                  "object_id": self.podset_id,
                                                                  "permission": "write"}])
        self.users = set(response[0]["user_ids"])

    def generate_service_name(self):
        if self.service_type == "NANNY":
            return self.nanny_service_id
        elif self.service_type == "DEPLOY":
            return self.stage_id
        else:
            return self.podset_id

    def generate_service_url(self):
        if self.service_type == "NANNY":
            return "<a href='https://nanny.yandex-team.ru/ui/#/services/catalog/%(service)s/'>[Nanny] %(service)s</a>" \
                   % {"service": self.generate_service_name()}
        elif self.service_type == "QYP":
            return "<a href='https://qyp.yandex-team.ru/vm/{}/{}'>[QYP] {}</a>" \
                .format(self.dcs[0], self.podset_id, self.generate_service_name())
        elif self.service_type == "DEPLOY":
            return "<a href='https://deploy.yandex-team.ru/project/{}'>[Deploy] {}</a>".format(self.stage_id, self.stage_id)
        else:
            return self.podset_id

    def generate_service_raw_url(self):
        if self.service_type == "NANNY":
            return "((https://nanny.yandex-team.ru/ui/#/services/catalog/%(service)s/ [Nanny] %(service)s))" \
                   % {"service": self.generate_service_name()}
        elif self.service_type == "QYP":
            return "((https://qyp.yandex-team.ru/vm/{}/{} [QYP] {}))" \
                .format(self.dcs[0], self.podset_id, self.generate_service_name())
        elif self.service_type == "DEPLOY":
            return "((https://deploy.yandex-team.ru/project/{stage_id} [DEPLOY] {name}))" \
                .format(stage_id=self.stage_id, name=self.generate_service_name())
        else:
            return self.podset_id

    def is_nanny(self):
        return self.service_type == "NANNY"

    def is_qyp(self):
        return self.service_type == "QYP"

    def is_deploy(self):
        return self.service_type == "DEPLOY"

    def get_expected_notification_level(self):
        current_notification_level = self.notification_level
        current_notification_date = self.notification_date

        # уже сообщали в письмах пользователям
        if self.creation_time < ACCOUNT_MONITORING_START_DATE:
            current_notification_level = max(1, current_notification_level)
            current_notification_date = max(ACCOUNT_MONITORING_START_DATE, current_notification_date)

        expected_notification_level, proposed_deadline, border_time = \
            find_expected_notification_level(
                self.podset_id,
                max(self.creation_time, ACCOUNT_MONITORING_START_DATE - timedelta(days=7)),
                (current_notification_date, current_notification_level),
                datetime.now(),
                self.ticket_deadline
            )

        LOGGER.debug("Podset {}, "
                     "current_notification_date={}, "
                     "current_notification_level={}, "
                     "podset_creation_time={}, "
                     "expected_notification_level={}, "
                     "proposed_deadline={}, "
                     "podset deadline={} "
                     "border_time={}".
                     format(self.podset_id,
                            current_notification_date,
                            current_notification_level,
                            self.creation_time,
                            expected_notification_level,
                            proposed_deadline,
                            self.ticket_deadline,
                            border_time))

        return current_notification_level, expected_notification_level, proposed_deadline


def standard_message_format(service):
    message_format = {"service_link": service.generate_service_url(),
                      "service_name": service.generate_service_name(),
                      "service_wiki_link": service.generate_service_raw_url(),
                      "podset_creation_time": service.creation_time.strftime("%d.%m.%Y"),
                      "delete_date": service.get_deletion_date().strftime("%d.%m.%Y"),
                      "owners": ",".join([owner + "@" for owner in service.get_owners()]),
                      "manager_staff": MANAGER_STAFF}

    if service.ticket is not None:
        message_format["ticket"] = service.ticket

    return message_format


def send_welcome_email(service, can_write, notification_level):
    if not can_write:
        LOGGER.debug("Tracecmd. Want to send_welcome_email for id={}, ticket={}".format(
            service.podset_id,
            service.ticket))
        return

    subject = "Сервис %(service_name)s находится в YP во временной квоте неделю" % standard_message_format(service)
    body = '''<p>Привет!</p>
    <p>Ваш сервис %(service_link)s находится в YP во временной квоте уже неделю, c %(podset_creation_time)s.
     Надеемся, вам нравится работать под управлением YP. Пришло время задумываться: или переезжать в постоянную квоту,
     или вернуть мощности в облако и поделиться ими с другими проектами.&nbsp;</p>
    <ul>
    <li><a href="https://wiki.yandex-team.ru/yp/Vremennye-kvoty/">Подробнее про временные квоты</a>&nbsp;</li>
    <li><a href="https://wiki.yandex-team.ru/yp/quotas/">Как заказать постоянную квоту</a></li>
    </ul>
    <p>Через неделю, за две недели до крайнего срока, мы заведем тикет про переезд.&nbsp;</p>
    <p>Это письмо мы отправили всем владельцам сервиса: %(owners)s.&nbsp;</p>
    ''' % standard_message_format(service)

    notify_service_email(service, subject, body, notification_level)


def notify_service_email(service, subject, body, notification_level, forced_to=None):
    s = smtplib.SMTP('yabacks.yandex.ru:25')

    msg = MIMEText(body, "html", "utf-8")

    msg['Subject'] = subject
    msg['From'] = MANAGER_CONTACT

    if forced_to is not None:
        recipients = MANAGER_CONTACT
    else:
        if len(service.get_owners()) > 0:
            recipients = ",".join("{}@yandex-team.ru".format(owner) for owner in service.get_owners())
        else:
            recipients = MANAGER_CONTACT

    msg['Cc'] = MANAGER_CONTACT
    msg['To'] = recipients

    LOGGER.info("Sending the message to {}".format(msg['To']))

    retries_count = 10
    while True:
        retries_count -= 1
        try:
            LOGGER.debug("Before e-mail to user {} for service {} , notification_level={}".format(
                recipients,
                service.podset_id,
                notification_level))

            s.sendmail("{staff}@yandex-team.ru".format(staff=MANAGER_STAFF), recipients, msg.as_string())
            LOGGER.info("e-mail to user {} for service {} sent, notification_level={}".format(
                recipients,
                service.podset_id,
                notification_level))

            break
        except SMTPRecipientsRefused as e:
            LOGGER.debug(e)

            if retries_count <= 0:
                raise

            smtp_code = e.smtp_code
            LOGGER.debug("SMTP cofe={}".format(smtp_code))
            if smtp_code in [432, 450]:
                time.sleep(30)

    s.quit()


def get_notification_text(notification_level):
    if notification_level == 1:
        return ("Сервис находится в YP во временной квоте неделю из разрешенного месяца. "
                "<a href='https://wiki.yandex-team.ru/yp/Vremennye-kvoty/'>Подробнее</a>", 0)  # INFO
    elif notification_level == 2:
        return ("Внимание! Сервис находится в YP во временной квоте две недели из разрешенного месяца. "
                "Через две недели мощности будут отключены. "
                "<a href='https://wiki.yandex-team.ru/yp/Vremennye-kvoty/'>Подробнее</a>", 1)  # WARNING
    elif notification_level == 3:
        return ("Внимание! Сервис находится в YP во временной квоте три недели из месяца, "
                "осталась всего неделя и мощности будут отключены. "
                "<a href='https://wiki.yandex-team.ru/yp/Vremennye-kvoty/'>Подробнее</a>", 2)  # ERROR
    elif notification_level >= 4:
        return ("Внимание! Сервис находится в YP во временной квоте слишком долго, "
                "мощности будут отключены в любой момент. "
                "<a href='https://wiki.yandex-team.ru/yp/Vremennye-kvoty/'>Подробнее</a>", 2)  # ERROR


def notify_services_in_nanny(nanny_services_levels):
    LOGGER.debug("pushing new notifications to nanny {}".format(nanny_services_levels))
    repo_stub = buildNannyRepoClient()

    r = retry_me(lambda: repo_stub.list_notifications(repo_api_pb2.ListNotificationsRequest()), 10)

    for (level, services) in nanny_services_levels.items():
        notification = [notification for notification in r.notifications
                        if notification.meta.id == "notification_yp_tmp_account_level{}".format(level)][0]

        req = repo_api_pb2.UpdateNotificationRequest()
        req.meta.CopyFrom(notification.meta)
        req.spec.CopyFrom(notification.spec)

        del req.spec.per_service.services[:]
        for service in services:
            srv = req.spec.per_service.services.add()
            srv.id = service

        srv = req.spec.per_service.services.add()
        srv.id = "0e9712d2-b841-11e8-96f8-529269fb1459"

        LOGGER.debug("Setting nanny notification level={} for services={}".format(level, services))
        retry_me(lambda: repo_stub.update_notification(req), 10)


def create_remove_notification_request(notification_id):
    LOGGER.debug("removing notification {}".format(notification_id))
    return repo_api_pb2.RemoveNotificationRequest(notification_id=notification_id)


def create_notification_request(message, notification_id, level):
    LOGGER.debug("adding notification type={}".format(notification_id))
    req = repo_api_pb2.CreateNotificationRequest()
    req.meta.id = notification_id
    req.meta.version = '1'
    req.meta.generation = 1
    req.meta.creation_time.GetCurrentTime()
    req.meta.author = "robot-yp-account"
    req.meta.last_modification_time.GetCurrentTime()

    # PER_SERVICE
    req.spec.type = 0
    req.spec.per_service.content = message
    req.spec.per_service.severity_level = level
    srv = req.spec.per_service.services.add()
    srv.id = "0e9712d2-b841-11e8-96f8-529269fb1459"

    return req


def buildNannyRepoClient():
    nanny_repo_client = retry_me(lambda: RequestsRpcClient('http://nanny.yandex-team.ru/api/repo',
                                                           request_timeout=10,
                                                           oauth_token=os.environ["NANNY_OAUTH"]),
                                 10)

    repo_stub = repo_api_stub.RepoServiceStub(nanny_repo_client)
    return repo_stub


def close_ticket(ticket):
    resp = STARTRACK_SESSION.get("https://st-api.yandex-team.ru/v2/issues/{}".format(ticket))
    resp.raise_for_status()

    if resp.json()["status"]["key"] != "closed":
        add_ticket_comment(ticket, "Сервис больше не использует временную квоту")

        resp = STARTRACK_SESSION.post("https://st-api.yandex-team.ru/v2/issues/{}/transitions/close/_execute".
                                      format(ticket),
                                      json={"resolution": "fixed"})
        resp.raise_for_status()


def clear_obsolete_services_from_database(conn, podsets, table):
    with conn.cursor() as cmd:
        cmd.execute("select id, ticket from {} where not(id=any(%s))".format(table), (list(podsets),))
        rows = cmd.fetchall()
        if len(rows) > 0:
            for row in rows:
                podset_id = row[0]
                ticket = row[1]
                LOGGER.info("Closing ticket {} for podset {}".format(ticket, podset_id))

                if ticket is not None:
                    try:
                        close_ticket(ticket)
                    except requests.exceptions.HTTPError as err:
                        LOGGER.debug("Error while closing ticket {} for podset {} err={}".format(ticket, podset_id, err))

        cmd.execute(
            "delete from {} where not(id=any(%s))".format(table), (list(podsets),))
        conn.commit()


def get_cluster_superusers(client):
    members = client.get_object("group", "superusers", selectors=["/spec/members"])[0]
    if not members:
        # Handle YsonEntity.
        return []
    else:
        return list(members)


def get_podsets_in_tmpquota_for_default(client):
    pod_sets = client.select_objects(
        "pod_set",
        filter="[/spec/account_id]='tmp' and [/spec/node_segment_id]='default'",
        selectors=["/meta/id"])
    return list(itertools.chain(*pod_sets))


def notify_add_ticket_comment(service, comment_template):
    add_ticket_comment(service.ticket, comment_template % standard_message_format(service))


def add_ticket_tag(session, service, tag):
    response = session.get("https://st-api.yandex-team.ru/v2/issues/{}".format(service.ticket))
    response.raise_for_status()
    tags = response.json().get("tags", [])
    tags.append(tag)
    tags = list(set(tags))
    response = session.patch("https://st-api.yandex-team.ru/v2/issues/{}".format(service.ticket), json={"tags": tags})
    response.raise_for_status()


def create_ticket(service, can_write):
    body = '''Сервис %(service_wiki_link)s находится во временой квоте YP две недели из четырех возможных, c %(podset_creation_time)s.
Нужно или перевести сервис в проектную квоту, или удалить его, в противном случае мощности будут из-под него выведены %(delete_date)s.
Рекомендуется этим начать заниматься заранее, так как из-за взаимозачета ресурсов процесс подтверждения заказанных мощностей может занимать много времени.
* ((https://wiki.yandex-team.ru/yp/tmp Подробнее про временные квоты))
* ((https://wiki.yandex-team.ru/yp/quotas Как заказать постоянную квоту))
Через неделю мы еще раз напомним, если сервис будет все еще не в проектной квоте.
P.S.: Пожалуйста, не закрывайте тикеты в этой очереди в пользу тикетов в вашей.
Наша автоматика не может найти ваши тикеты в новых очередях, чтобы предупредить об истечении срока.
---------------------------------------
Тикет создан от имени робота, призывайте явно %(manager_staff)s@, если у вас есть любые вопросы.
''' % standard_message_format(service)

    assignee = None
    if len(service.get_owners()) > 0:
        assignee = service.get_owners()[0]
    else:
        assignee = MANAGER_STAFF

    LOGGER.debug("Podset id={} owners={} assignee={}".format(service.podset_id,
                                                             service.get_owners(),
                                                             assignee))

    ticket_data = {"queue": STARTRACK_QUEUE,
                   "summary": "Перевести сервис {} в проектную квоту".format(service.generate_service_name()),
                   "type": 2,
                   "assignee": assignee,
                   "followers": service.get_owners(),
                   "description": body,
                   "deadline": service.get_deletion_date().strftime("%Y-%m-%d"),
                   "tags": ["tmp_account_monitoring"]
                   }

    if not can_write:
        LOGGER.debug("Tracecmd. Want to create_ticket_and_nanny_notification for id={}, ticket={}".format(
            service.podset_id,
            service.ticket))
        return

    LOGGER.info("Creating the ticket for service={} data={} ".format(service.podset_id, ticket_data))

    response = STARTRACK_SESSION.post("https://st-api.yandex-team.ru/v2/issues", json=ticket_data)

    if not response.ok:
        LOGGER.debug("Response status={} headers={} text={}".format(
            response.status_code, response.headers.encode("utf-8"), response.text.encode("utf-8")))

    response.raise_for_status()
    return response.json()["key"]


def add_ticket_comment(ticket, comment):
    STARTRACK_SESSION.post(
        "https://st-api.yandex-team.ru/v2/issues/{}/comments".format(ticket),
        json={"text": comment})


def round_to_day(dt):
    if dt is None:
        return None
    else:
        return datetime(dt.year, dt.month, dt.day)


def update_ticket_deadline(session, service, deadline, dry_run):
    LOGGER.debug("Update ticket {} deadline {} to {}".format(service.podset_id, service.ticket, deadline))

    if not dry_run:
        response = session.patch("https://st-api.yandex-team.ru/v2/issues/{}".format(service.ticket),
                                 json={"deadline": deadline.strftime("%Y-%m-%d")})
        response.raise_for_status()


def send_iam_ok():
    LOGGER.info("Sending I'm ok to juggler")
    result = requests.post("http://juggler-push.search.yandex.net/events",
                           json={
                               "source": "yp.tmp_account_monitoring",
                               "events": [
                                   {
                                       "description": "sync cycle ok",
                                       "host": "yp.tmp_account_monitoring",
                                       "instance": "",
                                       "service": "yp.tmp_account_monitoring",
                                       "status": "OK"
                                   }
                               ]
                           }, timeout=10)
    result.raise_for_status()


def main():
    parser = argparse.ArgumentParser('')
    parser.add_argument('--drop-db', action='store_true')
    parser.add_argument('--drop-notifications', action='store_true')

    parser.add_argument('--force-test-email', action='store_true')
    parser.add_argument('--dry-run', action='store_true')
    parser.add_argument('--st-queue', default="TEST", required=True)
    parser.add_argument('--only-test-services', action='store_true')
    parser.add_argument('--database', required=True)
    parser.add_argument('--table', required=True)

    args = parser.parse_args()

    active_pgaas_host = None
    for pgaas_host in os.environ['PGAAS_HOST'].split(","):
        pgaas_host = pgaas_host.strip()
        try:
            conn = psycopg2.connect(database=args.database,
                                    host=pgaas_host, port=os.environ['PGAAS_PORT'],
                                    user=os.environ['PGAAS_USER'],
                                    password=os.environ['PGAAS_PASSWORD'],
                                    sslmode="require")

            with conn.cursor() as cmd:
                cmd.execute("SELECT pg_is_in_recovery()")
                is_readonly_db = cmd.fetchall()[0][0]
                LOGGER.debug("host={}, is_read_only={}".format(pgaas_host, is_readonly_db))
                if not is_readonly_db:
                    active_pgaas_host = pgaas_host
                    LOGGER.debug("host={} is writable, using it".format(active_pgaas_host))
                    break
        except Exception:
            LOGGER.exception("Host {} is not responding, continue".format(pgaas_host))

    if active_pgaas_host is None:
        LOGGER.fatal("No writable pgaas host found, exiting...")
        raise("No writable pgaas host found")

    pgaas_port = os.environ['PGAAS_PORT']
    LOGGER.debug("Connecting to host={}:{}".format(active_pgaas_host, pgaas_port))
    conn = psycopg2.connect(database=args.database,
                            host=active_pgaas_host, port=pgaas_port,
                            user=os.environ['PGAAS_USER'],
                            password=os.environ['PGAAS_PASSWORD'],
                            sslmode="require")

    repo_stub = buildNannyRepoClient()

    global STARTRACK_QUEUE
    STARTRACK_QUEUE = args.st_queue

    if args.drop_db:
        with conn.cursor() as cmd:
            cmd.execute("DROP TABLE IF EXISTS {} CASCADE;".format(args.table))

            conn.commit()

    with conn.cursor() as cmd:
        cmd.execute("""SELECT EXISTS (
                           SELECT 1
                           FROM   information_schema.tables
                           WHERE  table_name = '{}');""".format(args.table))
        rows = cmd.fetchall()
        if not rows[0][0]:
            cmd.execute("create table {}("
                        "id TEXT not null  PRIMARY key, "
                        "notification_level int not null, "
                        "notification_date timestamp not null, "
                        "ticket text null)".format(args.table))
            conn.commit()

    if args.drop_notifications:
        for level in get_nofication_levels():
            req = create_remove_notification_request("notification_yp_tmp_account_level{}".format(level))
            retry_me(lambda: repo_stub.remove_notification(req), 10)

    notifications = []
    r = retry_me(lambda: repo_stub.list_notifications(repo_api_pb2.ListNotificationsRequest()), 10)
    for level in get_nofication_levels():
        notification_id = "notification_yp_tmp_account_level{}".format(level)

        notifications.extend([notification for notification in r.notifications if
                              notification.meta.id == notification_id])

    if len(notifications) == 0:
        for level in get_nofication_levels():
            notification_text, notification_level = get_notification_text(level)
            req = create_notification_request(notification_text,
                                              "notification_yp_tmp_account_level{}".format(level),
                                              notification_level)
            retry_me(lambda: repo_stub.create_notification(req), 10)

    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(os.environ["NANNY_OAUTH"])

    global STARTRACK_SESSION
    STARTRACK_SESSION = requests.Session()
    STARTRACK_SESSION.headers['Authorization'] = 'OAuth {}'.format(os.environ["ST_TOKEN"])

    services = dict()

    xdc_yp_client = YpClient(address="xdc.yp.yandex.net:8090", config={"token": find_token()})
    for cluster in DC_LIST:
        LOGGER.debug("Processing {}".format(cluster))
        yp_client = YpClient(address="{}.yp.yandex.net:8090".format(cluster), config={"token": find_token()})
        superusers = get_cluster_superusers(yp_client)
        LOGGER.debug("Cluster {} superusers is {}".format(cluster, superusers))

        podsets = get_podsets_in_tmpquota_for_default(yp_client)
        if args.only_test_services:
            podsets = TEST_SERVICES

        for podset_id in podsets:
            try:
                service = Service(podset_id, cluster, yp_client, xdc_yp_client, session)
                if podset_id not in services:
                    services[podset_id] = service
                else:
                    services[podset_id].merge(service)
            except YpNoSuchObjectError:
                LOGGER.exception("Service {podset_id} was not created because pod_set was not found".format(podset_id=podset_id))

    with conn.cursor() as cmd:
        cmd.execute("select id, notification_level,notification_date, ticket from {} "
                    "where id=ANY(%s)".format(args.table), (list(services.keys()),))

        rows = cmd.fetchall()

        for row in rows:
            db_podset_id = row[0]
            if db_podset_id in services:
                service = services[db_podset_id]
                service.load_from_db(notification_level=row[1], notification_date=row[2], ticket=row[3])

    LOGGER.debug(services)
    if args.only_test_services:
        for test_service in TEST_SERVICES:
            service = services[test_service]
            assert not service.is_empty()
            service.creation_time = ACCOUNT_MONITORING_START_DATE

        assert set(TEST_SERVICES) == set(services.keys())

    LOGGER.debug(services)

    def make_notification(service, notification_level, conn, send_notifications, args, session):
        LOGGER.debug("Podset {}, escalating notification_level to {}".format(service.podset_id, notification_level))

        can_write = not args.dry_run

        with conn.cursor() as cmd:
            LOGGER.debug("Want to update podset{} and {} notifications".format(
                service.podset_id,
                "send" if send_notifications else "skip"))

            if send_notifications:
                LOGGER.debug("Want to send notifications")
                get_notification(notification_level).do(can_write, service, session)

            LOGGER.debug("Updating the database")
            cmd.execute("select 1 from {} where id=%s".format(args.table), (service.podset_id,))
            if cmd.rowcount > 0:
                LOGGER.debug("Want update db [write={}] to {}".format(
                    can_write,
                    cmd.mogrify(
                        "update {} set notification_level=%s,notification_date=%s, ticket=%s where id=%s;".format(
                            args.table),
                        (notification_level, datetime.now(), service.ticket, service.podset_id,))))

                if can_write:
                    cmd.execute(
                        "update {} set notification_level=%s,notification_date=%s, ticket=%s where id=%s;".format(
                            args.table),
                        (notification_level, datetime.now(), service.ticket, service.podset_id,))

            else:
                LOGGER.debug("Want update db [write={}] to {}".format(
                    can_write,
                    cmd.mogrify(
                        "insert into {} (id,notification_level,notification_date,ticket) values (%s,%s,%s,%s)".format(
                            args.table),
                        (service.podset_id, notification_level, datetime.now(), service.ticket,))))

                if can_write:
                    cmd.execute(
                        "insert into {} (id,notification_level,notification_date,ticket) "
                        "values (%s,%s,%s,%s)".format(args.table),
                        (service.podset_id, notification_level, datetime.now(), service.ticket,))

            service.notification_level = notification_level
            service.notification_date = datetime.now()

            conn.commit()

    total_changes = 0
    for service_id, service in services.items():
        if service.is_empty():
            continue

        current_notification_level, expected_notification_level, proposed_deadline = \
            service.get_expected_notification_level()

        if expected_notification_level != current_notification_level:
            total_changes += 1
            notification_level = expected_notification_level

            podset_deadline = service.ticket_deadline
            service.ticket_deadline = proposed_deadline
            make_notification(service, notification_level, conn, notification_level > current_notification_level,
                              args, session)

            if proposed_deadline is not None and podset_deadline is not None and proposed_deadline > podset_deadline:
                update_ticket_deadline(session, service, proposed_deadline, args.dry_run)

    LOGGER.debug("Made {} changes".format(total_changes))

    for service in services.values():
        if service.is_nanny() and service.notification_level is not None and service.notification_level != 0:
            NANNY_NOTIFICATIONS[service.notification_level].append(service.nanny_service_id)

    if not args.dry_run:
        notify_services_in_nanny(NANNY_NOTIFICATIONS)

    if not args.dry_run:
        clear_obsolete_services_from_database(conn, services.keys(), args.table)

    if not args.dry_run:
        send_iam_ok()


if __name__ == '__main__':
    main()
