#include "printer.h"

#include <infra/yp_dns/libs/config/protos/config.pb.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/protos/config.pb.h>
#include <infra/libs/updatable_proto_config/holder.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/proto_config/config.h>
#include <library/cpp/proto_config/load.h>

namespace NYP::DNS {

struct TOptions {
    bool Original = false;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    TOptions res;

    NLastGetopt::TOpts opts;
    opts.AddLongOption('h', "help", "print usage").NoArgument().Handler1([&](const NLastGetopt::TOptsParser* p) {
        p->PrintUsage(Cout);
        Cout << Endl;
        exit(0);
    });
    opts.SetFreeArgsNum(0);

    opts
        .AddLongOption('o', "original", "Print original config (patch is not applied)")
        .Optional()
        .NoArgument()
        .StoreTrue(&res.Original);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return res;
}

NInfra::TLoggerConfig GetFakeLoggerConfig() {
    NInfra::TLoggerConfig loggerConfig;
    loggerConfig.SetBackend(NInfra::TLoggerConfig_ELogBackend_FILE);
    loggerConfig.SetPath("/dev/null");

    return loggerConfig;
}

void Print(const TConfig& config) {
    NJson::TJsonValue configJson;
    TStringStream str{NProtobufJson::Proto2Json(config)};
    Cout << NJson::WriteJson(NJson::ReadJsonTree(&str, /* allowComments */ false, /* throwOnError */ true), /* formatOutput */ true) << Endl;
}

int PrintConfig(int argc, const char* argv[]) {
    const TOptions options = ParseOptions(argc, argv);

    TConfig config;
    NProtoConfig::LoadConfigFromResource("/proto_config.json", config);


    if (options.Original) {
        Print(config);
    } else {
        NUpdatableProtoConfig::TConfigHolder<TConfig> holder(config, config.GetWatchPatchConfig(), GetFakeLoggerConfig());
        Print(*holder.Config());
    }

    return EXIT_SUCCESS;
}

} // namespace NYP::DNS
