#include "request_logger.h"
#include <util/string/cast.h>

template <>
TString ToString(const DNSName& name) {
    return !name.empty() ? TString{name.toStringNoDot()} : TString{};
}

namespace NYP::DNS {

TBackendEventsLogger::TBackendEventsLogger(NInfra::TLogger& logger)
    : LogFrame_(logger.SpawnFrame())
{
}

void TBackendEventsLogger::LogBackendRequestData(const QType& qtype, const DNSName& domain) const {
    LogFrame_->LogEvent(NEventlog::TBackendRequestData(ToString(qtype.getName()), ToString(domain.toLogString())));
}

void TBackendEventsLogger::LogPacketInfo(DNSPacket* packet) const {
    if (!packet) {
        return;
    }

    NEventlog::TPacketInfo packetInfo;
    packetInfo.SetId(packet->d.id);
    packetInfo.SetTCP(packet->d_tcp);
    packetInfo.SetResourceClass(packet->qclass);
    packetInfo.SetResourceType(ToString(packet->qtype.getName()));
    packetInfo.SetDomain(ToString(packet->qdomain.toLogString()));
    packetInfo.SetRemoteHost(ToString(packet->getRemote().toString()));
    packetInfo.SetRemotePort(packet->getRemote().getPort());
    if (ComboAddress* address = packet->d_anyLocal.get_ptr()) {
        packetInfo.SetDestinationAddress(ToString(address->toStringWithPort()));
    }
    packetInfo.SetLifeTimeMcs(packet->d_dt.udiffNoReset());

    LogFrame_->LogEvent(packetInfo);
}

void TBackendEventsLogger::LogLookupResult(ui32 recordsNumber) const {
    LogFrame_->LogEvent(NEventlog::TLookupResult(recordsNumber));
}

void TBackendEventsLogger::LogListCluster(const TString& cluster, ui64 timestamp) const {
    LogFrame_->LogEvent(NEventlog::TListCluster(cluster, timestamp));
}

void TBackendEventsLogger::LogListResult(const TString& zone, ui32 recordsNumber, bool success, const TString& message) const {
    LogFrame_->LogEvent(NEventlog::TListResult(zone, recordsNumber, success, message));
}

void TBackendEventsLogger::LogListResult(const DNSName& zone, ui32 recordsNumber, bool success, const TString& message) const {
    LogListResult(ToString(zone), recordsNumber, success, message);
}

void TBackendEventsLogger::LogUnallowedResourceType() const {
    LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TUnallowedResourceType());
}

void TBackendEventsLogger::LogUnknownZone(const DNSName& domain) const {
    LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TUnknownZone(ToString(domain)));
}

void TBackendEventsLogger::LogZoneInfo(const NYpDns::TZoneConfig& zone) const {
    LogFrame_->LogEvent(NEventlog::TZoneInfo(zone.GetName(), zone.GetIsAuthoritative()));
}

} // namespace NYP::DNS
