#include "service.h"

#include <infra/contrib/pdns/power_dns/arguments.hh>
#include <infra/contrib/pdns/power_dns/auth-packetcache.hh>
#include <infra/contrib/pdns/power_dns/auth-querycache.hh>
#include <infra/contrib/pdns/power_dns/dns.hh>
#include <infra/contrib/pdns/power_dns/dnspacket.hh>
#include <infra/contrib/pdns/power_dns/logger.hh>
#include <infra/contrib/pdns/power_dns/qtype.hh>
#include <infra/contrib/pdns/power_dns/statbag.hh>

#include <infra/libs/logger/self_closing_logger.h>
#include <infra/libs/sensors/sensor.h>
#include <infra/libs/sensors/sensor_registry.h>
#include <infra/libs/service_iface/request.h>
#include <infra/libs/service_iface/str_iface.h>
#include <infra/libs/yp_dns/record_set/record.h>
#include <infra/libs/yp_dns/record_set/record_set.h>
#include <infra/libs/yp_dns/replication/iterate.h>
#include <infra/libs/yp_dns/replication/merge.h>
#include <infra/libs/yp_dns/zone/response_policy.h>

#include <yp/cpp/yp/token.h>
#include <yp/yp_proto/yp/client/api/proto/data_model.pb.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/proto_config/config.h>
#include <library/cpp/proto_config/load.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/async.h>

#include <util/folder/path.h>
#include <util/random/entropy.h>
#include <util/random/shuffle.h>
#include <util/str_stl.h>
#include <util/system/env.h>
#include <util/system/hostname.h>
#include <util/system/mlock.h>

extern StatBag S;
extern AuthPacketCache PC;
extern AuthQueryCache QC;

size_t THash<DNSName>::operator()(const DNSName& name) const {
    return name.hash();
}

namespace NYP::DNS {

namespace NSensors {

constexpr TStringBuf NAMESPACE = "pdns";
constexpr TStringBuf YP_DNS_BACKEND_NAMESPACE = "backend.YP_DNS";

// Labels
constexpr TStringBuf PACKET_RECORD_TYPE = "packet_record_type";
constexpr TStringBuf RECORD_TYPE = "record_type";
constexpr TStringBuf ZONE = "zone";

constexpr TStringBuf UNKNOWN_ZONE = "<unknown>";

// Sensors
constexpr TStringBuf REQUESTS = "requests";
constexpr TStringBuf EMPTY_RESPONSES = "empty_responses";
constexpr TStringBuf UNALLOWED_RECORD_TYPE = "unallowed_record_type";
constexpr TStringBuf EMPTY_CLUSTERS_LIST = "empty_clusters_list";
constexpr TStringBuf RECORDS_NUMBER = "records_number";

constexpr TStringBuf PING_RESPONSE_TIME = "ping_response_time";
constexpr TStringBuf SHUTDOWN_RESPONSE_TIME = "shutdown_response_time";
constexpr TStringBuf SENSORS_RESPONSE_TIME = "sensors_response_time";
constexpr TStringBuf REOPEN_LOG_RESPONSE_TIME = "reopen_log_response_time";
constexpr std::initializer_list<std::tuple<TStringBuf, double, double>> HISTOGRAMS_INIT_PARAMETERS_YP_DNS = {
    {PING_RESPONSE_TIME, 1.5, 10.0},
    {SHUTDOWN_RESPONSE_TIME, 1.5, 10.0},
    {SENSORS_RESPONSE_TIME, 1.5, 10.0},
    {REOPEN_LOG_RESPONSE_TIME, 1.5, 10.0},
};

constexpr TStringBuf GET_SOA_RESPONSE_TIME = "get_soa_response_time";
constexpr TStringBuf LOOKUP_RESPONSE_TIME = "lookup_response_time";
constexpr TStringBuf LIST_ZONE_RESPONSE_TIME = "list_zone_response_time";
constexpr std::initializer_list<std::tuple<TStringBuf, double, double>> HISTOGRAMS_INIT_PARAMETERS_YP_DNS_BACKEND = {
    {GET_SOA_RESPONSE_TIME, 1.5, 10.0},
    {LOOKUP_RESPONSE_TIME, 1.5, 10.0},
};

} // namespace NSensors

namespace {

constexpr ui64 STORAGE_FORMAT_VERSION = 8;

const std::array<QType, 9> ALLOWED_TYPES = {{
    QType(QType::ANY),
    QType(QType::A),
    QType(QType::NS),
    QType(QType::PTR),
    QType(QType::AAAA),
    QType(QType::SRV),
    QType(QType::CNAME),
    QType(QType::TXT),
    QType(QType::SOA),
}};

constexpr std::array<TStringBuf, 3> KNOWN_SRV_SERVICES = {{
    TStringBuf("_host_"),
    TStringBuf("_ip_"),
    TStringBuf("_id_"),
}};

void PrettyPrintConfig(const TConfig& config) {
    NJson::TJsonValue configJson;
    TStringStream str{NProtobufJson::Proto2Json(config)};
    Cout << NJson::WriteJson(NJson::ReadJsonTree(&str, /* allowComments */ false, /* throwOnError */ true), /* formatOutput */ true) << Endl;
}

ui32 MakeSerialFromTimestamp(ui64 timestamp) {
    return timestamp >> 30u;
}

DomainInfo CreateZoneInfo(const NYpDns::TZoneConfig& zoneConfig, ui32 zoneId) {
    DomainInfo result;
    result.zone = DNSName(zoneConfig.GetName());
    result.id = zoneId;
    return result;
}

QType GetQueryType(NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType type) {
    using namespace NClient::NApi::NProto;
    switch (type) {
    case TDnsRecordSetSpec::TResourceRecord::A:
        return QType(QType::A);
    case TDnsRecordSetSpec::TResourceRecord::NS:
        return QType(QType::NS);
    case TDnsRecordSetSpec::TResourceRecord::CNAME:
        return QType(QType::CNAME);
    case TDnsRecordSetSpec::TResourceRecord::SOA:
        return QType(QType::SOA);
    case TDnsRecordSetSpec::TResourceRecord::PTR:
        return QType(QType::PTR);
    case TDnsRecordSetSpec::TResourceRecord::MX:
        return QType(QType::MX);
    case TDnsRecordSetSpec::TResourceRecord::TXT:
        return QType(QType::TXT);
    case TDnsRecordSetSpec::TResourceRecord::AAAA:
        return QType(QType::AAAA);
    case TDnsRecordSetSpec::TResourceRecord::SRV:
        return QType(QType::SRV);
    case TDnsRecordSetSpec::TResourceRecord::RRSIG:
        return QType(QType::RRSIG);
    case TDnsRecordSetSpec::TResourceRecord::NSEC:
        return QType(QType::NSEC);
    case TDnsRecordSetSpec::TResourceRecord::DNSKEY:
        return QType(QType::DNSKEY);
    case TDnsRecordSetSpec::TResourceRecord::CAA:
        return QType(QType::CAA);
    }
}

bool IsSrvServiceSpecified(const DNSName& domain) {
    TString service;
    try {
        service = domain.getRawLabel(0);
    } catch (...) {
        return false;
    }
    return IsIn(KNOWN_SRV_SERVICES, service);
}

DNSResourceRecord CreateNSRecord(const DNSName& domain, const TString& nameserver, const ui32 ttl) {
    DNSResourceRecord record;
    record.qname = domain.makeLowerCase();
    record.qtype = QType::NS;
    record.qclass = QClass::IN;
    record.ttl = ttl;
    record.content = nameserver;
    return record;
}

DNSResourceRecord CreateResourceRecordFromProto(const DNSName& domain, const NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& proto, const ui32 defaultTtl) {
    DNSResourceRecord record;
    record.qname = domain.makeLowerCase();
    record.qtype = GetQueryType(proto.type());
    record.qclass = QClass::IN;
    record.ttl = proto.has_ttl() ? proto.ttl() : defaultTtl;
    record.content = proto.data();
    return record;
}

TZoneMetaPtr DetermineZone(const TZonesList& zones, const DNSName& domain) {
    for (const auto& [zoneName, zoneMeta] : zones) {
        if (domain.isPartOf(zoneName)) {
            return zoneMeta;
        }
    }
    return nullptr;
}

template <typename TReq>
TStringBuf GetYpClusterName(NInfra::TRequestPtr<TReq> request) {
    TStringBuf path = request->Path();
    path.SkipPrefix("/");
    return path.SplitOff('/').NextTok('/');
}

bool Equals(const google::protobuf::Message& lhs, const google::protobuf::Message& rhs) {
    return google::protobuf::util::MessageDifferencer::Equivalent(lhs, rhs);
}

template <typename T>
bool Equals(const google::protobuf::RepeatedPtrField<T>& lhs, const google::protobuf::RepeatedPtrField<T>& rhs) {
    return std::equal(lhs.begin(), lhs.end(), rhs.begin(), rhs.end(), [](const T& lhsElement, const T& rhsElement) {
        return google::protobuf::util::MessageDifferencer::Equivalent(lhsElement, rhsElement);
    });
}

template <typename TContainer, typename TRandGen>
void ShuffleRepeatedProto(TContainer& values, TRandGen&& gen) {
    for (size_t i = 1; i < values.size(); ++i) {
        values[i].Swap(&values[gen.Uniform(i + 1)]);
    }
}

} // namespace

struct TModifyReplicaElementCallback : public NYPReplica::IModifyReplicaElementCallback<NYPReplica::TDnsRecordSetReplicaObject> {
    TModifyReplicaElementCallback(TDnsService* service)
        : Service_(service)
    {
    }

    void Do(const TMaybe<NYPReplica::TDnsRecordSetReplicaObject>& /* oldValue */, NYPReplica::TDnsRecordSetReplicaObject& newValue, NYPReplica::EObjectType newValueType) override {
        if (newValueType == NYPReplica::EObjectType::DELETE) {
            return;
        }

        const DNSName domain(newValue.GetKey());
        TZoneMetaPtr zoneMeta = DetermineZone(*Service_->GetZones(), domain);
        const TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;

        if (!zoneMeta || !zoneConfig) {
            return;
        }

        NClient::TDnsRecordSet recordSet = newValue.GetObject();
        auto& records = *recordSet.MutableSpec()->mutable_records();

        bool modify = false;

        const NYpDns::TFilterStoredRecordsPolicy& filterPolicy = zoneConfig->GetFilterStoredRecordsPolicy();
        switch (filterPolicy.GetOrder()) {
            case NYpDns::TFilterStoredRecordsPolicy::RANDOM: {
                const TString hashStr = TString::Join(HostName(), '#', newValue.GetKey());

                ShuffleRepeatedProto(records, TReallyFastRng32(MurmurHash<ui32>(hashStr.data(), hashStr.size())));
                modify = true;
                break;
            }
            case NYpDns::TFilterStoredRecordsPolicy::ORIGINAL:
            default:
                break;
        }

        if (i32 maxRecordsNumber = filterPolicy.GetMaxRecordsNumber(); maxRecordsNumber != -1 && maxRecordsNumber < records.size()) {
            records.Truncate(filterPolicy.GetMaxRecordsNumber());
            modify = true;
        }

        if (modify) {
            newValue = NYPReplica::TDnsRecordSetReplicaObject(recordSet);
        }
    }

private:
    TDnsService* Service_;
};

struct TUpdateCallback : public NYPReplica::IUpdateCallback {
    TUpdateCallback(TString cluster, TDnsService* service)
        : Cluster_(std::move(cluster))
        , Service_(service)
    {
    }

    void Do(ui64 timestamp, const TVector<NYPReplica::TUpdate<NYPReplica::TDnsRecordSetReplicaObject>>& updates) override {
        THashSet<TString> domains;
        for (const auto& [oldValue, newValue] : updates) {
            if (oldValue.Defined()) {
                domains.insert(oldValue->GetKey());
            } else {
                domains.insert(newValue.ReplicaObject.GetKey());
            }
        }
        for (const auto& domain : domains) {
            DNSName domainName(domain.data());
            QC.purgeExact(domainName);
            PC.purgeExact(domainName);

            if (IsSrvServiceSpecified(domainName)) {
                domainName.chopOff();
                QC.purgeExact(domainName);
                PC.purgeExact(domainName);
            }

            TZoneMetaPtr zoneMeta = DetermineZone(*Service_->GetZones(), domainName);
            if (zoneMeta) {
                TWriteGuard guard(zoneMeta->Mutex);
                zoneMeta->LastChangeTimestamp[Cluster_] = timestamp;
            }
        }
    }

private:
    const TString Cluster_;
    TDnsService* Service_;
};

TDnsService::TDnsService(const TConfig& config, const NInfra::TYtLoggerPtr ytLogger)
    : ConfigHolder_(config, config.GetWatchPatchConfig(), config.GetConfigUpdatesLoggerConfig())
    , Config_(ConfigHolder_.Accessor())
    , Logger_(CONFIG_SNAPSHOT_VALUE(Config_, GetLoggerConfig()))
    , BackupLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetBackupLoggerConfig()))
    , YtLogger_(ytLogger)
    , YpToken_(NClient::FindToken())
    , HttpService_(CONFIG_SNAPSHOT_VALUE(Config_, GetHttpServiceConfig()), CreateRouter(*this, *Config_))
    , ReplicasManagementPool_(CreateThreadPool(CONFIG_SNAPSHOT_VALUE(Config_, GetYPClusterConfigs().size())))
    , SensorGroup_(NSensors::NAMESPACE)
{
    const TConfigPtr configSnapshot = Config_.Get();
    if (configSnapshot->GetMemoryLock().GetLockSelfMemory()) {
        LockSelfMemory();
    }

    ConfigHolder_.SetSwitchConfigsCallback([this](const TConfig& oldConfig, const TConfig& newConfig) {
        SwitchConfigsCallback(oldConfig, newConfig);
    });

    UpdateTSIGKeys();
    InitTSIGSecrets();
    InitReplicas();
    UpdateZonesList(); // must be after InitReplicas
    InitSensors();
    InitListZonePool();
}

void TDnsService::Start() {
    ConfigHolder_.StartWatchPatch();
    StartReplicas();
    HttpService_.Start(Logger_.SpawnFrame());
}

void TDnsService::UpdateZonesList() {
    const TConfigPtr config = Config_.Get();

    TZonesListPtr zones = MakeAtomicShared<TZonesList>();
    zones->reserve(config->GetZones().size());

    THashMap<TString, TVector<std::pair<TString, const TYPReplica*>>> replicasByZone(config->GetZones().size());
    THashMap<ui32, TZoneMetaPtr> zoneMetaById(config->GetZones().size());
    THashMap<DNSName, TZoneMetaPtr> zoneMetaByName(config->GetZones().size());
    for (const NYpDns::TZoneConfig& zone : config->GetZones()) {
        const DNSName domain(zone.GetName());

        TZoneMetaPtr newMeta;
        if (const TZoneMetaPtr* currentMeta = ZoneMetaByName_.FindPtr(domain)) {
            newMeta = *currentMeta;
        } else {
            newMeta = MakeAtomicShared<TZoneMeta>(
                Config_.Accessor<NYpDns::TZoneConfig>({"Zones", zone.GetName()}),
                CreateZoneInfo(zone, NextZoneId_++)
            );
        }

        zones->emplace_back(domain, newMeta);
        zoneMetaById.emplace(newMeta->Info.id, newMeta);
        zoneMetaByName.emplace(zone.GetName(), newMeta);

        auto& replicas = replicasByZone[zone.GetName()];
        replicas.reserve(zone.GetYPClusters().size());
        for (const TString& cluster : zone.GetYPClusters()) {
            if (const auto* replicaHolderPtr = Replicas_.FindPtr(cluster); replicaHolderPtr && replicaHolderPtr->Get()) {
                replicas.emplace_back(cluster, replicaHolderPtr->Get());
            }
        }
    }

    TWriteGuard guard(ZonesMutex_);
    Zones_.Swap(zones);
    ZoneMetaById_.swap(zoneMetaById);
    ZoneMetaByName_.swap(zoneMetaByName);
    ReplicasByZone_.swap(replicasByZone);
}

void TDnsService::UpdateTSIGKeys() {
    TTSIGKeysPtr keys = MakeAtomicShared<TTSIGKeys>();
    const TConfigPtr config = Config_.Get();
    keys->reserve(config->GetTSIGConfig().GetKeys().size());
    for (const TTSIGKey& key : config->GetTSIGConfig().GetKeys()) {
        (*keys)[key.GetName()][key.GetAlgorithm()] = key.GetKey();
    }

    TWriteGuard guard(TSIGKeysMutex_);
    TSIGKeys_.Swap(keys);
}

void TDnsService::InitTSIGSecrets() {
    try {
        const TConfigPtr config = Config_.Get();
        TUnbufferedFileInput input(config->GetTSIGConfig().GetSecretsPath());
        TTSIGSecrets_ = NJson::ReadJsonTree(&input, /* throwOnError */ false);
    } catch (...) {
        Logger_.SpawnFrame()->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInitTSIGSecrets(CurrentExceptionMessage()));
    }
}

void TDnsService::InitReplicas() {
    const TConfigPtr config = Config_.Get();
    for (const auto& cluster : config->GetYPClusterConfigs()) {
        auto& replica = Replicas_.emplace(cluster.GetName(), MakeHolder<TYPReplica>(
            Config_.Accessor<NYPReplica::TYPReplicaConfig>("YPReplicaConfig"),
            Config_.Accessor<NYPReplica::TYPClusterConfig>({"YPClusterConfigs", cluster.GetName()}),
            NYP::NYPReplica::TTableRulesHolder<NYP::NYPReplica::TDnsRecordSetReplicaObject>(),
            YpToken_,
            BackupLogger_,
            STORAGE_FORMAT_VERSION
        )).first->second;

        replica->SetUpdateCallback(MakeHolder<TUpdateCallback>(cluster.GetName(), this));
        replica->SetModifyReplicaElementCallback<NYP::NYPReplica::TDnsRecordSetReplicaObject>(MakeHolder<TModifyReplicaElementCallback>(this));
    }
}

void TDnsService::StartReplicas() {
    TVector<NThreading::TFuture<void>> replicasStartFutures;
    for (auto& [cluster, replica] : Replicas_) {
        replicasStartFutures.push_back(NThreading::Async(
            [replicaPtr = replica.Get()] {
                replicaPtr->Start();
            },
            *ReplicasManagementPool_
        ));
    }
    NThreading::WaitExceptionOrAll(replicasStartFutures).GetValueSync();
}

void TDnsService::StopReplicas() {
    TVector<NThreading::TFuture<void>> replicasStopFutures;
    for (auto& [cluster, replica] : Replicas_) {
        replicasStopFutures.push_back(NThreading::Async(
            [replicaPtr = replica.Get()] {
                replicaPtr->Stop();
            },
            *ReplicasManagementPool_
        ));
    }
    NThreading::WaitAll(replicasStopFutures).GetValueSync();
}

void TDnsService::InitSensors() {
    const TConfigPtr config = Config_.Get();
    for (const TString& sensor : config->GetPowerDNSCounters().GetIntGauge()) {
        IntGaugeSensors_.emplace(sensor, NInfra::TIntGaugeSensor(SensorGroup_, sensor));
    }
    for (const TString& sensor : config->GetPowerDNSCounters().GetRate()) {
        RateSensors_.emplace(sensor, NInfra::TRateSensor(SensorGroup_, sensor));
    }
    for (const auto& [histogramName, baseOfHistogram, scaleOfHistogram] : NSensors::HISTOGRAMS_INIT_PARAMETERS_YP_DNS) {
        if (!HistogramSensors_.contains(histogramName)) {
            HistogramSensors_.emplace(histogramName, MakeIntrusive<NInfra::THistogramRateSensor>(
                SensorGroup_, histogramName, NMonitoring::ExponentialHistogram(NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, baseOfHistogram, scaleOfHistogram)));
        }
    }
}

void TDnsService::InitListZonePool() {
    const TConfigPtr config = Config_.Get();

    size_t threadsCount = 1;
    for (const NYpDns::TZoneConfig& zone : config->GetZones()) {
        if (zone.GetSelectRecordSetMode() == NYpDns::ESelectRecordSetMode::MERGE) {
            threadsCount = Max(threadsCount, static_cast<size_t>(zone.GetYPClusters().size()));
        }
    }
    ListZonePool_ = CreateThreadPool(threadsCount);
}

void TDnsService::SwitchConfigsCallback(const TConfig& oldConfig, const TConfig& newConfig) {
    for (int i = 0; i < oldConfig.GetYPClusterConfigs().size(); ++i) {
        Y_ENSURE(oldConfig.GetYPClusterConfigs(i).GetName() == newConfig.GetYPClusterConfigs(i).GetName());

        const auto& oldClusterConfig = oldConfig.GetYPClusterConfigs(i);
        const auto& newClusterConfig = newConfig.GetYPClusterConfigs(i);
        if (!Equals(oldClusterConfig.GetRollbackToBackup(), newClusterConfig.GetRollbackToBackup())) {
            if (newClusterConfig.GetRollbackToBackup().GetRollback()) {
                NApi::TReqRollbackToBackup request;
                *request.MutableRollbackConfig() = newClusterConfig.GetRollbackToBackup();
                NApi::TRspRollbackToBackup response;
                RollbackToBackup(newClusterConfig.GetName(), request, response);
            }
        }

        if (oldClusterConfig.GetEnableUpdates() != newClusterConfig.GetEnableUpdates()) {
            if (newClusterConfig.GetEnableUpdates()) {
                NApi::TRspStartUpdates response;
                StartUpdates(newClusterConfig.GetName(), NApi::TReqStartUpdates{}, response);
            } else {
                NApi::TRspStopUpdates response;
                StopUpdates(newClusterConfig.GetName(), NApi::TReqStopUpdates{}, response);
            }
        }
    }

    if (!Equals(oldConfig.GetZones(), newConfig.GetZones())) {
        UpdateZonesList();
    }

    if (!Equals(oldConfig.GetTSIGConfig().GetKeys(), newConfig.GetTSIGConfig().GetKeys())) {
        UpdateTSIGKeys();
    }
}

void TDnsService::Ping(NInfra::TRequestPtr<NApi::TReqPing>, NInfra::TReplyPtr<NApi::TRspPing> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NEventlog::TServiceRequestStart(NEventlog::EServiceRequestType::PING),
        NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::PING_RESPONSE_TIME)
    );

    NApi::TRspPing result;
    result.set_data("ok");
    reply->Set(result);
}

void TDnsService::Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown>, NInfra::TReplyPtr<NApi::TRspShutdown>) {
    {
        auto frame = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            NEventlog::TServiceRequestStart(NEventlog::EServiceRequestType::SHUTDOWN),
            NEventlog::TServiceRequestStop(),
            HistogramSensors_.at(NSensors::SHUTDOWN_RESPONSE_TIME)
        );

        StopReplicas();
        HttpService_.ShutDown();
    }
    _exit(0); // ¯\_(ツ)_/¯
}

void TDnsService::ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog>, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NEventlog::TServiceRequestStart(NEventlog::EServiceRequestType::REOPEN_LOG),
        NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::REOPEN_LOG_RESPONSE_TIME)
    );

    NApi::TRspReopenLog result;
    result.set_data("ok");
    reply->Set(result);

    Logger_.ReopenLog();
    BackupLogger_.ReopenLog();
    ConfigHolder_.ReopenLog();
    YtLogger_->ReopenLog();
}

void TDnsService::Sensors(NInfra::TRequestPtr<NApi::TReqSensors>, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NEventlog::TServiceRequestStart(NEventlog::EServiceRequestType::SENSORS),
        NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::SENSORS_RESPONSE_TIME)
    );

    const TConfigPtr config = Config_.Get();
    for (const auto& name : config->GetPowerDNSCounters().GetIntGauge()) {
        if (AtomicCounter* counter = S.getPointer(name)) {
            IntGaugeSensors_.at(name).Set(counter->load());
        } else {
            IntGaugeSensors_.at(name).Set(S.read(name));
        }
    }

    with_lock(RateSensorsMutex_) {
        for (const auto& name : config->GetPowerDNSCounters().GetRate()) {
            const ui64 oldValue = RateSensors_.at(name).Get();
            ui64 newValue = 0;
            if (AtomicCounter* counter = S.getPointer(name)) {
                newValue = counter->load();
            } else {
                newValue = S.read(name);
            }
            if (newValue >= oldValue) {
                RateSensors_.at(name).Add(newValue - oldValue);
            } else {
                Y_ASSERT(false && "Value of rate sensor became less");
            }
        }
    }

    for (const auto& [name, replica] : Replicas_) {
        Y_ENSURE(replica);
        replica->UpdateSensors();
    }

    NApi::TRspSensors result;
    TStringOutput stream(*result.MutableData());
    NInfra::SensorRegistryPrint(stream, NInfra::ESensorsSerializationType::SPACK_V1);
    reply->SetAttribute("Content-Type", "application/x-solomon-spack");

    reply->Set(result);
}

void TDnsService::Config(NInfra::TRequestPtr<NApi::TReqConfig>, NInfra::TReplyPtr<NApi::TRspConfig> reply) {
    NApi::TRspConfig result;
    TStringOutput stream(*result.MutableData());
    NJson::TJsonWriter writer(&stream, /* formatOutput */ true);
    NProtobufJson::Proto2Json(*Config_, writer);

    reply->Set(result);
}

void TDnsService::ListBackups(NInfra::TRequestPtr<NApi::TReqListBackups> request, NInfra::TReplyPtr<NApi::TRspListBackups> reply) {
    const TStringBuf clusterName = GetYpClusterName(request);

    NApi::TRspListBackups result;
    if (const auto* replicaPtr = Replicas_.FindPtr(clusterName); replicaPtr && replicaPtr->Get()) {
        const TVector<NYPReplica::TBackupInfo> infos = replicaPtr->Get()->ListBackups();
        for (const NYPReplica::TBackupInfo& info : infos) {
            NApi::TBackupInfo& backupInfo = *result.AddBackupInfos();
            backupInfo.SetId(info.Id);
            backupInfo.SetTimestamp(info.Timestamp);
            backupInfo.SetSize(info.Size);
            backupInfo.SetNumberFiles(info.NumberFiles);
            *backupInfo.MutableMeta() = info.Meta;
        }
    }
    reply->Set(result);
}

void TDnsService::RollbackToBackup(const TStringBuf clusterName, const NApi::TReqRollbackToBackup& request, NApi::TRspRollbackToBackup& response) {
    auto* replicaPtr = Replicas_.FindPtr(clusterName);

    if (!replicaPtr || !replicaPtr->Get()) {
        response.SetStatus(TStringBuilder() << "replica " << clusterName << " not found");
        return;
    }

    try {
        NYPReplica::TRollbackToBackupConfig rollbackConfig = request.GetRollbackConfig();
        rollbackConfig.SetRollback(true);

        if (replicaPtr->Get()->RollbackToBackup(rollbackConfig)) {
            response.SetStatus("ok");
        } else {
            response.SetStatus("failed");
        }
    } catch (...) {
        response.SetStatus(TStringBuilder() << "failed: " << CurrentExceptionMessage());
    }
}

void TDnsService::RollbackToBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup> request, NInfra::TReplyPtr<NApi::TRspRollbackToBackup> reply) {
    NApi::TRspRollbackToBackup response;
    RollbackToBackup(GetYpClusterName(request), request->Get(), response);
    reply->Set(response);
}

void TDnsService::StartUpdates(const TStringBuf clusterName, const NApi::TReqStartUpdates&, NApi::TRspStartUpdates& response) {
    auto* replicaPtr = Replicas_.FindPtr(clusterName);

    if (!replicaPtr || !replicaPtr->Get()) {
        response.SetData(TStringBuilder() << "replica " << clusterName << " not found");
        return;
    }

    replicaPtr->Get()->EnableUpdates();
    response.SetData("ok");
}

void TDnsService::StartUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates> request, NInfra::TReplyPtr<NApi::TRspStartUpdates> reply) {
    NApi::TRspStartUpdates response;
    StartUpdates(GetYpClusterName(request), request->Get(), response);
    reply->Set(response);
}

void TDnsService::StopUpdates(const TStringBuf clusterName, const NApi::TReqStopUpdates&, NApi::TRspStopUpdates& response) {
    auto* replicaPtr = Replicas_.FindPtr(clusterName);

    if (!replicaPtr || !replicaPtr->Get()) {
        response.SetData(TStringBuilder() << "replica " << clusterName << " not found");
        return;
    }

    replicaPtr->Get()->DisableUpdates();
    response.SetData("ok");
}

void TDnsService::StopUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates> request, NInfra::TReplyPtr<NApi::TRspStopUpdates> reply) {
    NApi::TRspStopUpdates response;
    StopUpdates(GetYpClusterName(request), request->Get(), response);
    reply->Set(response);
}

void TDnsService::LockSelfMemory() {
    try {
        LockAllMemory(LockCurrentMemory);
    } catch (const yexception& e) {
        NInfra::NLogEvent::TServiceMemoryLockError ev("Failed to lock memory: " + CurrentExceptionMessage());
        Logger_.SpawnFrame()->LogEvent(ELogPriority::TLOG_ERR, ev);
    }
}

TDnsBackend::TDnsBackend(TDnsService& service)
    : Service_(service)
    , Logger_(service.GetLogger())
    , BaseSensorGroup_(service.GetSensorGroup(), NSensors::YP_DNS_BACKEND_NAMESPACE)
    , SensorGroup_(BaseSensorGroup_)
    , RecordsPtr_(0)
    , RandomGenerator_(Seed())
    , ListZoneMutex_(service.GetListZoneMutex())
    , ListZonePool_(service.GetListZonePool())
{
    InitSensors();
}

void TDnsBackend::InitSensors() {
    for (const auto& [histogramName, baseOfHistogram, scaleOfHistogram] : NSensors::HISTOGRAMS_INIT_PARAMETERS_YP_DNS_BACKEND) {
        if (!HistogramSensors_.contains(histogramName)) {
            HistogramSensors_.emplace(histogramName, MakeIntrusive<NInfra::THistogramRateSensor>(
                SensorGroup_, histogramName, NMonitoring::ExponentialHistogram(NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, baseOfHistogram, scaleOfHistogram)));
        }
    }
}

bool TDnsBackend::ListZoneFromClusterWithLargestTimestamp(
    const DNSName& zone,
    const NYpDns::TZoneConfig& zoneConfig,
    TVector<DNSResourceRecord>& records,
    TBackendEventsLoggerPtr logger,
    const NInfra::TSensorGroup& sensorGroup
) {
    const TVector<std::pair<TString, const TYPReplica*>>& zoneReplicas = Service_.GetReplicas(zoneConfig.GetName());
    if (zoneReplicas.empty()) {
        NInfra::TRateSensor(sensorGroup, NSensors::EMPTY_CLUSTERS_LIST).Inc();
        return false;
    }

    auto replicaIt = zoneReplicas.cend();
    ui64 maxYpTimestamp = 0;
    THolder<TYPReplica::TReplicaSnapshot> clusterSnapshot;
    for (auto it = zoneReplicas.cbegin(); it != zoneReplicas.cend(); ++it) {
        const auto& [cluster, replica] = *it;
        auto snapshot = MakeHolder<TYPReplica::TReplicaSnapshot>(replica->GetReplicaSnapshot());
        ui64 ypTimestamp = replica->GetYpTimestamp(*snapshot).GetOrElse(0);
        if (replicaIt == zoneReplicas.cend() || maxYpTimestamp < ypTimestamp) {
            replicaIt = it;
            maxYpTimestamp = ypTimestamp;
            clusterSnapshot.Reset(snapshot);
        }
    }

    const auto& [cluster, replica] = *replicaIt;
    logger->LogListCluster(cluster, maxYpTimestamp);

    TListOptions listOptions;
    listOptions.Snapshot = *clusterSnapshot;
    listOptions.Filter = [&zone](const TStringBuf key, const TVector<NYPReplica::TStorageElement<NYPReplica::TDnsRecordSetReplicaObject>>&) {
        return DNSName(std::string{key}).isPartOf(zone);
    };
    const auto elements = replica->ListElements<NYPReplica::TDnsRecordSetReplicaObject>(listOptions);

    Records_.reserve(elements.size());
    for (const auto& [key, value] : elements) {
        const auto object = value.front().ReplicaObject.GetObject();
        for (const auto& record : object.Spec().records()) {
            Records_.emplace_back(CreateResourceRecordFromProto(DNSName(key), record, zoneConfig.GetDefaultTtl()));
        }
    }

    return true;
}

TDnsBackend::TListRecordSetsBatchResult TDnsBackend::ListMulticlusterZoneBatch(
    const TVector<std::pair<TString, const TYPReplica*>>& zoneReplicas,
    const NYpDns::TZoneConfig& zoneConfig,
    THashMap<TString, TListOptions> replicasListOptions,
    const ui64 limit
) const {
    auto listFromReplica = [](const TYPReplica& replica, const TListOptions& listOptions) {
        auto elements = replica.ListElements<NYPReplica::TDnsRecordSetReplicaObject>(listOptions);

        TVector<NYpDns::TSerializedRecordSet> result;
        result.reserve(elements.size());
        for (auto&& [key, values] : elements) {
            result.emplace_back(std::move(values.front().ReplicaObject));
        }
        return result;
    };

    THashMap<TString, NThreading::TFuture<TVector<NYpDns::TSerializedRecordSet>>> listResults(zoneReplicas.size());
    for (const auto& [cluster, replica] : zoneReplicas) {
        TListOptions& listOptions = replicasListOptions[cluster];
        listOptions.Limit = limit ? (limit + zoneReplicas.size() - 1) : 0;

        listResults[cluster] = NThreading::Async(
            [&listFromReplica, replica = replica, &listOptions] {
                return listFromReplica(*replica, listOptions);
            },
            ListZonePool_
        );
    }

    THashMap<TString, TVector<NYpDns::TSerializedRecordSet>> recordSets(zoneReplicas.size());
    for (auto& [cluster, resultFuture] : listResults) {
        recordSets.emplace(std::move(cluster), resultFuture.ExtractValueSync());
    }

    TListRecordSetsBatchResult result;
    TString lastId;
    bool hasReachedLimit = false;
    NYpDns::Iterate(recordSets, [&result, &zoneConfig, limit, &lastId, &hasReachedLimit](const NYpDns::TRecordSetReplicas& recordSetReplicas) {
        if (limit && result.ListedRecordSetsNumber == limit) {
            hasReachedLimit = true;
            return false;
        }

        TMaybe<NYpDns::TRecordSet> mergedRecordSet = NYpDns::MergeInOne(
            recordSetReplicas.Replicas,
            NYpDns::TMergeOptions()
                .SetFormChangelist(false)
                .SetMergeAcls(false)
        );
        if (mergedRecordSet.Defined()) {
            DNSName domain(mergedRecordSet->Meta().id());
            for (const auto& record : mergedRecordSet->Spec().records()) {
                result.Records.push_back(CreateResourceRecordFromProto(domain, record, zoneConfig.GetDefaultTtl()));
            }
            ++result.ListedRecordSetsNumber;
        }

        lastId = recordSetReplicas.Fqdn;
        return true;
    });

    result.HasReachedEnd = !hasReachedLimit &&
        AllOf(recordSets, [&replicasListOptions](const auto& it) { return !replicasListOptions[it.first].Limit || it.second.size() < replicasListOptions[it.first].Limit; });
    for (auto& [cluster, listOptions] : replicasListOptions) {
        listOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
        listOptions.SeekKey = lastId;
        result.ContinuationOptions[cluster] = std::move(listOptions);
    }

    return result;
}

bool TDnsBackend::ListMulticlusterZone(
    const DNSName& zone,
    const NYpDns::TZoneConfig& zoneConfig,
    TVector<DNSResourceRecord>& records,
    TBackendEventsLoggerPtr logger,
    const NInfra::TSensorGroup& sensorGroup
) {
    const TVector<std::pair<TString, const TYPReplica*>>& zoneReplicas = Service_.GetReplicas(zoneConfig.GetName());
    if (zoneReplicas.empty()) {
        NInfra::TRateSensor(sensorGroup, NSensors::EMPTY_CLUSTERS_LIST).Inc();
        return false;
    }

    THashMap<TString, TListOptions> replicasListOptions;
    for (const auto& [cluster, replica] : zoneReplicas) {
        TListOptions& listOptions = replicasListOptions[cluster];
        listOptions.Snapshot = replica->GetReplicaSnapshot();
        listOptions.Filter = [&zone](const TStringBuf key, const TVector<NYPReplica::TStorageElement<NYPReplica::TDnsRecordSetReplicaObject>>&) {
            return DNSName(std::string{key}).isPartOf(zone);
        };
    }

    const ui64 batchSize = 10'000;
    while (true) {
        TListRecordSetsBatchResult listResultBatch = ListMulticlusterZoneBatch(zoneReplicas, zoneConfig, replicasListOptions, batchSize);

        std::move(listResultBatch.Records.begin(), listResultBatch.Records.end(), std::back_inserter(records));
        replicasListOptions = std::move(listResultBatch.ContinuationOptions);

        if (listResultBatch.HasReachedEnd) {
            break;
        }
    }

    return true;
}

bool TDnsBackend::list(const DNSName& domain, int, bool) {
    TBackendEventsLoggerPtr logger = TBackendEventsLogger::CreateBackendEventsLogger(Logger_);
    auto subframe = logger->CreateSubframe(
        NEventlog::TBackendRequestStart(NEventlog::EBackendRequestType::LIST),
        NEventlog::TBackendRequestStop()
    );

    TTryGuard<TMutex> guard(ListZoneMutex_);
    if (!guard.WasAcquired()) {
        logger->LogListResult(domain, 0, false, "Busy");
        return false;
    }

    TZoneMetaPtr zoneMeta = DetermineZone(domain);
    TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;
    if (!zoneMeta || !zoneConfig || DNSName(zoneConfig->GetName()) != domain) {
        logger->LogUnknownZone(domain);
        NInfra::TRateSensor(MakeSensorGroup(QType(QType::AXFR), NSensors::UNKNOWN_ZONE), NSensors::REQUESTS).Inc();
        return false;
    }

    logger->LogZoneInfo(*zoneConfig);

    NInfra::TSensorGroup sensorGroup = BaseSensorGroup_;

    sensorGroup.AddLabel(NSensors::ZONE, zoneConfig->GetName());
    NInfra::TDurationSensor durationSensor(sensorGroup, NSensors::LIST_ZONE_RESPONSE_TIME);

    sensorGroup.AddLabel(NSensors::RECORD_TYPE, QType(QType::AXFR).getName());
    NInfra::TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    Records_.clear();
    RecordsPtr_ = 0;

    if (zoneConfig->GetIsAuthoritative()) {
        for (const TString& nameserver : zoneConfig->GetNameservers()) {
            Records_.emplace_back(CreateNSRecord(domain, nameserver, zoneConfig->GetNSRecordTtl()));
        }
    }

    bool successful = false;
    switch (zoneConfig->GetSelectRecordSetMode()) {
        case NYpDns::ESelectRecordSetMode::LARGEST_TIMESTAMP: {
            successful = ListZoneFromClusterWithLargestTimestamp(domain, *zoneConfig, Records_, logger, sensorGroup);
            break;
        }
        case NYpDns::ESelectRecordSetMode::MERGE: {
            successful = ListMulticlusterZone(domain, *zoneConfig, Records_, logger, sensorGroup);
            break;
        }
    }

    logger->LogListResult(zoneConfig->GetName(), Records_.size(), successful);
    durationSensor.Update();
    NInfra::TIntGaugeSensor(sensorGroup, NSensors::RECORDS_NUMBER).Set(Records_.size());

    return successful;
}

void TDnsBackend::lookup(const QType& queryType, const DNSName& domain, DNSPacket* packet, int) {
    Records_.clear();
    RecordsPtr_ = 0;
    SkippedLookup_ = false;

    if (domain.isWildcard()) {
        SkippedLookup_ = true;
        return;
    }

    BackendLogger_.Reset(TBackendEventsLogger::CreateBackendEventsLogger(Logger_));
    Subframes_.push_back(BackendLogger_->CreateSubframe(
        NEventlog::TBackendRequestStart(NEventlog::EBackendRequestType::LOOKUP),
        NEventlog::TBackendRequestStop(),
        HistogramSensors_.at(NSensors::LOOKUP_RESPONSE_TIME)
    ));

    BackendLogger_->LogBackendRequestData(queryType, domain);
    BackendLogger_->LogPacketInfo(packet);

    SensorGroup_ = BaseSensorGroup_;
    SensorGroup_.AddLabel(NSensors::RECORD_TYPE, queryType.getName());
    if (packet) {
        SensorGroup_.AddLabel(NSensors::PACKET_RECORD_TYPE, packet->qtype.getName());
    }

    TZoneMetaPtr zoneMeta = DetermineZone(domain);
    const TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;
    if (!zoneMeta || !zoneConfig) {
        BackendLogger_->LogUnknownZone(domain);

        SensorGroup_.AddLabel(NSensors::ZONE, NSensors::UNKNOWN_ZONE);
        NInfra::TRateSensor(SensorGroup_, NSensors::REQUESTS).Inc();

        return;
    } else {
        BackendLogger_->LogZoneInfo(*zoneConfig);

        SensorGroup_.AddLabel(NSensors::ZONE, zoneConfig->GetName());
        NInfra::TRateSensor(SensorGroup_, NSensors::REQUESTS).Inc();
    }

    if (!IsIn(ALLOWED_TYPES, queryType)) {
        BackendLogger_->LogUnallowedResourceType();
        NInfra::TRateSensor(SensorGroup_, NSensors::UNALLOWED_RECORD_TYPE).Inc();
        return;
    }

    const bool isAny = queryType == QType::ANY;

    if (isAny || queryType == QType::NS) {
        if (zoneConfig->GetIsAuthoritative() && domain == DNSName(zoneConfig->GetName())) {
            for (const TString& nameserver : zoneConfig->GetNameservers()) {
                Records_.emplace_back(CreateNSRecord(domain, nameserver, zoneConfig->GetNSRecordTtl()));
            }
        }
        if (!isAny) {
            return;
        }
    }

    TVector<TString> keysToCheck = {domain.makeLowerCase().toStringNoDot().data()};
    if ((isAny || !packet || packet->qtype == QType::PTR || packet->qtype == QType::ANY) && zoneConfig->GetSupportPTR()) {
        keysToCheck.emplace_back(domain.makeLowerCase().toString());
    }

    if ((isAny || !packet || packet->qtype == QType::SRV || packet->qtype == QType::ANY) && zoneConfig->GetSupportSRV()) {
        if (!domain.isWildcard() && !IsSrvServiceSpecified(domain)) {
            keysToCheck.emplace_back((DNSName("_id_") + domain).makeLowerCase().toStringNoDot());
        }
    }

    TVector<NClient::TDnsRecordSet> selectionResult;

    const TVector<std::pair<TString, const TYPReplica*>>& zoneReplicas = Service_.GetReplicas(zoneConfig->GetName());
    for (const TString& key : keysToCheck) {
        TMaybe<NClient::TDnsRecordSet> keySelectionResult;

        switch (zoneConfig->GetSelectRecordSetMode()) {
            case NYpDns::ESelectRecordSetMode::LARGEST_TIMESTAMP: {
                ui64 maxTimestamp = 0;
                for (const auto& [cluster, replica] : zoneReplicas) {
                    auto replicaSelectionResult = replica->GetByKey<NYPReplica::TDnsRecordSetReplicaObject>(key);
                    if (replicaSelectionResult.Defined() && !replicaSelectionResult->Objects.empty() && maxTimestamp < replicaSelectionResult->YpTimestamp) {
                        maxTimestamp = replicaSelectionResult->YpTimestamp;
                        keySelectionResult = std::move(replicaSelectionResult->Objects.front());
                    }
                }
                break;
            }
            case NYpDns::ESelectRecordSetMode::MERGE: {
                THashMap<TString, TMaybe<NYpDns::TRecordSet>> recordSetReplicas(zoneReplicas.size());
                for (const auto& [cluster, replica] : zoneReplicas) {
                    auto replicaSelectionResult = replica->GetByKey<NYPReplica::TDnsRecordSetReplicaObject>(key);
                    if (replicaSelectionResult.Defined() && !replicaSelectionResult->Objects.empty()) {
                        recordSetReplicas.emplace(cluster, replicaSelectionResult->Objects.front());
                    } else {
                        recordSetReplicas.emplace(cluster, Nothing());
                    }
                }
                TMaybe<NYpDns::TRecordSet> mergedRecordSet = NYpDns::MergeInOne(
                    recordSetReplicas,
                    NYpDns::TMergeOptions()
                        .SetFormChangelist(false)
                        .SetMergeAcls(false)
                );
                if (mergedRecordSet.Defined()) {
                    keySelectionResult = mergedRecordSet->MakeYpObject();
                }
                break;
            }
        }

        if (keySelectionResult.Defined()) {
            selectionResult.push_back(std::move(*keySelectionResult));
        }
    }

    TMap<NYpDns::ERecordType, TVector<DNSResourceRecord>> type2Records;
    for (const NClient::TDnsRecordSet& dnsRecordSet : selectionResult) {
        for (const auto& record : dnsRecordSet.Spec().records()) {
            if (isAny || queryType == GetQueryType(record.type())) {
                DNSResourceRecord resourceRecord = CreateResourceRecordFromProto(domain, record, zoneConfig->GetDefaultTtl());
                type2Records[NYpDns::GetRecordType(record.type())].push_back(std::move(resourceRecord));
            }
        }
    }

    for (auto& [type, records] : type2Records) {
        SortUnique(records);
        if (const NYpDns::TTypeResponsePolicy* responsePolicy = NYpDns::FindTypeResponsePolicy(*zoneConfig, type)) {
            if (responsePolicy->GetOrder() == NYpDns::TTypeResponsePolicy::RANDOM) {
                Shuffle(records.begin(), records.end(), RandomGenerator_);
            }
            if (i32 recordsNumber = responsePolicy->GetMaxRecordsNumber(); recordsNumber != -1 && recordsNumber < records.size()) {
                records.resize(recordsNumber);
            }
        }
        std::move(records.begin(), records.end(), std::back_inserter(Records_));
    }
}

bool TDnsBackend::get(DNSResourceRecord& resourceRecord) {
    if (RecordsPtr_ == Records_.size()) {
        if (BackendLogger_) {
            BackendLogger_->LogLookupResult(Records_.size());
            Subframes_.clear();
            BackendLogger_.Drop();
        }

        if (!SkippedLookup_ && Records_.empty()) {
            NInfra::TRateSensor(SensorGroup_, NSensors::EMPTY_RESPONSES).Inc();
        }

        return false;
    }

    std::swap(resourceRecord, Records_[RecordsPtr_++]);
    return true;
}

ui64 TDnsBackend::GetSerial(const NYpDns::TZoneConfig& zoneConfig) const {
    ui64 timestamp = 0;
    const TVector<std::pair<TString, const TYPReplica*>>& zoneReplicas = Service_.GetReplicas(zoneConfig.GetName());
    for (const auto& [cluster, replica] : zoneReplicas) {
        timestamp = Max(timestamp, replica->GetYpTimestamp().GetOrElse(0));
    }
    return NYpDns::MakeSerialFromYpTimestamp(timestamp);
}

ui64 TDnsBackend::GetLastChangeTimestamp(const DNSName& zoneName) const {
    TZoneMetaPtr meta = Service_.GetZone(zoneName);
    ui64 result = 0;
    TReadGuard guard(meta->Mutex);
    for (const auto& [cluster, timestamp] : meta->LastChangeTimestamp) {
        result = Max(result, timestamp);
    }
    return result;
}

bool TDnsBackend::getSOA(const DNSName& domain, SOAData& soaData) {
    TBackendEventsLoggerPtr logger = TBackendEventsLogger::CreateBackendEventsLogger(Logger_);
    auto subframe = logger->CreateSubframe(
        NEventlog::TBackendRequestStart(NEventlog::EBackendRequestType::GET_SOA),
        NEventlog::TBackendRequestStop(),
        HistogramSensors_.at(NSensors::GET_SOA_RESPONSE_TIME)
    );

    logger->LogBackendRequestData(QType(QType::SOA), domain);

    TZoneMetaPtr zoneMeta = DetermineZone(domain);
    const TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;

    if (!zoneMeta || !zoneConfig) {
        logger->LogUnknownZone(domain);
        logger->LogLookupResult(0);

        NInfra::TSensorGroup sensorGroup = MakeSensorGroup(QType(QType::SOA), NSensors::UNKNOWN_ZONE);
        NInfra::TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();
        NInfra::TRateSensor(sensorGroup, NSensors::EMPTY_RESPONSES).Inc();

        return false;
    }

    NInfra::TRateSensor(MakeSensorGroup(QType(QType::SOA), zoneConfig->GetName()), NSensors::REQUESTS).Inc();

    soaData.qname = DNSName(zoneConfig->GetName());
    soaData.nameserver = DNSName(zoneConfig->GetPrimaryNameserver());
    soaData.hostmaster = DNSName(zoneConfig->GetHostmaster());
    soaData.ttl = zoneConfig->GetSOARecordTtl();
    soaData.serial = GetSerial(*zoneConfig);
    soaData.refresh = zoneConfig->GetSOARefresh();
    soaData.retry = zoneConfig->GetSOARetry();
    soaData.expire = arg().asNum("soa-expire-default");
    soaData.default_ttl = zoneConfig->GetSOARecordMinimumTtl();
    soaData.db = this;

    logger->LogZoneInfo(*zoneConfig);
    logger->LogLookupResult(1);

    return true;
}

bool TDnsBackend::getDomainMetadata(const DNSName& name, const std::string& kind, std::vector<std::string>& meta) {
    if (kind == "TSIG-ALLOW-AXFR") {
        const TAtomicSharedPtr<const TConfig> config = Service_.GetConfig().Get();
        const auto& keys = config->GetTSIGConfig().GetKeys();
        meta.reserve(keys.size());
        for (const TTSIGKey& key : keys) {
            meta.emplace_back(key.GetKey());
        }
        return true;
    } else if (kind == "ALLOW-AXFR-FROM") {
        TZoneMetaPtr zoneMeta = DetermineZone(name);
        const TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;
        if (!zoneMeta || !zoneConfig) {
            return false;
        }

        if (!zoneConfig->GetTransferConfig().GetEnabled()) {
            return false;
        }

        meta.reserve(zoneConfig->GetTransferConfig().GetAxfrAllowList().size());
        for (const TString& ip : zoneConfig->GetTransferConfig().GetAxfrAllowList()) {
            meta.emplace_back(ip);
        }
        return true;
    }

    return false;
}

bool TDnsBackend::getTSIGKey(const DNSName& name, DNSName* algorithm, string* content) {
    const TTSIGKeysPtr keys = Service_.GetTSIGKeys();
    if (!keys) {
        return false;
    }

    const auto* keysByAlgo = keys->FindPtr(name.toStringNoDot());
    if (!keysByAlgo) {
        return false;
    }

    if (algorithm->empty()) {
        *algorithm = DNSName(keysByAlgo->begin()->first);
    }

    const TString* keyName = keysByAlgo->FindPtr(algorithm->toStringNoDot());
    if (!keyName) {
        return false;
    }

    NJson::TJsonValue secret;
    if (!Service_.GetTSIGSecrets().GetValue(*keyName, &secret) || !secret.IsString()) {
        return false;
    }

    *content = secret.GetString().data();
    return true;
}

void TDnsBackend::alsoNotifies(const DNSName& domain, std::set<string>* ips) {
    if (!ips) {
        return;
    }

    // only one backend should list ips to notify
    if (!ips->empty()) {
        return;
    }

    TZoneMetaPtr zoneMeta = Service_.GetZone(domain);
    const TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;
    if (!zoneMeta || !zoneConfig) {
        return;
    }

    if (!zoneConfig->GetTransferConfig().GetEnabled()) {
        return;
    }

    ips->insert(zoneConfig->GetTransferConfig().GetNotifyAddresses().begin(), zoneConfig->GetTransferConfig().GetNotifyAddresses().end());
}

void TDnsBackend::getUpdatedMasters(std::vector<DomainInfo>* domains) {
    if (!domains) {
        return;
    }

    // only one backend should list updates zones
    if (!domains->empty()) {
        return;
    }

    for (const auto& [zoneName, zoneMeta] : *Service_.GetZones()) {
        const TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = zoneMeta ? zoneMeta->Config.Get() : nullptr;
        if (!zoneConfig || !zoneConfig->GetTransferConfig().GetEnabled()) {
            continue;
        }

        const ui64 serial = MakeSerialFromTimestamp(GetLastChangeTimestamp(zoneName));
        TReadGuard guard(zoneMeta->Mutex);
        if (serial > zoneMeta->Info.notified_serial) {
            DomainInfo& info = domains->emplace_back(zoneMeta->Info);
            info.serial = serial;
            info.backend = this;
        }
    }
}

void TDnsBackend::setNotified(uint32_t id, uint32_t serial) {
    TZoneMetaPtr zoneMeta = Service_.GetZone(id);
    if (!zoneMeta) {
        return;
    }

    TWriteGuard guard(zoneMeta->Mutex);
    zoneMeta->Info.notified_serial = serial;
}

TZoneMetaPtr TDnsBackend::DetermineZone(const DNSName& domain) const {
    return ::NYP::DNS::DetermineZone(*Service_.GetZones(), domain);
}

NInfra::TSensorGroup TDnsBackend::MakeSensorGroup(const QType& resourceType, const TStringBuf zone) const {
    NInfra::TSensorGroup group = BaseSensorGroup_;
    group.AddLabel(NSensors::RECORD_TYPE, resourceType.getName());
    group.AddLabel(NSensors::ZONE, zone);
    return group;
}

TDnsFactory::TDnsFactory()
    : BackendFactory("YP_DNS")
{
}

DNSBackend* TDnsFactory::make(const std::string&) {
    try {
        with_lock (InitServiceMutex_) {
            if (!Service_) {
                TConfig config;
                if (!GetConfigPath().empty()) {
                    NProtoConfig::LoadConfigFromResource(GetConfigPath(), config);
                } else if (TString conf; NResource::FindExact("/test_proto_config.json", &conf)) {
                    NProtoConfig::ParseConfigFromJson(conf, config);
                } else {
                    NProtoConfig::LoadConfigFromResource("/proto_config.json", config);
                }
                PatchConfig(config);
                PrettyPrintConfig(config);
                InitYtEnvironment(config);
                Service_ = MakeHolder<TDnsService>(config, YtLogger_);
                Service_->Start();
            }
        }
        return new TDnsBackend(*Service_);
    } catch (...) {
        TBackTrace::FromCurrentException().PrintTo(Cerr);
        throw;
    }
}

void TDnsFactory::declareArguments(const std::string&) {
    arg().set("service-port", "Port for handling service requests (ping, shutdown, reopenlog, etc.)") = "0";
    arg().set("instance-name", "Instance name used in requests to YP Updates Coordinator") = "";
    arg().set("location", "Location name") = "";
    arg().set("coordinator-service", "Service name in terms of YP Updates Coordinator") = "";
    arg().set("config-path") = "";
}

ui16 TDnsFactory::GetServicePort() const {
    ui16 port = arg().asNum("service-port");
    if (port == 0) {
        port = arg().asNum("local-port") + 1;
    }
    return port;
}

TString TDnsFactory::GetInstanceName() const {
    return TString{arg()["instance-name"]};
}

TString TDnsFactory::GetLocation() const {
    return TString{arg()["location"]};
}

TString TDnsFactory::GetCoordinatorService() const {
    return TString{arg()["coordinator-service"]};
}

TString TDnsFactory::GetConfigPath() const {
    return TString{arg()["config-path"]};
}

void TDnsFactory::PatchConfig(TConfig& config) const {
    config.MutableHttpServiceConfig()->SetPort(GetServicePort());

    const TString instanceName = GetInstanceName();
    const TString location = GetLocation();
    const TString coordinatorService = GetCoordinatorService();

    for (NYPReplica::TYPClusterConfig& clusterConfig : *config.MutableYPClusterConfigs()) {
        if (clusterConfig.GetRetrieveTargetStateConfig().HasCoordinator()) {
            if (!instanceName.empty()) {
                clusterConfig.MutableRetrieveTargetStateConfig()->MutableCoordinator()->MutableClientConfig()->SetInstanceName(instanceName);
            }

            if (!location.empty()) {
                clusterConfig.MutableRetrieveTargetStateConfig()->MutableCoordinator()->MutableClientConfig()->SetInstanceLocation(location);
            }

            if (!coordinatorService.empty()) {
                clusterConfig.MutableRetrieveTargetStateConfig()->MutableCoordinator()->MutableClientConfig()->SetService(coordinatorService);
            }
        }
    }
}

void TDnsFactory::InitYtEnvironment(const TConfig& config) {
    YtLogger_ = NInfra::InitYtLogger(config.GetYtLoggerConfig());
}

} // namespace NYP::DNS
