#pragma once

#include <infra/yp_dns/libs/config/protos/config.pb.h>
#include <infra/yp_dns/libs/logger/events/events_decl.ev.pb.h>
#include <infra/yp_dns/libs/logger/request_logger.h>
#include <infra/yp_dns/libs/router_api/router_api.h>
#include <infra/yp_dns/libs/service_iface/service_iface.h>

#include <infra/libs/http_service/service.h>
#include <infra/libs/logger/yt_logger/logger.h>
#include <infra/libs/updatable_proto_config/accessor.h>
#include <infra/libs/updatable_proto_config/holder.h>
#include <infra/libs/yp_replica/replica_objects.h>
#include <infra/libs/yp_replica/yp_replica.h>

#include <infra/contrib/pdns/power_dns/dnsbackend.hh>

#include <library/cpp/json/json_reader.h>

#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/random/fast.h>
#include <util/system/rwlock.h>
#include <util/thread/pool.h>

template <>
struct THash<DNSName> {
    size_t operator()(const DNSName& name) const;
};

namespace NYP::DNS {

    struct TZoneMeta {
        TZoneMeta(NUpdatableProtoConfig::TAccessor<NYpDns::TZoneConfig>&& config, DomainInfo&& info)
            : Config(std::move(config))
            , Info(std::move(info))
        {
        }

        TRWMutex Mutex;
        NUpdatableProtoConfig::TAccessor<NYpDns::TZoneConfig> Config;
        DomainInfo Info;
        THashMap<TString, ui64> LastChangeTimestamp;
    };

    using TConfigPtr = TAtomicSharedPtr<const TConfig>;

    using TZoneMetaPtr = TAtomicSharedPtr<TZoneMeta>;

    using TZonesList = TVector<std::pair<DNSName, TZoneMetaPtr>>;
    using TZonesListPtr = TAtomicSharedPtr<TZonesList>;

    using TTSIGKeys = THashMap<TString, TMap<TString, TString>>;  // name -> algorithm -> key
    using TTSIGKeysPtr = TAtomicSharedPtr<TTSIGKeys>;
    using TTSIGSecrets = NJson::TJsonValue;

    using TYPReplica = NYPReplica::TYPReplica<NYPReplica::TDnsRecordSetReplicaObject>;

    namespace {
        struct TDnsService: public IService {
        public:
            TDnsService(const TConfig& config, const NInfra::TYtLoggerPtr ytLogger);

            void Start() override;

            void Ping(NInfra::TRequestPtr<NApi::TReqPing>, NInfra::TReplyPtr<NApi::TRspPing> reply) override;

            void Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown>, NInfra::TReplyPtr<NApi::TRspShutdown> reply) override;

            void ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog>, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) override;

            void Sensors(NInfra::TRequestPtr<NApi::TReqSensors>, NInfra::TReplyPtr<NApi::TRspSensors> reply) override;

            void Config(NInfra::TRequestPtr<NApi::TReqConfig>, NInfra::TReplyPtr<NApi::TRspConfig> reply) override;

            void ListBackups(NInfra::TRequestPtr<NApi::TReqListBackups>, NInfra::TReplyPtr<NApi::TRspListBackups>) override;

            void RollbackToBackup(const TStringBuf clusterName, const NApi::TReqRollbackToBackup&, NApi::TRspRollbackToBackup&);
            void RollbackToBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup>, NInfra::TReplyPtr<NApi::TRspRollbackToBackup>) override;

            void StartUpdates(const TStringBuf clusterName, const NApi::TReqStartUpdates&, NApi::TRspStartUpdates&);
            void StartUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates>, NInfra::TReplyPtr<NApi::TRspStartUpdates>) override;

            void StopUpdates(const TStringBuf clusterName, const NApi::TReqStopUpdates&, NApi::TRspStopUpdates&);
            void StopUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates>, NInfra::TReplyPtr<NApi::TRspStopUpdates>) override;

            const THashMap<TString, THolder<TYPReplica>>& GetReplicas() const {
                return Replicas_;
            }

            const TVector<std::pair<TString, const TYPReplica*>>& GetReplicas(const TString& zone) const {
                return ReplicasByZone_.ValueRef(zone, Default<TVector<std::pair<TString, const TYPReplica*>>>());
            }

            const NUpdatableProtoConfig::TAccessor<TConfig>& GetConfig() const {
                return Config_;
            }

            TZonesListPtr GetZones() const {
                TReadGuard guard(ZonesMutex_);
                return Zones_;
            }

            TZoneMetaPtr GetZone(ui32 id) const {
                TReadGuard guard(ZonesMutex_);
                const TZoneMetaPtr* it = ZoneMetaById_.FindPtr(id);
                return it ? *it : nullptr;
            }

            TZoneMetaPtr GetZone(const DNSName& name) const {
                TReadGuard guard(ZonesMutex_);
                const TZoneMetaPtr* it = ZoneMetaByName_.FindPtr(name);
                return it ? *it : nullptr;
            }

            TTSIGKeysPtr GetTSIGKeys() const {
                TReadGuard guard(TSIGKeysMutex_);
                return TSIGKeys_;
            }

            const TTSIGSecrets& GetTSIGSecrets() const {
                return TTSIGSecrets_;
            }

            NInfra::TLogger& GetLogger() {
                return Logger_;
            }

            NInfra::TLogger& GetBackupLogger() {
                return BackupLogger_;
            }

            const NInfra::TSensorGroup& GetSensorGroup() const {
                return SensorGroup_;
            }

            TMutex& GetListZoneMutex() {
                return ListZoneMutex_;
            }

            IThreadPool& GetListZonePool() {
                return *ListZonePool_;
            }

        private:
            void LockSelfMemory();

            void UpdateZonesList();
            void UpdateTSIGKeys();
            void InitTSIGSecrets();
            void InitReplicas();
            void StartReplicas();
            void StopReplicas();

            void InitSensors();

            void InitListZonePool();

            void SwitchConfigsCallback(const TConfig& oldConfig, const TConfig& newConfig);

        private:
            NUpdatableProtoConfig::TConfigHolder<TConfig> ConfigHolder_;
            NUpdatableProtoConfig::TAccessor<TConfig> Config_;
            NInfra::TLogger Logger_;
            NInfra::TLogger BackupLogger_;
            NInfra::TYtLoggerPtr YtLogger_;
            TString YpToken_;
            NInfra::THttpService HttpService_;

            THolder<IThreadPool> ReplicasManagementPool_;

            TRWMutex ZonesMutex_;
            ui32 NextZoneId_ = 0;
            TZonesListPtr Zones_;
            THashMap<ui32, TZoneMetaPtr> ZoneMetaById_;
            THashMap<DNSName, TZoneMetaPtr> ZoneMetaByName_;

            TMutex ListZoneMutex_;
            THolder<IThreadPool> ListZonePool_;

            TRWMutex TSIGKeysMutex_;
            TTSIGKeysPtr TSIGKeys_;
            TTSIGSecrets TTSIGSecrets_;

            THashMap<TString, THolder<TYPReplica>> Replicas_;
            THashMap<TString, TVector<std::pair<TString, const TYPReplica*>>> ReplicasByZone_;
            const NInfra::TSensorGroup SensorGroup_;
            THashMap<TString, NInfra::TIntGaugeSensor> IntGaugeSensors_;
            THashMap<TString, NInfra::TRateSensor> RateSensors_;
            THashMap<TString, TIntrusivePtr<NInfra::THistogramRateSensor>> HistogramSensors_;
            TMutex RateSensorsMutex_;
        };

    }

    class TDnsBackend: public DNSBackend {
    public:
        using TListOptions = TYPReplica::TListOptions<NYPReplica::TDnsRecordSetReplicaObject>;

        TDnsBackend(TDnsService& service);
        bool list(const DNSName&, int, bool) override;
        void lookup(const QType& queryType, const DNSName& domain, DNSPacket* packet, int) override;
        bool get(DNSResourceRecord& resourceRecord) override;
        bool getSOA(const DNSName& domain, SOAData& soaData) override;

        bool getDomainMetadata(const DNSName& name, const std::string& kind, std::vector<std::string>& meta) override;
        bool getTSIGKey(const DNSName& name, DNSName* algorithm, string* content) override;

        void alsoNotifies(const DNSName& domain, std::set<string>* ips) override;
        void getUpdatedMasters(std::vector<DomainInfo>* domains) override;
        void setNotified(uint32_t id, uint32_t serial) override;

    private:
        ui64 GetSerial(const NYpDns::TZoneConfig& zoneConfig) const;

        ui64 GetLastChangeTimestamp(const DNSName& zoneConfig) const;

        struct TListRecordSetsBatchResult {
            TVector<DNSResourceRecord> Records;
            THashMap<TString, TListOptions> ContinuationOptions;
            ui64 ListedRecordSetsNumber = 0;
            bool HasReachedEnd = false;
        };

        bool ListZoneFromClusterWithLargestTimestamp(
            const DNSName& zone,
            const NYpDns::TZoneConfig& zoneConfig,
            TVector<DNSResourceRecord>& records,
            TBackendEventsLoggerPtr logger,
            const NInfra::TSensorGroup& sensorGroup
        );

        TListRecordSetsBatchResult ListMulticlusterZoneBatch(
            const TVector<std::pair<TString, const TYPReplica*>>& zoneReplicas,
            const NYpDns::TZoneConfig& zoneConfig,
            THashMap<TString, TListOptions> replicasListOptions,
            const ui64 limit
        ) const;

        bool ListMulticlusterZone(
            const DNSName& zone,
            const NYpDns::TZoneConfig& zoneConfig,
            TVector<DNSResourceRecord>& records,
            TBackendEventsLoggerPtr logger,
            const NInfra::TSensorGroup& sensorGroup
        );

    private:
        void InitSensors();
        TZoneMetaPtr DetermineZone(const DNSName& domain) const;
        const NYpDns::TTypeResponsePolicy* FindTypeResponsePolicy(const NYpDns::TZoneConfig& zoneConfig, const QType& recordType) const;
        NInfra::TSensorGroup MakeSensorGroup(const QType& resourceType, const TStringBuf zone) const;

    private:
        const TDnsService& Service_;
        NInfra::TLogger& Logger_;
        TBackendEventsLoggerPtr BackendLogger_;
        const NInfra::TSensorGroup BaseSensorGroup_;
        NInfra::TSensorGroup SensorGroup_;
        TVector<NInfra::TFrameBioChapter> Subframes_;
        THashMap<TStringBuf, TIntrusivePtr<NInfra::THistogramRateSensor>> HistogramSensors_;

    private:
        TVector<DNSResourceRecord> Records_;
        size_t RecordsPtr_;
        bool SkippedLookup_ = false;
        TReallyFastRng32 RandomGenerator_;

        TMutex& ListZoneMutex_;
        IThreadPool& ListZonePool_;
    };

    class TDnsFactory: public BackendFactory {
    public:
        TDnsFactory();
        DNSBackend* make(const std::string&);
        void declareArguments(const std::string&) override;

    private:
        ui16 GetServicePort() const;
        TString GetInstanceName() const;
        TString GetLocation() const;
        TString GetCoordinatorService() const;
        TString GetConfigPath() const;

        void PatchConfig(TConfig& config) const;
        void InitYtEnvironment(const TConfig& config);

    private:
        TMutex InitServiceMutex_;
        THolder<TDnsService> Service_;
        NInfra::TYtLoggerPtr YtLogger_;
    };

}
