#pragma once

#include <infra/libs/logger/event_iterator.h>
#include <infra/libs/logger/log_viewer.h>

#include <util/generic/string.h>
#include <util/system/types.h>
#include <library/cpp/protobuf/json/proto2json.h>

namespace NYP::DNS::NPrivate {

using TDataUnit = TMap<TString, TVector<TString>>;
using TData = TVector<TDataUnit>;
using TEventProcessor = std::function<void(const NInfra::TEvent*, TDataUnit&)>;
using TEventProcessors = TVector<TEventProcessor>;

class TViewStatistics: public NInfra::NViewLog::IViewLog {
public:
    TViewStatistics()
        : EventFactory_(NProtoBuf::TEventFactory::Instance()) {}

    void PrintDataUnit(const TDataUnit& dataUnit) const override;

protected:
    void ProcessDataCore(TData& data, const TString& keyToFind) const;

    ui64 IdByName(const TString& name) {
        return EventFactory_->IdByName(name);
    }

private:
    NProtoBuf::TEventFactory* EventFactory_;
};

class TViewStatisticsRemoteAddress: public TViewStatistics {
public:
    TViewStatisticsRemoteAddress()
        : AllowedId_(IdByName("TPacketInfo")) {}

    void ProcessEvent(const NInfra::TEvent* event, TDataUnit& dataUnit) const override;

    bool FilterFrame(const TDataUnit& dataUnit) const override {
        return dataUnit.contains("RemoteHost");
    }

    void ProcessData(TData& data) const override {
        ProcessDataCore(data, "RemoteHost");
    }

private:
    ui64 AllowedId_;
};

REGISTER_VIEWER(remote_address,
                TViewStatisticsRemoteAddress,
                "Print a sorted table of occurences of remote addresses.")

class TViewStatisticsRequestAddress: public TViewStatistics {
public:
    TViewStatisticsRequestAddress()
        : AllowedId_(IdByName("TPacketInfo")) {}
    void ProcessEvent(const NInfra::TEvent* event, TDataUnit& dataUnit) const override;

    bool FilterFrame(const TDataUnit& dataUnit) const override {
        return dataUnit.contains("DestinationAddress");
    }

    void ProcessData(TData& data) const override {
        ProcessDataCore(data, "DestinationAddress");
    }

private:
    ui64 AllowedId_;
};

REGISTER_VIEWER(request_address,
               TViewStatisticsRequestAddress,
               "Print a sorted table of occurences of requested addresses.")

class TViewStatisticsEmptyResponse: public TViewStatistics {
public:
    TViewStatisticsEmptyResponse();

    void ProcessEvent(const NInfra::TEvent* event, TDataUnit& dataUnit) const override;

    bool FilterFrame(const TDataUnit& dataUnit) const override {
        return dataUnit.contains("DestinationAddress") && dataUnit.contains("FrameMatched");
    }

    void ProcessData(TData& data) const override {
        ProcessDataCore(data, "DestinationAddress");
    }

private:
    TEventProcessors EventProcessors_;
};

REGISTER_VIEWER(empty_response,
               TViewStatisticsEmptyResponse,
               "Print a sorted table of occurences of requested addresses, "
               "responses to which turned out to be empty.")

class TViewContainsEventDNS: public NInfra::NViewLog::TViewContainsEvent {
    void InitializePresetFilters(THashMap<TString, TString>& presetFilters) const override {
        presetFilters = {
            {"gencfg_AAAA_empty_responses", "TBackendRequestData/ResourceType/ANY,TPacketInfo/ResourceType/AAAA,TZoneInfo/Name/gencfg-c.yandex.net,TLookupResult/RecordsNumber/0"}
        };
    }
};

REGISTER_VIEWER(contains_event_dns,
                TViewContainsEventDNS,
                "Regular contains event mode with preset filters (-p).")
} // namespace NYP::DNS::NPrivate
