#include <infra/yp_dns/monitoring/tools/watch_dns_record_sets_monitoring/protos/events.ev.pb.h>

#include <yp/cpp/yp/client.h>
#include <yp/cpp/yp/data_model.h>
#include <yp/cpp/yp/token.h>
#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_printer.h>

#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/http/client/client.h>
#include <library/cpp/http/client/query.h>
#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/generic/map.h>

using namespace NYP::NClient;

static const TDuration TIMEOUT = TDuration::Seconds(2);
static const ui64 CHUNK_SIZE = 25000;

static const TString NOT_EXIST = "<not exist>";

struct TOptions {
    TString YpAddress;
    TString LogPath;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    Y_ENSURE(argc == 3);

    TOptions result;
    result.YpAddress = argv[1];
    result.LogPath = argv[2];
    return result;
}

NMonitoring::TMetricRegistry& SensorsRegistry(const TString& ypAddress = {}) {
    static NMonitoring::TMetricRegistry registry({{"yp_address", ypAddress}});
    return registry;
}

void ResetSensors() {
    for (const TString& sensor : {"equal", "unequal", "checks", "error"}) {
        SensorsRegistry().IntGauge({{"sensor", sensor}})->Set(0);
    }
}

void PushSensors() {
    TString data;
    TStringOutput stream{data};
    {
        auto encoder = NMonitoring::EncoderSpackV1(&stream, NMonitoring::ETimePrecision::SECONDS, NMonitoring::ECompression::IDENTITY);
        SensorsRegistry().Accept(TInstant::Now(), encoder.Get());
    }

    NHttp::TFetchQuery request{
        "http://solomon.yandex.net/push?project=yp_dns&service=watch_monitoring&cluster=watch_monitoring",
        NHttp::TFetchOptions{}.SetPostData(data).SetContentType(TString{NMonitoring::NFormatContenType::SPACK})
    };
    NHttp::Fetch(request);
    ResetSensors();
}

TMap<TString, TDnsRecordSet> SelectAllDnsRecordSets(TClient& client, ui64 timestamp, NInfra::TLogFramePtr frame) {
    TMap<TString, TDnsRecordSet> result;
    TString filter;
    while (true) {
        TSelectObjectsResult selected = client.SelectObjects<TDnsRecordSet>({"/meta/id", "/spec"}, filter, TSelectObjectsOptions{}.SetLimit(CHUNK_SIZE), timestamp).GetValue(TIMEOUT);
        TString lastId;
        for (const TSelectorResult& selectorResult : selected.Results) {
            TDnsRecordSet recordSet;
            selectorResult.Fill(
                recordSet.MutableMeta()->mutable_id(),
                recordSet.MutableSpec()
            );
            TString id = recordSet.Meta().id();
            frame->LogEvent(NEventLog::TSelectedDnsRecordSet(recordSet.Meta().id(), NProtobufJson::Proto2Json(recordSet.Spec())));
            result.emplace(id, std::move(recordSet));
            lastId = std::move(id);
        }

        if (selected.Results.size() < CHUNK_SIZE) {
            break;
        } else {
            filter = TStringBuilder() << "[/meta/id] > \"" << lastId << "\"";
        }
    }
    return result;
}

TVector<TDnsRecordSet> SelectObjects(TClient& client, const THashSet<TString>& updated, ui64 timestamp) {
    TVector<TDnsRecordSet> result;
    result.reserve(updated.size());
    for (const TString& id : updated) {
        TSelectorResult selectorResult = client.GetObject<TDnsRecordSet>(id, {"/meta/id", "/spec"}, timestamp).GetValue(TIMEOUT);
        TDnsRecordSet recordSet;
        selectorResult.Fill(
            recordSet.MutableMeta()->mutable_id(),
            recordSet.MutableSpec()
        );
        result.push_back(std::move(recordSet));
    }
    return result;
}

TWatchObjectsResult WatchObjects(TClient& client, ui64 startTimestamp, ui64 timestamp, NInfra::TLogFramePtr frame) {
    TWatchObjectsOptions opts = TWatchObjectsOptions().SetStartTimestamp(startTimestamp)
                                                      .SetTimestamp(timestamp)
                                                      .SetTimeLimit(TIMEOUT)
                                                      .SetEventCountLimit(CHUNK_SIZE);
    TWatchObjectsResult result = client.WatchObjects<TDnsRecordSet>(opts).GetValue(TIMEOUT);
    frame->LogEvent(NEventLog::TWatchObjectResultInfo(result.Events.size(), result.Timestamp, result.ContinuationToken));
    Y_ENSURE(result.Events.size() < CHUNK_SIZE, "Events number equal to event_count_limit");

    return result;
}

void ApplyEvents(TClient& client, TMap<TString, TDnsRecordSet>& records, const TWatchObjectsResult& events, ui64 timestamp, NInfra::TLogFramePtr frame) {
    THashSet<TString> updatedIds;
    THashSet<TString> removedIds;
    for (const auto& event : events.Events) {
        frame->LogEvent(NEventLog::TDnsRecordSetEvent(event.object_id(), NApi::NProto::EEventType_Name(event.event_type()), event.timestamp()));
        switch (event.event_type()) {
            case NApi::NProto::ET_OBJECT_CREATED:
            case NApi::NProto::ET_OBJECT_UPDATED:
                removedIds.erase(event.object_id());
                updatedIds.insert(event.object_id());
                break;
            case NApi::NProto::ET_OBJECT_REMOVED:
                updatedIds.erase(event.object_id());
                removedIds.insert(event.object_id());
                break;
            default:
                break;
        }
    }

    for (const TString& id : removedIds) {
        records.erase(id);
    }

    TVector<TDnsRecordSet> updated = SelectObjects(client, updatedIds, timestamp);
    for (TDnsRecordSet& recordSet : updated) {
        frame->LogEvent(NEventLog::TUpdatedDnsRecordSet(recordSet.Meta().id(), NProtobufJson::Proto2Json(recordSet.Spec())));
        records[recordSet.Meta().id()] = std::move(recordSet);
    }
}

void LogDiff(const TMap<TString, TDnsRecordSet>& lhs, const TMap<TString, TDnsRecordSet>& rhs, NInfra::TLogFramePtr frame) {
    auto lhsIt = lhs.begin();
    auto rhsIt = rhs.begin();
    while (lhsIt != lhs.end() || rhsIt != rhs.end()) {
        if (lhsIt == lhs.end()) {
            frame->LogEvent(NEventLog::TDiff(rhsIt->first, NOT_EXIST, NProtobufJson::Proto2Json(rhsIt->second.Spec())));
            ++rhsIt;
        } else if (rhsIt == rhs.end()) {
            frame->LogEvent(NEventLog::TDiff(lhsIt->first, NProtobufJson::Proto2Json(lhsIt->second.Spec()), NOT_EXIST));
            ++lhsIt;
        } else {
            if (lhsIt->first < rhsIt->first) {
                frame->LogEvent(NEventLog::TDiff(lhsIt->first, NProtobufJson::Proto2Json(lhsIt->second.Spec()), NOT_EXIST));
                ++lhsIt;
            } else if (lhsIt->first > rhsIt->first) {
                frame->LogEvent(NEventLog::TDiff(rhsIt->first, NOT_EXIST, NProtobufJson::Proto2Json(rhsIt->second.Spec())));
                ++rhsIt;
            } else {
                if (!(lhsIt->second == rhsIt->second)) {
                    frame->LogEvent(NEventLog::TDiff(lhsIt->first, NProtobufJson::Proto2Json(lhsIt->second.Spec()), NProtobufJson::Proto2Json(rhsIt->second.Spec())));
                }
                ++lhsIt;
                ++rhsIt;
            }
        }
    }
}

void Run(TClient& client, NInfra::TLogger& logger) {
    ui64 startTimestamp = client.GenerateTimestamp().GetValue(TIMEOUT);
    TMap<TString, TDnsRecordSet> baseRecords = SelectAllDnsRecordSets(client, startTimestamp, logger.SpawnFrame());

    while (true) {
        Sleep(TDuration::Seconds(10));

        NInfra::TLogFramePtr frame = logger.SpawnFrame();

        ui64 nextTimestamp = client.GenerateTimestamp().GetValue(TIMEOUT);
        SensorsRegistry().IntGauge({{"sensor", "checks"}})->Set(1);
        frame->LogEvent(NEventLog::TStartCheck::FromFields(startTimestamp, nextTimestamp));

        TWatchObjectsResult events = WatchObjects(client, startTimestamp, nextTimestamp, frame);
        ApplyEvents(client, baseRecords, events, nextTimestamp, frame);
        TMap<TString, TDnsRecordSet> selected = SelectAllDnsRecordSets(client, nextTimestamp, frame);
        if (baseRecords != selected) {
            SensorsRegistry().IntGauge({{"sensor", "unequal"}})->Set(1);
            frame->LogEvent(NEventLog::TNotEqual());
            LogDiff(selected, baseRecords, frame);
        } else {
            SensorsRegistry().IntGauge({{"sensor", "equal"}})->Set(1);
            frame->LogEvent(NEventLog::TEqual());
        }
        startTimestamp = nextTimestamp;
        PushSensors();
    }
}

int Run(int argc, const char* argv[]) {
    TOptions opts = ParseOptions(argc, argv);
    TClient client(TClientOptions().SetToken(FindToken()).SetAddress(opts.YpAddress));

    NInfra::TLoggerConfig loggerConfig;
    loggerConfig.SetPath(opts.LogPath);
    loggerConfig.SetRotatePath(opts.LogPath + ".PREV");
    loggerConfig.SetLevel("INFO");
    loggerConfig.SetQueueSize(1024 * 1024);
    loggerConfig.SetMaxLogSizeBytes(1024ull * 1024 * 1024 * 4);
    NInfra::TLogger logger(loggerConfig);

    SensorsRegistry(opts.YpAddress);

    try {
        ResetSensors();
        Run(client, logger);
    } catch (...) {
        SensorsRegistry().IntGauge({{"sensor", "error"}})->Set(1);
        PushSensors();
        logger.SpawnFrame()->LogEvent(NEventLog::TErrorMessage(CurrentExceptionMessage()));
        return 1;
    }

    return 0;
}

int main(int argc, const char* argv[]) {
    TModChooser modChooser;
    modChooser.AddMode("run", Run, "Run");
    modChooser.AddMode("print_log", NInfra::PrintEventLog, "Print log");

    try {
        return modChooser.Run(argc, argv);
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
    return 0;
}
