import os
import requests
import retry
import socket

from requests.packages.urllib3 import Retry
from yp.client import YpClient, find_token


def get_retry_session(retries=5, backoff_factor=2, status_forcelist=(500, 502, 503, 504)):
    session = requests.Session()
    adapter = requests.adapters.HTTPAdapter(max_retries=Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
    ))
    session.mount('http://', adapter)
    return session


def get_current_clusters():
    clusters = os.getenv('YP_MONITORING_CLUSTERS')
    if clusters is not None:
        return clusters.split(',')

    return [os.uname()[1].split('.')[1]]


def get_address(hostname):
    return socket.getaddrinfo(hostname, 53, proto=socket.IPPROTO_UDP)[0][4][0]


def get_yp_client(address):
    if not hasattr(get_yp_client, 'clients'):
        get_yp_client.clients = {}
    if address not in get_yp_client.clients:
        get_yp_client.clients[address] = YpClient(address=address, config={
            'token': find_token(),
            'request_timeout': 10 * 1000,
        })
    return get_yp_client.clients[address]


@retry.retry(tries=3, delay=2, backoff=2)
def watch_objects(yp_client, start_timestamp):
    return yp_client.watch_objects('dns_record_set', start_timestamp=start_timestamp)


@retry.retry(tries=3, delay=2, backoff=2)
def select_objects(yp_client, selectors, limit, continuation_token):
    return yp_client.select_objects('dns_record_set',
                                    selectors=selectors,
                                    limit=limit,
                                    options={'continuation_token': continuation_token},
                                    enable_structured_response=True)


@retry.retry(tries=3, delay=2, backoff=2)
def get_object(yp_client, id, selectors):
    return yp_client.get_object('dns_record_set',
                                object_identity=id,
                                enable_structured_response=True,
                                selectors=selectors,
                                options={'ignore_nonexistent': True})
