import argparse
import itertools
import json
import os

import dns.name

from library.python.find_root import detect_root


YP_DNS_CONFIG_PATH = 'infra/yp_dns/config/config.json'
EXPORT_SERVICE_CONFIG_PATH = 'infra/yp_yandex_dns_export/config/config.json'
BRIDGE_CONFIG_PATH = 'infra/yp_dns_api/config/bridge_config.json'
REPLICATOR_CONFIG_PATH = 'infra/yp_dns_api/config/replicator_config.json'

YP_CLUSTERS = [
    'sas-test',
    'man-pre',
    'sas',
    'man',
    'vla',
    'myt',
    'iva',
    'xdc',
]

# zone types
SLAYER_REVERSE = 'slayer_reverse'

ZONE_TYPES = [
    SLAYER_REVERSE,
]


def load_config(path):
    return json.load(open(path))


def save_config(config, path):
    with open(path, 'w') as f:
        json.dump(config, f, indent=4)
        f.write('\n')


def add_zone_to_export_service_config(config_path, zones):
    config = load_config(config_path)

    presented_zones = set(dns.name.from_text(zone['Name']) for zone in config['ExportConfig']['Zones'])

    for zone, ztype, clusters in zones:
        if dns.name.from_text(zone) in presented_zones:
            continue

        if ztype == SLAYER_REVERSE:
            if not zone.endswith('.'):
                zone += '.'

            source_name = 'slayer_{}'.format(zone)
            config['ExportConfig']['Sources'].append({
                'Name': source_name,
                'Receive': {
                    'SlayerReverse': {
                        'Zone': zone,
                    }
                },
            })

            config['ExportConfig']['Zones'].append({
                'Name': zone,
                'Sources': [
                    source_name,
                ],
            })

    save_config(config, config_path)


def add_zone_to_yp_dns_config(config_path, zones):
    config = load_config(config_path)

    presented_zones = set(dns.name.from_text(zone['Name']) for zone in config['Zones'])

    for zone, ztype, clusters in zones:
        if dns.name.from_text(zone) in presented_zones:
            continue

        if zone.endswith('.'):
            zone = zone[:-1]

        if ztype == SLAYER_REVERSE:
            config['Zones'].append({
                'Name': zone,
                'PrimaryNameserver': 'ns1.yp-dns.yandex.net',
                'Nameservers': [
                    'ns1.yp-dns.yandex.net',
                    'ns2.yp-dns.yandex.net'
                ],
                'IsAuthoritative': True,
                'SupportPTR': True,
                'YPClusters': clusters,
            })

    save_config(config, config_path)


def add_zone_to_bridge_config(config_path, zones):
    config = load_config(config_path)

    presented_zones = set(dns.name.from_text(zone['Name']) for zone in config['ZoneConfigs'])

    presented_static_zones = set(
        dns.name.from_text(zone['Name'])
        for zone in config['DynamicZonesConfig']['ZonesManagerServiceConfig']['ZonesManagerConfig']['StaticZones']['Zones']
    )

    for zone, ztype, clusters in zones:
        if zone.endswith('.'):
            zone = zone[:-1]

        dns_zone_name = dns.name.from_text(zone)

        if ztype == SLAYER_REVERSE:
            if dns_zone_name not in presented_zones:
                config['ZoneConfigs'].append({
                    'Name': zone,
                    'Clusters': clusters,
                    'Owners': [
                        'robot-ypdnsapi',
                        'robot-ypdnsapi-repl',
                    ],
                    'WriteToChangelist': True,
                })

            if dns_zone_name not in presented_static_zones:
                config['DynamicZonesConfig']['ZonesManagerServiceConfig']['ZonesManagerConfig']['StaticZones']['Zones'].append({
                    'Name': zone,
                    'PrimaryNameserver': 'ns1.yp-dns.yandex.net',
                    'Nameservers': [
                        'ns1.yp-dns.yandex.net',
                        'ns2.yp-dns.yandex.net'
                    ],
                    'SupportPTR': True,
                    'YPClusters': clusters,
                    'SelectRecordSetMode': 'MERGE',
                })

    save_config(config, config_path)


def add_zone_to_replicator_config(config_path, zones):
    config = load_config(config_path)

    presented_zones = set()
    reverse_ip6_group_idx = None
    for idx, group_config in enumerate(config['ZonesReplicationConfigs']):
        if group_config['GroupName'] == 'reverse-ip6':
            reverse_ip6_group_idx = idx

        for zone_repl_config in group_config['ZoneReplicatorConfigs']:
            presented_zones.add(dns.name.from_text(zone_repl_config['Zone']))

    if reverse_ip6_group_idx is None:
        config['ZonesReplicationConfigs'].append({
            'GroupName': 'reverse-ip6',
            'ClusterConfigs': [],
            'ZoneReplicatorConfigs': [],
        })
        reverse_ip6_group_idx = -1

    reverse_group = config['ZonesReplicationConfigs'][reverse_ip6_group_idx]
    presented_in_ip6_group_clusters = set(map(lambda cfg: cfg['ClusterName'], reverse_group['ClusterConfigs']))

    for zone, ztype, clusters in zones:
        if dns.name.from_text(zone) in presented_zones:
            continue

        if zone.endswith('.'):
            zone = zone[:-1]

        if ztype == SLAYER_REVERSE:
            for cluster in clusters:
                if cluster not in presented_in_ip6_group_clusters:
                    reverse_group['ClusterConfigs'].append({
                        'ClusterName': cluster,
                        'Address': '{}.yp.yandex.net:8090'.format(cluster),
                        'ReadOnlyMode': False,
                        'Timeout': '30s',
                    })
                    presented_in_ip6_group_clusters.add(cluster)

            reverse_group['ZoneReplicatorConfigs'].append({
                'Zone': zone,
                'Clusters': clusters,
            })

    save_config(config, config_path)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-z', '--zone', required=True, nargs='+',
                        help='List of zones to add')
    parser.add_argument('-t', '--type', required=True, choices=ZONE_TYPES,
                        help='Type of zone')
    parser.add_argument('-c', '--yp-dns-clusters', required=True, nargs='+', choices=YP_CLUSTERS,
                        help='List of YP clusters where the zone will be used in YP DNS')
    parser.add_argument('-a', '--dns-api-clusters', required=True, nargs='+', choices=YP_CLUSTERS,
                        help='List of YP clusters where the zone will be stored by YP DNS API')
    return parser.parse_args()


def main():
    args = parse_args()

    arcadia_root_path = detect_root(os.getcwd())

    zones = list(itertools.product(args.zone, [args.type], [args.yp_dns_clusters]))
    yp_dns_config_path = os.path.join(arcadia_root_path, YP_DNS_CONFIG_PATH)
    add_zone_to_yp_dns_config(yp_dns_config_path, zones)

    export_service_config_path = os.path.join(arcadia_root_path, EXPORT_SERVICE_CONFIG_PATH)
    add_zone_to_export_service_config(export_service_config_path, zones)

    zones = list(itertools.product(args.zone, [args.type], [args.dns_api_clusters]))
    bridge_config_path = os.path.join(arcadia_root_path, BRIDGE_CONFIG_PATH)
    add_zone_to_bridge_config(bridge_config_path, zones)

    replicator_config_path = os.path.join(arcadia_root_path, REPLICATOR_CONFIG_PATH)
    add_zone_to_replicator_config(replicator_config_path, zones)


if __name__ == '__main__':
    main()
