#include "apply_zone_snapshot.h"

#include "logging.h"
#include "processing_record_set.h"

#include <infra/libs/yp_dns/record_set/record.h>

#include <library/cpp/retry/retry.h>

#include <util/generic/algorithm.h>
#include <util/generic/map.h>

namespace NYpDns::NApplyZoneSnapshotToYp {

TRecordSet AddZoneToRecordSet(TRecordSet recordSet, const TString& nameZone) {
    recordSet.SetZone(nameZone);
    return recordSet;
}

bool AreRecordSetsEqual(const TRecordSet& firstRecordSet, const TRecordSet& secondRecordSet) {
    if (firstRecordSet.Spec().records().size() != secondRecordSet.Spec().records().size()) {
        return false;
    }
    const DNSName domain(firstRecordSet.Meta().id());
    TVector<TRecord> firstRecordReserve;
    firstRecordReserve.reserve(firstRecordSet.Spec().records_size());
    TVector<TRecord> secondRecordReserve;
    secondRecordReserve.reserve(secondRecordSet.Spec().records_size());
    for (const auto& record : firstRecordSet.Spec().records()) {
        firstRecordReserve.emplace_back(CreateRecordFromProto(domain, record));
    }
    for (const auto& record : secondRecordSet.Spec().records()) {
        secondRecordReserve.emplace_back(CreateRecordFromProto(domain, record));
    }
    Sort(firstRecordReserve);
    Sort(secondRecordReserve);
    return Equal(firstRecordReserve.begin(), firstRecordReserve.end(), secondRecordReserve.begin());
}

void CreateRequest(TLazyCreateRequestsApplier& lazyCreateRequestsApplier,
                   const TRecordSet& recordSet,
                   const TApplyZoneSnapshotOptions& options)
{
    if (!options.ChangeNSAndSOA) {
        TRecordSet recordSetWithoutNSAndSOA = SplitRecordSet(
            recordSet, {
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::SOA,
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::NS
            }
        ).first;
        if (!recordSetWithoutNSAndSOA.Spec().records().empty()) {
            PrintLogCreateRequest(recordSetWithoutNSAndSOA);
            lazyCreateRequestsApplier.AddRequest(AddZoneToRecordSet(recordSetWithoutNSAndSOA, options.NameZone));
        }
    } else {
        PrintLogCreateRequest(recordSet);
        lazyCreateRequestsApplier.AddRequest(AddZoneToRecordSet(recordSet, options.NameZone));
    }
}

void UpdateRequest(TLazyUpdateRequestsApplier& lazyUpdateRequestsApplier,
                   const TRecordSet& before, 
                   const TRecordSet& after,
                   const TApplyZoneSnapshotOptions& options) 
{
    if (!options.ChangeNSAndSOA) {
        auto [beforeWithoutNSAndSOA, beforeWithNSAndSOA] = SplitRecordSet(
            before, {
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::SOA,
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::NS
            }
        );
        TRecordSet afterWithoutNSAndSOA = SplitRecordSet(
            after, {
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::SOA,
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::NS
            }
        ).first;
        if (!AreRecordSetsEqual(beforeWithoutNSAndSOA, afterWithoutNSAndSOA)) {
            TRecordSet finalRecordSet = MergeRecordSet(beforeWithNSAndSOA, afterWithoutNSAndSOA);
            if (finalRecordSet.Spec().records_size()) {
                PrintLogUpdateRequest(before.Meta().id(), before, finalRecordSet);
                lazyUpdateRequestsApplier.AddRequest(before.Meta().id(), {NYP::NClient::TSetRequest("/spec", finalRecordSet.Spec())}, {});
            }
        }
    } else {
        if (!AreRecordSetsEqual(before, after)) {
            PrintLogUpdateRequest(before.Meta().id(), before, after);
            lazyUpdateRequestsApplier.AddRequest(before.Meta().id(), {NYP::NClient::TSetRequest("/spec", after.Spec())}, {});
        }
    }
}

void RemoveRequest(TLazyRemoveRequestsApplier& lazyRemoveRequestsApplier, 
                   TLazyUpdateRequestsApplier& lazyUpdateRequestsApplier,
                   const TRecordSet& recordsSet,
                   const TApplyZoneSnapshotOptions& options) 
{
    if (!options.ChangeNSAndSOA) {
        TRecordSet recordSetWithNSAndSOA = SplitRecordSet(
            recordsSet, {
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::SOA,
                NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::NS
            }
        ).second;
        if (recordSetWithNSAndSOA.Spec().records_size()) {
            PrintLogUpdateRequest(recordsSet.Meta().id(), recordsSet, recordSetWithNSAndSOA);
            lazyUpdateRequestsApplier.AddRequest(recordsSet.Meta().id(), {NYP::NClient::TSetRequest("/spec", recordSetWithNSAndSOA.Spec())}, {});
            return;
        }
    }
    PrintLogRemoveRequest(recordsSet.Meta().id());
    lazyRemoveRequestsApplier.AddRequest(recordsSet.Meta().id());
}

void ApplyZoneSnapshot(const TMap<TString, TRecordSet>& recordSetsFromZonefile, 
                       const TMap<TString, TRecordSet>& recordSetsFromYp,
                       const TApplyZoneSnapshotOptions& options)
{
    TLazyCreateRequestsApplier lazyCreateRequestsApplier(options.Factory, options.ApplierOptions);
    TLazyUpdateRequestsApplier lazyUpdateRequestsApplier(options.Factory, options.ApplierOptions);
    TLazyRemoveRequestsApplier lazyRemoveRequestsApplier(options.Factory, options.ApplierOptions);

    auto iterZonefile = recordSetsFromZonefile.begin();
    auto iterYp = recordSetsFromYp.begin();

    INFO_LOG << "Start processing differences between record sets from zonefile and record sets from YP-client" << Endl;
    try {
        while (iterZonefile != recordSetsFromZonefile.end() || iterYp != recordSetsFromYp.end()) {
            if (iterYp == recordSetsFromYp.end()) {
                CreateRequest(lazyCreateRequestsApplier, iterZonefile->second, options);
                ++iterZonefile;
            } else if (iterZonefile == recordSetsFromZonefile.end()) {
                RemoveRequest(lazyRemoveRequestsApplier, lazyUpdateRequestsApplier, iterYp->second, options);
                ++iterYp;
            } else if (iterYp->first < iterZonefile->first) {
                RemoveRequest(lazyRemoveRequestsApplier, lazyUpdateRequestsApplier, iterYp->second, options);
                ++iterYp;
            } else if (iterYp->first > iterZonefile->first) {
                CreateRequest(lazyCreateRequestsApplier, iterZonefile->second, options);
                ++iterZonefile;
            } else {
                UpdateRequest(lazyUpdateRequestsApplier, iterYp->second, iterZonefile->second, options);
                ++iterYp;
                ++iterZonefile;
            }
        }
        lazyCreateRequestsApplier.Finish();
        lazyUpdateRequestsApplier.Finish();
        lazyRemoveRequestsApplier.Finish();
    } catch (...) {
        ERROR_LOG << "Error when writing: " << CurrentExceptionMessage() << Endl;
        throw;
    }
    INFO_LOG << "Done. All differences have been processed" << Endl;
}

} // NYpDns::NApplyZoneSnapshotToYp
