#include "lazy_requests_applier.h"

#include <library/cpp/logger/global/global.h>

namespace NYpDns::NApplyZoneSnapshotToYp {

NYP::NClient::TTransactionPtr ILazyRequestsApplier::CreateTransaction() {
    INFO_LOG << "Creating a transaction" << Endl;
    try {
        NYP::NClient::TTransactionPtr transaction = *DoWithRetry<NYP::NClient::TTransactionPtr, NYP::NClient::TResponseError> (
            [this] { return this->TransactionFactory_.CreateTransaction(); },
            Options_.RetryOptions,
            /* throwLast */ true
        );
        INFO_LOG << "The transaction was created with the id: " << transaction->TransactionId() << Endl;
        return transaction;
    } catch (...) {
        ERROR_LOG << "Transaction creation error: " << CurrentExceptionMessage() << Endl;
        throw;
    }
}

void ILazyRequestsApplier::CommitTransaction(NYP::NClient::TTransactionPtr transaction) {
    INFO_LOG << "Commiting a transaction with the id: " << transaction->TransactionId() << Endl;
    try {
        DoWithRetry<NYP::NClient::TResponseError> (
            [&transaction] { transaction->Commit().GetValue(transaction->ClientOptions().Timeout() * 2); },
            Options_.RetryOptions,
            /* throwLast */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Transaction commit error: " << CurrentExceptionMessage() << Endl;
        throw;
    }
}

void ILazyRequestsApplier::Finish() {
    if (CurrentNumberOfRequests_) {
        Push();
    }
    if (Eptr_) {
        std::rethrow_exception(Eptr_);
    }
}

void TLazyCreateRequestsApplier::AddRequest(const TRecordSet& recordSet) {
    Requests_.emplace_back(NYP::NClient::TCreateObjectRequest(recordSet.MakeYpObject()));
    ILazyRequestsApplier::AddRequest();
}

NYP::NClient::TTransactionPtr TLazyCreateRequestsApplier::ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) {
    try {
        INFO_LOG << "Applying creation requests to the transaction with the id: " << transaction->TransactionId() << Endl;
        DoWithRetry<NYP::NClient::TResponseError>(
            [this, &transaction] {
                transaction->CreateObjects(this->Requests_).GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            this->Options_.RetryOptions,
            /* throwLast */ true
        );
        INFO_LOG << "Done. Applied " << Requests_.size() << " requests" << Endl;
    } catch (...) {
        ERROR_LOG << "Error applying creation requests: " << CurrentExceptionMessage() << Endl;
        Eptr_ = std::current_exception();
    }
    Requests_.clear();
    return transaction;
}

void TLazyUpdateRequestsApplier::AddRequest(const TString& objectId, 
                const TVector<NYP::NClient::TSetRequest>& updateVec, 
                const TVector<NYP::NClient::TRemoveRequest>& removeVec) 
{
    Requests_.emplace_back(NYP::NClient::TUpdateRequest(NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET, objectId, updateVec, removeVec));
    ILazyRequestsApplier::AddRequest();
}

NYP::NClient::TTransactionPtr TLazyUpdateRequestsApplier::ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) {
    try {
        INFO_LOG << "Applying update requests to the transaction with the id: " << transaction->TransactionId() << Endl;
        DoWithRetry<NYP::NClient::TResponseError>(
            [this, &transaction] {
                transaction->UpdateObjects(this->Requests_).GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            this->Options_.RetryOptions,
            /* throwLast */ true
        );
        INFO_LOG << "Done. Applied " << Requests_.size() << " requests" << Endl;
    } catch (...) {
        ERROR_LOG << "Error applying update requests: " << CurrentExceptionMessage() << Endl;
        Eptr_ = std::current_exception();
    }
    Requests_.clear();
    return transaction;
}

void TLazyRemoveRequestsApplier::AddRequest(const TString& objectId) {
    Requests_.emplace_back(NYP::NClient::TRemoveObjectRequest(NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET, objectId));
    ILazyRequestsApplier::AddRequest();
}

NYP::NClient::TTransactionPtr TLazyRemoveRequestsApplier::ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) {
    try {
        INFO_LOG << "Applying remove requests to the transaction with the id: " << transaction->TransactionId() << Endl;
        DoWithRetry<NYP::NClient::TResponseError>(
            [this, &transaction] {
                transaction->RemoveObjects(this->Requests_).GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            this->Options_.RetryOptions,
            /* throwLast */ true
        );
        INFO_LOG << "Done. Applied " << Requests_.size() << " requests" << Endl;
    } catch (...) {
        ERROR_LOG << "Error applying remove requests: " << CurrentExceptionMessage() << Endl;
        Eptr_ = std::current_exception();
    }
    Requests_.clear();
    return transaction;
}

} // NYpDns::NApplyZoneSnapshotToYp
