#pragma once

#include <infra/libs/yp_dns/record_set/record_set.h>

#include <yp/cpp/yp/client.h>

#include <library/cpp/retry/retry.h>

#include <exception>

namespace NYpDns::NApplyZoneSnapshotToYp {

struct TLazyRequestsApplierOptions {
    TLazyRequestsApplierOptions(TRetryOptions options)
        : RetryOptions(std::move(options))
    {
    }

    TLazyRequestsApplierOptions& SetBufferSize(ui64 bufferSize) {
        BufferSize = bufferSize;
        return *this;
    }

    TRetryOptions RetryOptions;
    ui64 BufferSize = 500;
};

class ILazyRequestsApplier {
public:
    ILazyRequestsApplier(NYP::NClient::TTransactionFactory& factory, const TLazyRequestsApplierOptions& options)
        : TransactionFactory_(factory)
        , Options_(options)
    {
    }

    void Finish();

private:
    NYP::NClient::TTransactionPtr CreateTransaction();

    void CommitTransaction(NYP::NClient::TTransactionPtr transaction);

protected:
    virtual NYP::NClient::TTransactionPtr ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) = 0;

    void Push() {
        try {
            CommitTransaction(ApplyRequestsToTransaction(CreateTransaction()));
        } catch (...) {
            Eptr_ = std::current_exception();
        }
        CurrentNumberOfRequests_ = 0;
    }

    void AddRequest() {
        ++CurrentNumberOfRequests_;
        if (CurrentNumberOfRequests_ == Options_.BufferSize) {
            Push();
        }
    }

protected:
    NYP::NClient::TTransactionFactory& TransactionFactory_;
    const TLazyRequestsApplierOptions& Options_;
    ui64 CurrentNumberOfRequests_ = 0;
    std::exception_ptr Eptr_;
};

class TLazyCreateRequestsApplier : public ILazyRequestsApplier {
public:
    using ILazyRequestsApplier::ILazyRequestsApplier;
    
    void AddRequest(const TRecordSet& recordSet);

private:
    NYP::NClient::TTransactionPtr ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) override;

private:
    TVector<NYP::NClient::TCreateObjectRequest> Requests_;
};

class TLazyUpdateRequestsApplier : public ILazyRequestsApplier {
public:
    using ILazyRequestsApplier::ILazyRequestsApplier;

    void AddRequest(const TString& objectId,
                    const TVector<NYP::NClient::TSetRequest>& updateVec, 
                    const TVector<NYP::NClient::TRemoveRequest>& removeVec);

private:
    NYP::NClient::TTransactionPtr ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) override;

private:
    TVector<NYP::NClient::TUpdateRequest> Requests_;
};

class TLazyRemoveRequestsApplier : public ILazyRequestsApplier {
public:
    using ILazyRequestsApplier::ILazyRequestsApplier;

    void AddRequest(const TString& objectId);

private:
    NYP::NClient::TTransactionPtr ApplyRequestsToTransaction(NYP::NClient::TTransactionPtr transaction) override;

private:
    TVector<NYP::NClient::TRemoveObjectRequest> Requests_;
};

} // NYpDns::NApplyZoneSnapshotToYp
