#include "readers.h"

#include "logging.h"

#include <infra/libs/yp_dns/record_set/record.h>

#include <yp/cpp/yp/token.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/retry/retry.h>

#include <util/datetime/cputimer.h>

namespace NYpDns::NApplyZoneSnapshotToYp {

TMap<TString, TRecordSet> GetRecordSetsFromZonefile(IZoneReader& reader) {
    TMap<TString, TRecordSet> recordSets;
    
    TRecord recordFromZonefile;
    INFO_LOG << "Load record sets from zonefile" << Endl;
    try {
        while (reader.ReadRecord(recordFromZonefile)) {
            const TString recordName(recordFromZonefile.Name.toStringNoDot());
            TRecordSet& recordSet = recordSets[recordName];
            *recordSet.MutableSpec()->add_records() = CreateProtoFromRecord(recordFromZonefile);
            if (recordSet.Meta().id().empty()) {
                recordSet.MutableMeta()->set_id(recordName);
            }
        }
    } catch (...) {
        ERROR_LOG << "Reader error: " << CurrentExceptionMessage() << Endl;
        throw;
    }
    INFO_LOG << "Done" << Endl;
    return recordSets;
}

TMap<TString, TRecordSet> GetRecordSetsFromYp(NYP::NClient::TClient& client, const ui64 timestamp, TStringBuf zoneName) {
    TMap<TString, TRecordSet> recordSets;
    zoneName.ChopSuffix(".");

    const TRetryOptions retryOptions = TRetryOptions()
        .WithCount(3)
        .WithSleep(TDuration::Seconds(10))
        .WithIncrement(TDuration::Seconds(5));

    const TString filter = TString::Join("regex_partial_match(\"(^|\\.)", zoneName, "\\.?$\", [/meta/id]) = %true");
    TMaybe<TString> continuationToken;
    const ui64 limitInRequest = 5000;
    
    INFO_LOG << "Load record sets from YP-client" << " with timestamp: " << timestamp << Endl;
    for (int batch_number = 1;; ++batch_number) {
        NYP::NClient::TSelectObjectsOptions selectObjectOptions;
        selectObjectOptions.SetLimit(limitInRequest);

        if (continuationToken.Defined()) {
            selectObjectOptions.SetContinuationToken(*continuationToken);
        }
        INFO_LOG << "Load batch №" << batch_number << Endl;
        NYP::NClient::TSelectObjectsResult result;
        try {
            result = *DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
                [&client, &filter, &selectObjectOptions, timestamp] {
                    return client.SelectObjects<NYP::NClient::TDnsRecordSet>(
                        {"/meta/id", "/spec"},
                        filter,
                        selectObjectOptions,
                        timestamp
                    ).GetValue(client.Options().Timeout() * 2);
                },
                retryOptions,
                /* throwLast */ true
            );
        } catch (...) {
            ERROR_LOG << "YP-client error: " << CurrentExceptionMessage() << Endl;
            throw;
        }

        for (auto recordSetFromRequest : result.Results) {
            TRecordSet recordSet;
            recordSetFromRequest.Fill(recordSet.MutableMeta()->mutable_id(), recordSet.MutableSpec());
            PrintLogSelectObject(recordSet);
            TString recordSetId = recordSet.Meta().id();
            recordSets.emplace(std::move(recordSetId), std::move(recordSet));
        }

        INFO_LOG << "Done. Batch №" << batch_number << " with the size of " << result.Results.size() << " has been uploaded" << Endl;

        if (result.Results.size() == limitInRequest) {
            continuationToken = result.ContinuationToken;
        } else {
            break;
        }
    }
    INFO_LOG << "Done. All record sets are received" << Endl;
    return recordSets;
}

} // NYpDns::NApplyZoneSnapshotToYp
