#include "run_tool.h"

#include "apply_zone_snapshot.h"
#include "readers.h"
#include "lazy_requests_applier.h"

#include <library/cpp/getopt/small/last_getopt.h>

#include <yp/cpp/yp/token.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/retry/retry.h>

namespace NYpDns::NApplyZoneSnapshotToYp {

TOptions ParseOptions(int argc, const char* argv[]) {
    TOptions result;

    NLastGetopt::TOpts opts;

    opts.AddHelpOption();

    opts
        .AddLongOption("zone", "Name of the zone")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.ZoneName);

    opts
        .AddLongOption("zonefile-path", "Path to the zonefile")
        .Required()
        .RequiredArgument("PATH")
        .StoreResult(&result.ZoneFilePath);

    opts
        .AddLongOption("yp-cluster", "Name of the YP cluster")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.YpCluster);
    
    opts
        .AddLongOption("read-only", "Read-only mode")
        .Optional()
        .NoArgument()
        .StoreTrue(&result.ReadOnlyMode);

    opts
        .AddLongOption("change-ns-soa", "Changing NS and SOA records")
        .Optional()
        .NoArgument()
        .StoreTrue(&result.ChangeNSAndSOA);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

void RunApplyZoneSnapshotToYp(const TOptions& options) {
    const NYP::NClient::TClientOptions clientOptions = NYP::NClient::TClientOptions()
        .SetAddress(options.YpCluster)
        .SetEnableSsl(true)
        .SetEnableBalancing(true)
        .SetTimeout(TDuration::Seconds(15))
        .SetToken(NYP::NClient::FindToken())
        .SetReadOnlyMode(options.ReadOnlyMode);
    NYP::NClient::TClient client(clientOptions);

    const TRetryOptions retryOptions = TRetryOptions()
        .WithCount(3)
        .WithSleep(TDuration::Seconds(30))
        .WithIncrement(TDuration::Seconds(15));

    TZoneFileReader fileReader(options.ZoneFilePath);
    const TMap<TString, TRecordSet> recordSetsFromZonefile = GetRecordSetsFromZonefile(fileReader);

    const TRetryOptions retryOptionsForApplier = TRetryOptions()
        .WithCount(3)
        .WithSleep(TDuration::Seconds(5))
        .WithIncrement(TDuration::Seconds(3));

    TLazyRequestsApplierOptions applierOptions(retryOptionsForApplier);

    DoWithRetry<NYP::NClient::TResponseError>(
        [&client, &retryOptions, &applierOptions, &options, &recordSetsFromZonefile] {
            INFO_LOG << "Getting a timestamp" << Endl;
            ui64 timestamp;
            try {
                timestamp = *DoWithRetry<ui64, NYP::NClient::TResponseError>(
                    [&client] { return client.GenerateTimestamp().GetValue(client.Options().Timeout() * 2); },
                    retryOptions,
                    /* throwLast */ true
                );
            } catch (...) {
                ERROR_LOG << "Error getting timestamp: " << CurrentExceptionMessage() << Endl;
                throw;
            }
            INFO_LOG << "Done. Timestamp: " << timestamp << Endl;

            const NYP::NClient::TTransactionOptions transactionOptions = NYP::NClient::TTransactionOptions()
                .SetStartTimestamp(timestamp);
            NYP::NClient::TTransactionFactory factory(client, transactionOptions);
            const TApplyZoneSnapshotOptions applySnapshotOptions(factory, applierOptions, options.ZoneName, options.ChangeNSAndSOA);
 
            ApplyZoneSnapshot(recordSetsFromZonefile,
                 GetRecordSetsFromYp(client, timestamp, options.ZoneName),
                 applySnapshotOptions
            );
        },
        retryOptions,
        /* throwLast */ true
    );
}

void RunApplyZoneSnapshotToYp(int argc, const char* argv[]) {
    const TOptions options = ParseOptions(argc, argv);
    RunApplyZoneSnapshotToYp(options);
}

} // NYpDns::NApplyZoneSnapshotToYp
