#include "config_loader.h"

#include <infra/yp_dns/libs/config/protos/config.pb.h>

#include <library/cpp/proto_config/load.h>
#include <library/cpp/protobuf/json/json2proto.h>

#include <util/stream/file.h>

namespace NYpDns::NGenerateZoneFiles {

TVector<TZoneConfig> LoadZoneConfigsFromYpDnsConfig() {
    NProtoConfig::TLoadConfigOptions loadOptions;
    loadOptions.Resource = "/proto_config.json";
    NYP::DNS::TConfig serviceConfig = NProtoConfig::LoadWithOptions<NYP::DNS::TConfig>(loadOptions);
    return TVector<TZoneConfig>{serviceConfig.GetZones().begin(), serviceConfig.GetZones().end()};
}

TVector<TZoneConfig> LoadZoneConfigsFromCustomJsonConfig(const TFsPath& configPath) {
    TUnbufferedFileInput input(configPath);
    const NJson::TJsonValue jsonConfig = NJson::ReadJsonTree(&input, /* throwOnError */ true);

    NJson::TJsonValue jsonZoneConfigs;
    if (jsonConfig.IsArray()) {
        jsonZoneConfigs = jsonConfig;
    } else {
        jsonZoneConfigs = jsonConfig["Zones"];
    }

    Y_ENSURE(jsonZoneConfigs.IsArray(), "Cannot parse config: zone configs list not found");

    TVector<TZoneConfig> result;
    result.reserve(jsonConfig.GetArray().size());

    for (const NJson::TJsonValue& jsonZoneConfig : jsonConfig.GetArray()) {
        TZoneConfig zoneConfig;
        NProtobufJson::Json2Proto(
            jsonZoneConfig,
            zoneConfig,
            NProtobufJson::TJson2ProtoConfig()
                .SetMapAsObject(true)
        );
        result.emplace_back(std::move(zoneConfig));
    }

    return result;
}

TVector<TZoneConfig> LoadZoneConfigs(const TFsPath& configPath) {
    if (!configPath) {
        return LoadZoneConfigsFromYpDnsConfig();
    }

    return LoadZoneConfigsFromCustomJsonConfig(configPath);
}

} // namespace NYpDns::NGenerateZoneFiles
