#include "generate_zonefiles.h"
#include "config_loader.h"
#include "yp_cluster_data.h"
#include "zone.h"

#include <infra/libs/yp_dns/zone/protos/config.pb.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/threading/future/async.h>

#include <util/folder/path.h>
#include <util/system/backtrace.h>
#include <util/thread/pool.h>

namespace NYpDns::NGenerateZoneFiles {

TOptions ParseOptions(int argc, const char* argv[]) {
    TOptions result;

    NLastGetopt::TOpts opts;

    opts.AddHelpOption();

    opts
        .AddLongOption('c', "config", "Path to config with zones description")
        .Optional()
        .RequiredArgument("PATH")
        .StoreResult(&result.ZonesConfigPath);

    opts
        .AddLongOption('o', "output-dir", "Path to directory where to write zonefiles")
        .Optional()
        .RequiredArgument("PATH")
        .DefaultValue(TFsPath::Cwd())
        .StoreResult(&result.OutputDir);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

void GenerateZoneFiles(const TOptions& options) {
    options.OutputDir.MkDirs();

    const TVector<TZoneConfig> zoneConfigs = LoadZoneConfigs(options.ZonesConfigPath);
    THolder<IThreadPool> zonesThreadPool = CreateThreadPool(10);
    TVector<TZone> zones = CreateZones(zoneConfigs, *zonesThreadPool);

    TVector<NThreading::TFuture<void>> futures;
    futures.reserve(zones.size());

    for (TZone& zone : zones) {
        futures.push_back(zone.GenerateZoneFile(options.OutputDir));
    }

    NThreading::WaitAll(futures).GetValueSync();
}

int RunGenerateZoneFiles(int argc, const char *argv[]) {
    const TOptions options = ParseOptions(argc, argv);
    try {
        GenerateZoneFiles(options);
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        Cerr << "Backtrace:" << Endl;
        TBackTrace::FromCurrentException().PrintTo(Cerr);
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}

} // namespace NYpDns::NGenerateZoneFiles