#include "yp_cluster_data.h"

#include <yp/cpp/yp/token.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/retry/retry.h>

#include <util/system/guard.h>

namespace NYpDns::NGenerateZoneFiles {

TYpClusterData::TYpClusterData(TString name, const NYP::NClient::TClientOptions& ypClientOptions, IThreadPool& pool)
    : Name_(std::move(name))
    , YpClient_(NYP::NClient::CreateClient(ypClientOptions))
    , Pool_(pool)
{
}

NThreading::TFuture<TVector<TSerializedRecordSet>> TYpClusterData::RecordSets() {
    TGuard<TMutex> guard(RecordSetsMutex_);
    if (!RecordSets_.Initialized()) {
        RecordSets_ = NThreading::Async([this] { return LoadRecordSets(); }, Pool_);
    }
    return RecordSets_;
}

TVector<TSerializedRecordSet> TYpClusterData::LoadRecordSets() const {
    static const TVector<TString> selectors = {
        "/meta/id",
        "/spec/records",
        "/labels/changelist",
    };

    static const TRetryOptions retryOptions = TRetryOptions()
        .WithCount(10)
        .WithSleep(TDuration::Seconds(1))
        .WithIncrement(TDuration::Seconds(1))
        .WithRandomDelta(TDuration::MilliSeconds(500));

    const ui64 timestamp = *DoWithRetry<ui64, NYP::NClient::TResponseError>(
        [this]{ return YpClient_->GenerateTimestamp().GetValue(YpClient_->Options().Timeout() * 2); },
        retryOptions,
        /* throwLast */ true
    );
    INFO_LOG << "Load record sets from YP-" << to_upper(Name_) << " with timestamp " << timestamp << Endl;

    TVector<TSerializedRecordSet> result;

    NYP::NClient::TSelectObjectsOptions options;
    options.SetLimit(1000);
    while (true) {
        NYP::NClient::TSelectObjectsResult chunkResult = *DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
            [this, &options, timestamp] {
                return YpClient_->SelectObjects<NYP::NClient::TDnsRecordSet>(
                    selectors,
                    /* filter */ {},
                    options,
                    timestamp
                ).GetValue(YpClient_->Options().Timeout() * 2);
            },
            retryOptions,
            /* throwLast */ true
        );

        for (const NYP::NClient::TSelectorResult& selectorResult : chunkResult.Results) {
            TRecordSet recordSet;
            recordSet.SetYpTimestamp(timestamp);

            NJson::TJsonValue changelistJson;
            selectorResult.Fill(
                recordSet.MutableMeta()->mutable_id(),
                recordSet.MutableSpec()->mutable_records(),
                &changelistJson
            );
            if (changelistJson.IsMap()) {
                recordSet.MutableChangelist()->FromJson(changelistJson);
            }

            result.emplace_back(std::move(recordSet));
        }

        options.SetContinuationToken(chunkResult.ContinuationToken);
        if (chunkResult.Results.size() < options.Limit()) {
            break;
        }
    }

    result.shrink_to_fit();

    return result;
}

using TYpClusterDataRef = TAtomicSharedPtr<TYpClusterData>;

TYpClusterDataFactory::TYpClusterDataFactory(size_t threads)
    : ThreadPool_(CreateThreadPool(threads))
{
}

TYpClusterDataRef TYpClusterDataFactory::GetYpClusterData(const TString& clusterName) {
    TGuard<TMutex> guard(Mutex_);
    THashMap<TString, TYpClusterDataRef>::insert_ctx insertCtx;
    auto it = YpClusterDataHolders_.find(clusterName, insertCtx);
    if (it != YpClusterDataHolders_.end()) {
        return it->second;
    }
    return YpClusterDataHolders_.emplace_direct(
        insertCtx,
        clusterName,
        MakeAtomicShared<TYpClusterData>(
            clusterName,
            NYP::NClient::TClientOptions()
                .SetAddress(clusterName)
                .SetToken(NYP::NClient::FindToken())
                .SetTimeout(TDuration::Seconds(10))
                .SetReadOnlyMode(true),
            *ThreadPool_
        )
    )->second;
}

TYpClusterDataRef GetYpClusterData(const TString& clusterName) {
    return Singleton<TYpClusterDataFactory>()->GetYpClusterData(clusterName);
}

} // namespace NYpDns::NGenerateZoneFiles
