#include "zone.h"

#include <infra/libs/yp_dns/record_set/record.h>
#include <infra/libs/yp_dns/zonefile/writer.h>

#include <infra/libs/yp_dns/replication/iterate.h>
#include <infra/libs/yp_dns/replication/merge.h>

#include <library/cpp/logger/global/global.h>

namespace NYpDns::NGenerateZoneFiles {

TZone::TZone(TZoneConfig config, IThreadPool& pool)
    : Config_(std::move(config))
    , Name_(Config_.GetName())
    , ClusterDatas_(Config_.GetYPClusters().size())
    , RecordSetsReplicas_(Config_.GetYPClusters().size())
    , Pool_(pool)
{
    for (const TString& clusterName : Config_.GetYPClusters()) {
        ClusterDatas_.emplace(clusterName, GetYpClusterData(clusterName));
        RecordSetsReplicas_[clusterName];
    }
}

NThreading::TFuture<void> TZone::Load() {
    TGuard<TMutex> guard(InitMutex_); if (InitFuture_.Initialized()) {
        return InitFuture_;
    }

    TVector<NThreading::TFuture<void>> loadReplicasFutures;
    loadReplicasFutures.reserve(ClusterDatas_.size());
    for (const auto& [clusterName, clusterData] : ClusterDatas_) {
        loadReplicasFutures.push_back(
            clusterData->RecordSets().Apply([this, &clusterName = clusterName](const NThreading::TFuture<TVector<TSerializedRecordSet>>& clusterReplicas) {
                return NThreading::Async([this, clusterReplicas, &clusterName] {
                        for (const TSerializedRecordSet& recordSetReplica : clusterReplicas.GetValueSync()) {
                            AddRecordSetReplicaIfBelongs(clusterName, recordSetReplica);
                        }
                    },
                    Pool_
                );
            })
        );
    }
    return InitFuture_ = NThreading::WaitAll(loadReplicasFutures).Apply([this](const auto& f) {
        f.TryRethrow();

        TVector<NThreading::TFuture<void>> futures;
        futures.reserve(RecordSetsReplicas_.size());
        for (auto& [cluster, replicas] : RecordSetsReplicas_) {
            futures.push_back(NThreading::Async(
                [&replicas = replicas] {
                    auto getKey = [](const TSerializedRecordSet& recordSet) {
                        return recordSet.GetObject().Meta().id();
                    };
                    if (!IsSortedBy(replicas.begin(), replicas.end(), getKey)) {
                        SortBy(replicas, getKey);
                    }
                },
                Pool_
            ));
        }
        return NThreading::WaitAll(futures);
    });
}

NThreading::TFuture<void> TZone::GenerateZoneFile(const TFsPath& outputDir) {
    return Load().Apply([this, &outputDir](const auto& f) {
        f.TryRethrow();

        TZoneWriterOptions options;
        options.ZoneConfig = Config_;

        const TFsPath outputPath = outputDir / Config_.GetName();
        TZoneFileWriter output(outputPath, std::move(options));

        // Add SOA record
        // TODO(dima-zakharov): create serial number from YP timestamp
        output.WriteRecord(CreateSOARecord(Config_, 0));

        // Add NS records
        for (const TString& nameserver : Config_.GetNameservers()) {
            output.WriteRecord(CreateNSRecord(Config_.GetName(), nameserver, Config_.GetNSRecordTtl()));
        }

        // Add other records
        Iterate(RecordSetsReplicas_, [&output](const TRecordSetReplicas& replicas) {
            TMaybe<TRecordSet> mergedRecordSet = MergeInOne(
                replicas.Replicas,
                TMergeOptions()
                    .SetFormChangelist(false)
                    .SetMergeAcls(false)
            );
            if (!mergedRecordSet.Defined()) {
                return true;
            }

            output << *mergedRecordSet;
            return true;
        });
        output.Finish();

        INFO_LOG << "Done generating zone file for " << Name_.toString() << ": " << outputPath << Endl;
    });
}

void TZone::AddRecordSetReplicaIfBelongs(const TString& clusterName, const TSerializedRecordSet& recordSetReplica) {
    TRecordSet recordSet = recordSetReplica.GetObject();
    if (DNSName(recordSet.Meta().id()).isPartOf(Name_)) {
        RecordSetsReplicas_[clusterName].emplace_back(recordSetReplica);
    }
}

TVector<TZone> CreateZones(const TVector<TZoneConfig>& zoneConfigs, IThreadPool& threadPool) {
    TVector<TZone> result;
    result.reserve(zoneConfigs.size());

    for (const TZoneConfig& zoneConfig : zoneConfigs) {
        result.emplace_back(zoneConfig, threadPool);
    }

    return result;
}

} // namespace NYpDns::NGenerateZoneFiles
