#include "client.h"

#include <infra/libs/service_iface/str_iface.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/protobuf/json/proto2json_printer.h>

#include <util/stream/mem.h>

namespace NInfra::NYpDnsApi {

namespace {

template <typename TRequest, typename TResponse>
int RunClient(int argc, const char* argv[], std::function<void(TClient&, TRequestPtr<TRequest>, TReplyPtr<TResponse>)>&& command) {
    NLastGetopt::TOpts opts;

    TString address;
    TString data;

    opts.AddLongOption('a', "address", "Bridge gRPC address")
        .Required()
        .RequiredArgument()
        .StoreResult(&address);

    opts.AddLongOption("data", "data")
        .Optional()
        .RequiredArgument()
        .StoreResult(&data);

    NLastGetopt::TOptsParseResult parsedOpts(&opts, argc, argv);

    TClient client(address);

    NProtobufJson::TProto2JsonConfig proto2JsonConfig;
    proto2JsonConfig.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);

    TRequest request;
    if (!data.empty()) {
        TMemoryInput input(data);
        request = NProtobufJson::Json2Proto<TRequest>(input);

        Cout << "request:" << Endl << NProtobufJson::Proto2Json(request, proto2JsonConfig) << Endl;
    }
    auto req = RequestPtr<TProtoRequest<TRequest>>("", std::move(request), NInfra::TAttributes());

    TResponse response;
    TAttributes responseAttributes;
    auto rsp = ReplyPtr<TProtoReply<TResponse>>(response, responseAttributes);
    command(client, req, rsp);

    Cout << "response:" << Endl << NProtobufJson::Proto2Json(response, proto2JsonConfig) << Endl;
    return EXIT_SUCCESS;
}

int Ping(int argc, const char* argv[]) {
    return RunClient<NApi::TReqPing, NApi::TRspPing>(argc, argv, [](TClient& client, TRequestPtr<NApi::TReqPing> request, TReplyPtr<NApi::TRspPing> reply) {
        client.Ping(request, reply);
    });
}

int UpdateRecords(int argc, const char* argv[]) {
    return RunClient<NApi::TReqUpdateRecords, NApi::TRspUpdateRecords>(argc, argv, [](TClient& client, TRequestPtr<NApi::TReqUpdateRecords> request, TReplyPtr<NApi::TRspUpdateRecords> reply) {
        client.UpdateRecords(request, reply);
    });
}

int ListZoneRecordSets(int argc, const char* argv[]) {
    return RunClient<NApi::TReqListZoneRecordSets, NApi::TRspListZoneRecordSets>(argc, argv, [](TClient& client, TRequestPtr<NApi::TReqListZoneRecordSets> request, TReplyPtr<NApi::TRspListZoneRecordSets> reply) {
        client.ListZoneRecordSets(request, reply);
    });
}

int ListZones(int argc, const char* argv[]) {
    return RunClient<NYpDns::NDynamicZones::NApi::TReqListZones, NYpDns::NDynamicZones::NApi::TRspListZones>(
        argc, argv,
        [](TClient& client, TRequestPtr<NYpDns::NDynamicZones::NApi::TReqListZones> request, TReplyPtr<NYpDns::NDynamicZones::NApi::TRspListZones> reply) {
            client.ListZones(request, reply);
        }
    );
}

int CreateZone(int argc, const char* argv[]) {
    return RunClient<NYpDns::NDynamicZones::NApi::TReqCreateZone, NYpDns::NDynamicZones::NApi::TRspCreateZone>(
        argc, argv,
        [](TClient& client, TRequestPtr<NYpDns::NDynamicZones::NApi::TReqCreateZone> request, TReplyPtr<NYpDns::NDynamicZones::NApi::TRspCreateZone> reply) {
            client.CreateZone(request, reply);
        }
    );
}

int RemoveZone(int argc, const char* argv[]) {
    return RunClient<NYpDns::NDynamicZones::NApi::TReqRemoveZone, NYpDns::NDynamicZones::NApi::TRspRemoveZone>(
        argc, argv,
        [](TClient& client, TRequestPtr<NYpDns::NDynamicZones::NApi::TReqRemoveZone> request, TReplyPtr<NYpDns::NDynamicZones::NApi::TRspRemoveZone> reply) {
            client.RemoveZone(request, reply);
        }
    );
}

} // namespace

int RunClient(int argc, const char* argv[]) {
    TModChooser modChooser;
    modChooser.AddMode("ping", Ping, "/ping");
    modChooser.AddMode("update_records", UpdateRecords, "/records/update");
    modChooser.AddMode("list_zone_record_sets", ListZoneRecordSets, "/list_zone_record_sets");
    modChooser.AddMode("list_zones", ListZones, "/zones/list");
    modChooser.AddMode("create_zone", CreateZone, "/zone/create");
    modChooser.AddMode("remove_zone", RemoveZone, "/zone/remove");

    return modChooser.Run(argc, argv);
}

TClient::TClient(const TStringBuf address, const TDuration timeout)
    : Channel_(grpc::CreateChannel(static_cast<std::string>(address), grpc::InsecureChannelCredentials()))
    , Stub_(NApi::TYpDnsApiBridgeService::NewStub(Channel_))
    , Timeout_(timeout)
    , ZonesManagerServiceClient_(
        NYpDns::NDynamicZones::TZonesManagerServiceClientOptions()
            .SetAddress(TString{address})
            .SetTimeout(timeout)
    )
{
}

void TClient::Ping(TRequestPtr<NApi::TReqPing> request, TReplyPtr<NApi::TRspPing> reply) {
    GrpcRequest(request, reply, &NApi::TYpDnsApiBridgeService::Stub::Ping);
}

void TClient::UpdateRecords(TRequestPtr<NApi::TReqUpdateRecords> request, TReplyPtr<NApi::TRspUpdateRecords> reply) {
    GrpcRequest(request, reply, &NApi::TYpDnsApiBridgeService::Stub::UpdateRecords);
}

void TClient::ListZoneRecordSets(TRequestPtr<NApi::TReqListZoneRecordSets> request, TReplyPtr<NApi::TRspListZoneRecordSets> reply) {
    GrpcRequest(request, reply, &NApi::TYpDnsApiBridgeService::Stub::ListZoneRecordSets);
}

void TClient::ListZones(TRequestPtr<NYpDns::NDynamicZones::NApi::TReqListZones> request, TReplyPtr<NYpDns::NDynamicZones::NApi::TRspListZones> reply) {
    ZonesManagerServiceClient_.ListZones(request, reply);
}

void TClient::CreateZone(TRequestPtr<NYpDns::NDynamicZones::NApi::TReqCreateZone> request, TReplyPtr<NYpDns::NDynamicZones::NApi::TRspCreateZone> reply) {
    ZonesManagerServiceClient_.CreateZone(request, reply);
}

void TClient::RemoveZone(TRequestPtr<NYpDns::NDynamicZones::NApi::TReqRemoveZone> request, TReplyPtr<NYpDns::NDynamicZones::NApi::TRspRemoveZone> reply) {
    ZonesManagerServiceClient_.RemoveZone(request, reply);
}

} // namespace NInfra::NYpDnsApi
