#pragma once

#include <infra/yp_dns_api/bridge/cluster_scoring/protos/config/config.pb.h>

#include <infra/yp_dns_api/bridge/misc/types.h>

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <util/system/mutex.h>

#include <atomic>

namespace NInfra::NYpDnsApi {

////////////////////////////////////////////////////////////////////////////////

class TClustersScoring;

////////////////////////////////////////////////////////////////////////////////

struct TClusterScore {
	TMutex Mutex;
	std::atomic<ui32> SequentialFails = 0;
	TInstant LastTryTime;

	TDuration TimeSinceLastTry() const {
		return TInstant::Now() - LastTryTime;
	}
};

using TClusterScorePtr = TAtomicSharedPtr<TClusterScore>;

////////////////////////////////////////////////////////////////////////////////

/// An interface to report actions were taken with cluster.
/// User is responsible to call corresponding methods.
/// An instance can be obtained from `TClusterDescriptor`.
class TClusterUseReporter {
    friend class TClusterDescriptor;

public:
	using TTryId = ui64;

	/// Call this when you are trying to use cluster in your task.
	void Try() {
		TGuard guard(Score_->Mutex);
		Score_->LastTryTime = TInstant::Now();
	}

	/// Call this if the task you've run with this cluster failed.
	/// Also you should consider to call `Try()` method before this.
	void Failed() noexcept {
		++Score_->SequentialFails;
	}

	/// Call this if the task you've run with this cluster succeeded.
	/// Also you should consider to call `Try()` method before this.
	void Succeeded() noexcept {
		Score_->SequentialFails.store(0);
	}

private:
	TClusterUseReporter(TClusterScorePtr score)
		: Score_(score)
	{
		Y_ENSURE(Score_);
	}

private:
	TClusterScorePtr Score_;
};

////////////////////////////////////////////////////////////////////////////////

/// NB: Do not use after `TClustersScoring` instance is destroyed.
///     We have a const reference to it here.
class TClusterDescriptor {
public:
	TClusterDescriptor(
		TClusterId clusterId,
		const TClustersScoring& scoring
	) noexcept
		: ClusterId_(std::move(clusterId))
		, Scoring_(scoring)
	{
	}

	const TClusterId& ClusterId() const noexcept {
		return ClusterId_;
	}

	TClusterUseReporter GetUseReporter() const;

private:
	const TClusterId ClusterId_;
	const TClustersScoring& Scoring_;
};

////////////////////////////////////////////////////////////////////////////////

/// The way the scoring machine will reorder clusters.
enum class EScoreAlgorithm {
	// [BEGIN EScoreAlgorithm::PessimizeWithFails]
	/// Clusters with less number of fails have a higher priority.
	///
	/// Order of clusters in result:
	/// 1. Clusters with positive fails counter which were tried a long time ago.
	/// 2. Clusters with zero fails counter.
	/// 3. Clusters with positive fails counter except those from 1st point.
	PessimizeWithFails,
	// [END EScoreAlgorithm::PessimizeWithFails]
};

template <EScoreAlgorithm Algo>
struct TScoringAlgorithmOptions;

template <>
struct TScoringAlgorithmOptions<EScoreAlgorithm::PessimizeWithFails> {
	/// The time during which the cluster will be pessimized
	/// when compared with others.
	TDuration AfterFailPessimizeTime;

	TScoringAlgorithmOptions(const TPessimizeWithFailsAlgoConfig& config);
};

////////////////////////////////////////////////////////////////////////////////

struct TClustersScoringOptions {
	TVector<TClusterId> ClusterIds;
};

class TClustersScoring: public TThrRefBase {
	friend class TClusterDescriptor;

public:
	TClustersScoring(TClustersScoringOptions options);

	TVector<TClusterDescriptor> Score(
		TVector<TClusterId> clusters,
		const TScoringAlgorithmConfig& algoConfig
	) const;

	/// Reorders clusters according to the chosen algorithm.
	/// Returns list of cluster descriptors ordered by clusters priority at the moment.
	/// First elements in result vector have higher priority, last elements have lower priority.
	template <EScoreAlgorithm Algo>
	TVector<TClusterDescriptor> Score(
		TVector<TClusterId> clusters,
		const TScoringAlgorithmOptions<Algo>& options
	) const {
		Reorder(clusters, options);
		return CreateDescriptors(std::move(clusters));
	}

private:
	TVector<TClusterDescriptor> CreateDescriptors(TVector<TClusterId>&& clusters) const;

	template <EScoreAlgorithm Algo>
	void Reorder(
		TVector<TClusterId>& clusters,
		const TScoringAlgorithmOptions<Algo>& options
	) const;

private: /* Reorder algorithms */

 	/// @see EScoreAlgorithm::PessimizeWithFails
	void PessimizeWithFails(
		TVector<TClusterId>& clusters,
		const TScoringAlgorithmOptions<EScoreAlgorithm::PessimizeWithFails>& options
	) const;

private:
	const TClustersScoringOptions Options_;
	THashMap<TClusterId, TClusterScorePtr> Scores_;
};

////////////////////////////////////////////////////////////////////////////////

template <EScoreAlgorithm Algo>
void TClustersScoring::Reorder(
	TVector<TClusterId>& clusters,
	const TScoringAlgorithmOptions<Algo>& options
) const {
	switch (Algo) {
		case EScoreAlgorithm::PessimizeWithFails: {
			PessimizeWithFails(clusters, options);
			break;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra::NYpDnsApi
