#include "make_events.h"

namespace NInfra::NYpDnsApi {

using namespace NEventlog::NBridge;
using namespace NYP::NYPReplica;

using NYP::NClient::NApi::NProto::TDnsRecordSetSpec;

TString AttributesToString(const TAttributes& attributes) {
    TString result;
    for (const auto& [key, value] : attributes) {
        result += TString::Join(key, "=", value, ";");
    }
    return result;
}

TDnsRecord MakeRecordEvent(const TDnsRecordSetSpec::TResourceRecord& record) {
    TDnsRecord result;
    result.SetTtl(record.ttl());
    result.SetClass(record.class_());
    result.SetType(TDnsRecordSetSpec::TResourceRecord::EType_Name(record.type()));
    result.SetData(record.data());
    return result;
}

TDnsRecord MakeRecordEvent(const NApi::TRecord& record) {
    TDnsRecord result;
    result.SetTtl(record.ttl());
    result.SetClass(record.class_());
    result.SetType(record.type());
    result.SetData(record.data());
    return result;
}

google::protobuf::RepeatedPtrField<TDnsRecord> MakeRecordsEvent(const google::protobuf::RepeatedPtrField<TDnsRecordSetSpec::TResourceRecord>& records) {
    google::protobuf::RepeatedPtrField<TDnsRecord> result;
    for (const TDnsRecordSetSpec::TResourceRecord& record : records) {
        *result.Add() = MakeRecordEvent(record);
    }
    return result;
}

TUpdateRecordRequest MakeUpdateRecordRequestEvent(const TRecordRequest& request) {
    TUpdateRecordRequest event;
    event.SetFqdn(request.Request->update().fqdn());
    event.SetType(NApi::ERecordType_Name(request.Request->update().type()));
    event.SetData(request.Request->update().data());
    event.SetTtl(request.Request->update().ttl());
    event.SetClass(request.Request->update().class_());
    if (request.ZoneConfig) {
        event.SetZone(request.ZoneConfig->GetName());
    } else {
        event.SetZone("<UNKNOWN>");
    }
    return event;
}

TRemoveRecordRequest MakeRemoveRecordRequestEvent(const TRecordRequest& request) {
    TRemoveRecordRequest event;
    event.SetFqdn(request.Request->remove().fqdn());
    event.SetType(NApi::ERecordType_Name(request.Request->remove().type()));
    event.SetData(request.Request->remove().data());
    if (request.ZoneConfig) {
        event.SetZone(request.ZoneConfig->GetName());
    } else {
        event.SetZone("<UNKNOWN>");
    }
    return event;
}

TUpdateRecordResponse MakeUpdateRecordResponseEvent(const TRecordRequest& request) {
    TUpdateRecordResponse event;
    event.SetFqdn(request.Request->update().fqdn());
    event.SetType(NApi::ERecordType_Name(request.Request->update().type()));
    event.SetData(request.Request->update().data());
    event.SetTtl(request.Request->update().ttl());
    event.SetClass(request.Request->update().class_());
    if (request.ZoneConfig) {
        event.SetZone(request.ZoneConfig->GetName());
    } else {
        event.SetZone("<UNKNOWN>");
    }
    event.SetStatus(NApi::TRspUpdateRecord::EUpdateRecordStatus_Name(request.Response.update().status()));
    event.SetErrorMessage(request.Response.update().error_message());
    event.SetCluster(request.Response.update().cluster());
    return event;
}

TRemoveRecordResponse MakeRemoveRecordResponseEvent(const TRecordRequest& request) {
    TRemoveRecordResponse event;
    event.SetFqdn(request.Request->remove().fqdn());
    event.SetType(NApi::ERecordType_Name(request.Request->remove().type()));
    event.SetData(request.Request->remove().data());
    if (request.ZoneConfig) {
        event.SetZone(request.ZoneConfig->GetName());
    } else {
        event.SetZone("<UNKNOWN>");
    }
    event.SetStatus(NApi::TRspRemoveRecord::ERemoveRecordStatus_Name(request.Response.remove().status()));
    event.SetErrorMessage(request.Response.remove().error_message());
    event.SetCluster(request.Response.remove().cluster());
    return event;
}

TYpGetRecordSetsRequest MakeYpGetRecordSetsRequestEvent(NYP::NClient::TClientPtr client, const TVector<TString>& ids, const TVector<TString>& selectors) {
    TYpGetRecordSetsRequest event;
    event.SetRecordSetsNumber(ids.size());
    event.SetTimestamp(client->Options().SnapshotTimestamp());
    event.MutableSelectors()->Reserve(selectors.size());
    for (const TString& selector : selectors) {
        event.AddSelectors(selector);
    }
    event.MutableIds()->Reserve(ids.size());
    for (const TString& id : ids) {
        event.AddIds(id);
    }
    return event;
}

TListZoneRecordSetsRequest MakeListZoneRecordSetsRequestEvent(const TRequestPtr<NApi::TReqListZoneRecordSets> request) {
    TListZoneRecordSetsRequest event;
    event.SetAttributes(AttributesToString(request->Attributes()));

    const NApi::TReqListZoneRecordSets& req = request->Get();
    event.SetZone(req.zone());
    event.SetLimit(req.limit());
    event.SetContinuationToken(req.continuation_token());

    return event;
}

TListZoneRecordSetsResponse MakeListZoneRecordSetsResponseEvent(const NApi::TRspListZoneRecordSets& response) {
    TListZoneRecordSetsResponse event;
    event.SetStatus(NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(response.status()));
    *event.MutableYpTimestamps() = response.yp_timestamps();
    event.SetRecordSetsNumber(response.record_sets().size());
    event.SetContinuationToken(response.continuation_token());
    *event.MutableBannedClusters() = response.banned_clusters();
    return event;
}

TListZoneRecordSetResponse MakeListZoneRecordSetResponseEvent(const NApi::TRecordSet& recordSet) {
    TListZoneRecordSetResponse event;
    event.SetId(recordSet.id());
    for (const NApi::TRecord&  record : recordSet.records()) {
        *event.add_records() = MakeRecordEvent(record);
    }
    return event;
}

TSeekOptions MakeSeekOptionsEvent(const TYPReplica<TDnsRecordSetReplicaObject>::TListOptions<TDnsRecordSetReplicaObject>& listOptions) {
    TSeekOptions event;
    event.SetRuleId(listOptions.TableInfo.RuleID);
    if (listOptions.TableInfo.TableName.Defined()) {
        event.SetTableName(*listOptions.TableInfo.TableName);
    }
    event.SetMode(ToString(listOptions.SeekType));
    event.SetKey(listOptions.SeekKey);
    return event;
}

TListZoneRecordSetsFromReplicaRequest MakeListZoneRecordSetsFromReplicaRequestEvent(
    const TYPReplica<TDnsRecordSetReplicaObject>& replica,
    const DNSName& zone,
    const TYPReplica<TDnsRecordSetReplicaObject>::TListOptions<TDnsRecordSetReplicaObject>& listOptions
) {
    TListZoneRecordSetsFromReplicaRequest event;
    event.SetCluster(replica.Name());
    event.SetZone(TString{zone.toStringNoDot()});
    event.SetLimit(listOptions.Limit);
    *event.MutableSeekOptions() = MakeSeekOptionsEvent(listOptions);
    return event;
}

TListZoneRecordSetsFromYPRequest MakeListZoneRecordSetsFromYPRequestEvent(
    const TStringBuf cluster,
    const DNSName& zone,
    const TYPReplica<TDnsRecordSetReplicaObject>::TListOptions<TDnsRecordSetReplicaObject>& listOptions
) {
    TListZoneRecordSetsFromYPRequest event;
    event.SetCluster(TString{cluster});
    event.SetZone(TString{zone.toStringNoDot()});
    event.SetLimit(listOptions.Limit);
    *event.MutableSeekOptions() = MakeSeekOptionsEvent(listOptions);
    return event;
}

TYpSelectZoneRecordSetsRequest MakeYpSelectZoneRecordSetsRequestEvent(
    const ui64 timestamp,
    const DNSName& zone,
    const TYPReplica<TDnsRecordSetReplicaObject>::TListOptions<TDnsRecordSetReplicaObject>& listOptions
) {
    TYpSelectZoneRecordSetsRequest event;
    event.SetTimestamp(timestamp);
    event.SetZone(TString{zone.toStringNoDot()});
    event.SetLimit(listOptions.Limit);
    *event.MutableSeekOptions() = MakeSeekOptionsEvent(listOptions);
    return event;
}

} // namespace NInfra::NYpDnsApi
