#include "banned_clusters.h"

#include <google/protobuf/descriptor.h>
#include <google/protobuf/reflection.h>

#include <algorithm>

namespace NInfra::NYpDnsApi {

TBannedClustersForMethod::TBannedClustersForMethod(
    const TBannedClustersConfig& initialConfig,
    int bannedClustersConfigFieldTagId
)
    : BannedClustersConfigFieldDescriptor_(
        TBannedClustersConfig::descriptor()->FindFieldByNumber(bannedClustersConfigFieldTagId))
{
    Y_ENSURE(BannedClustersConfigFieldDescriptor_);

    const auto clusters = GetBannedClusters(initialConfig);
    Banned_ = THashSet<TClusterId>(clusters.begin(), clusters.end());
}

NProtoBuf::RepeatedFieldRef<TString> TBannedClustersForMethod::GetBannedClusters(
    const TBannedClustersConfig& config
) const {
    return config.GetReflection()->GetRepeatedFieldRef<TString>(
        config,
        BannedClustersConfigFieldDescriptor_);
}

void TBannedClustersForMethod::OnConfigUpdate(
    const TBannedClustersConfig& oldConfig,
    const TBannedClustersConfig& newConfig
) {
    const NProtoBuf::RepeatedFieldRef<TString> bannedClustersOld = GetBannedClusters(oldConfig);
    const NProtoBuf::RepeatedFieldRef<TString> bannedClustersNew = GetBannedClusters(newConfig);

    if (!std::equal(bannedClustersOld.begin(), bannedClustersOld.end(),
                    bannedClustersNew.begin(), bannedClustersNew.end()))
    {
        THashSet<TClusterId> newBanned(
            bannedClustersNew.begin(),
            bannedClustersNew.end());

        TWriteGuard guard(Lock_);
        Banned_.swap(newBanned);
    }
}

bool TBannedClustersForMethod::IsBanned(const TClusterId& cluster) const {
    TReadGuard guard(Lock_);
    return Banned_.contains(cluster);
}

TBannedClusters::TBannedClusters(NUpdatableProtoConfig::TAccessor<TBannedClustersConfig> config)
    : Config_(std::move(config))
    , BannedForUpdateRecords_(*Config_, TBannedClustersConfig::kUpdateRecordsFieldNumber)
    , BannedForListZoneRecordSets_(*Config_, TBannedClustersConfig::kListZoneRecordSetsFieldNumber)
{
    Config_.SubscribeForUpdate([this](
        const TBannedClustersConfig& oldConfig,
        const TBannedClustersConfig& newConfig,
        const NUpdatableProtoConfig::TWatchContext&
    ) {
        BannedForUpdateRecords_.OnConfigUpdate(oldConfig, newConfig);
        BannedForListZoneRecordSets_.OnConfigUpdate(oldConfig, newConfig);
    });
}

bool TBannedClusters::IsBannedForUpdateRecords(const TClusterId& cluster) const {
    return BannedForUpdateRecords_.IsBanned(cluster);
}

bool TBannedClusters::IsBannedForListZoneRecordSets(const TClusterId& cluster) const {
    return BannedForListZoneRecordSets_.IsBanned(cluster);
}

} // namespace NInfra::NYpDnsApi
