#include "record_request.h"

#include "find_zone.h"

namespace NInfra::NYpDnsApi {

TRecordRequest::TRecordRequest(const NApi::TRecordRequest& request, const TZoneConfigs& zoneConfigs, const TDynamicZones& dynamicZones)
    : Request(&request)
{
    TString fqdn;
    if (Request->has_update()) {
        fqdn = Request->update().fqdn();
    } else if (Request->has_remove()) {
        fqdn = Request->remove().fqdn();
    }

    Fqdn = DNSName(fqdn);
    Fqdn.makeUsLowerCase();

    ZoneConfig = DetectZone(zoneConfigs, dynamicZones, Fqdn);
}

void TRecordRequest::SetRspStatus(
    NApi::TRspUpdateRecord::EUpdateRecordStatus updateStatus,
    NApi::TRspRemoveRecord::ERemoveRecordStatus removeStatus,
    const TString& errorMessage,
    const TString& cluster
) {
    // Update status only if it was empty or it is a status for another cluster
    if (Request->has_update()) {
        if (Response.update().status() == NApi::TRspUpdateRecord::UNDEFINED ||
            (!Response.update().cluster().empty() && Response.update().cluster() != cluster)
        ) {
            Response.mutable_update()->set_status(updateStatus);
            Response.mutable_update()->set_error_message(errorMessage);
            Response.mutable_update()->set_cluster(cluster);
        }
    } else if (Request->has_remove()) {
        if (Response.remove().status() == NApi::TRspRemoveRecord::UNDEFINED ||
            (!Response.update().cluster().empty() && Response.remove().cluster() != cluster)
        ) {
            Response.mutable_remove()->set_status(removeStatus);
            Response.mutable_remove()->set_error_message(errorMessage);
            Response.mutable_remove()->set_cluster(cluster);
        }
    }
}

void TRecordRequest::SetRspOkStatus(const TString& cluster) {
    SetRspStatus(NApi::TRspUpdateRecord::OK, NApi::TRspRemoveRecord::OK, {}, cluster);
}

void TRecordRequest::SetRspUnknownZoneStatus() {
    SetRspStatus(NApi::TRspUpdateRecord::UNKNOWN_ZONE, NApi::TRspRemoveRecord::UNKNOWN_ZONE);
}

void TRecordRequest::SetRspYpErrorStatus(const TString& errorMessage, const TString& cluster) {
    SetRspStatus(NApi::TRspUpdateRecord::YP_ERROR, NApi::TRspRemoveRecord::YP_ERROR, errorMessage, cluster);
}

void TRecordRequest::SetRspValidationErrorStatus(const TString& errorMessage, const TString& cluster) {
    SetRspStatus(NApi::TRspUpdateRecord::VALIDATION_ERROR, NApi::TRspRemoveRecord::VALIDATION_ERROR, errorMessage, cluster);
}

void TRecordRequest::SetRspInvalidHintStatus(const TString& HintCluster) {
    SetRspStatus(
        NApi::TRspUpdateRecord::INVALID_HINT,
        NApi::TRspRemoveRecord::INVALID_HINT, 
        TStringBuilder() << "Invalid cluster \"" << HintCluster << "\": "
                         << "Not found in set of \"" << ZoneConfig->GetName() << "\" zone clusters ["
                         << JoinSeq(", ", ZoneConfig->GetClusters()) << "]"
    );
}

void TRecordRequest::SetRspClusterBannedStatus(const TString& cluster) {
    SetRspStatus(NApi::TRspUpdateRecord::CLUSTER_BANNED, NApi::TRspRemoveRecord::CLUSTER_BANNED, "Cluster is banned to use by Bridge", cluster);
}

void TRecordRequest::SetRspZoneUpdatesDisabledStatus() {
    SetRspStatus(NApi::TRspUpdateRecord::ZONE_UPDATES_DISABLED, NApi::TRspRemoveRecord::ZONE_UPDATES_DISABLED);
}

void TRecordRequest::SetRspRequestThrottledStatus(const TString& cluster, const ui64 changesPerMinute) {
    SetRspStatus(
        NApi::TRspUpdateRecord::REQUEST_THROTTLED,
        NApi::TRspRemoveRecord::REQUEST_THROTTLED,
        TStringBuilder() << "Request was throttled because the number of changes in the record set"
                         << "during the last minute was more than " << ZoneConfig->GetMaxNumberChangesInRecordSetPerMinute()
                         << " (" << changesPerMinute << " changes during the last minute)",
        cluster
    );
}

void TRecordRequest::SetRspChangelistOverflowStatus(const TString& cluster, const ui64 size) {
    SetRspStatus(
        NApi::TRspUpdateRecord::CHANGELIST_OVERFLOW,
        NApi::TRspRemoveRecord::CHANGELIST_OVERFLOW,
        TStringBuilder() << "Changelist size is already " << size << " (Max size for " << ZoneConfig->GetName() << " zone: " << ZoneConfig->GetMaxChangelistSize() << ")", 
        cluster
    );
}

} // namespace NInfra::NYpDnsApi
