from util.generic.string cimport TStringBuf

import json
import logging
import multiprocessing
import requests
import time


cdef extern from "infra/yp_dns_api/bridge/daemon/main.h" namespace "NInfra::NYpDnsApi":
    int RunDaemon(const TStringBuf)


def _run_daemon(config):
    logging.debug('Start YP DNS API')
    logging.debug('Config: {}'.format(config))
    RunDaemon(config)


class YpDnsApiBridge(object):
    DEFAULT_BRIDGE_HTTP_PORT = 8080
    DEFAULT_GRPC_PORT = 8081
    DEFAULT_ADMIN_HTTP_PORT = 8082

    def __init__(self, config):
        self._config = config
        self._bridge_http_port = self._config.get('BridgeHttpServiceConfig', {}).get('Port')
        if self._bridge_http_port is None:
            self._bridge_http_port = self._config.setdefault('BridgeHttpServiceConfig', {})['Port'] = self.DEFAULT_BRIDGE_HTTP_PORT

        self._grpc_port = self._config.get('GrpcServiceConfig', {}).get('Port')
        if self._grpc_port is None:
            self._grpc_port = self._config.setdefault('GrpcServiceConfig', {})['Port'] = self.DEFAULT_GRPC_PORT

        self._admin_http_port = self._config.get('AdminHttpServiceConfig', {}).get('Port')
        if self._admin_http_port is None:
            self._admin_http_port = self._config.setdefault('AdminHttpServiceConfig', {})['Port'] = self.DEFAULT_ADMIN_HTTP_PORT

        config_json = bytes(json.dumps(self._config), 'utf8')
        self._process = multiprocessing.Process(target=_run_daemon, args=(config_json,))

    def _wait_start(self):
        while True:
            try:
                r = self.ping()
                r.raise_for_status()
                break
            except Exception as e:
                logging.debug(e)
                time.sleep(5)

    @property
    def config(self):
        return self._config

    @property
    def bridge_http_port(self):
        return self._bridge_http_port

    @property
    def grpc_port(self):
        return self._grpc_port

    @property
    def admin_http_port(self):
        return self._admin_http_port

    def ping(self):
        return requests.get('http://localhost:{}/ping'.format(self.bridge_http_port))

    def start(self):
        self._process.start()
        self._wait_start()
        logging.debug('YP DNS API started')

    def stop(self):
        logging.debug('Shutdown YP DNS API')
        try:
            r = requests.get('http://localhost:{}/shutdown'.format(self._admin_http_port))
            r.raise_for_status()
            logging.debug('YP DNS API shutdown response: [{}] {}'.format(r.status_code, r.content))
        except Exception as e:
            logging.debug(e)
        if self._process.is_alive():
            self._process.terminate()
