#include "service.h"

#include <infra/yp_dns_api/bridge/logger/events/events_decl.ev.pb.h>
#include <infra/yp_dns_api/bridge/logger/make_events.h>
#include <infra/yp_dns_api/bridge/misc/find_zone.h>
#include <infra/yp_dns_api/bridge/router_api/router_api.h>
#include <infra/yp_dns_api/bridge/sensors/sensors.h>

#include <infra/yp_dns_api/libs/yp/execute.h>

#include <infra/libs/memory_lock/memory_lock.h>
#include <infra/libs/sensors/sensor.h>
#include <infra/libs/yp_dns/dns_names/view.h>
#include <infra/libs/yp_dns/dynamic_zones/services/protos/api/bridge.pb.h>
#include <infra/libs/yp_dns/record_set/record.h>
#include <infra/libs/yp_dns/record_set/record_set.h>
#include <infra/libs/yp_dns/replication/apply_request.h>
#include <infra/libs/yp_dns/replication/iterate.h>
#include <infra/libs/yp_dns/replication/merge.h>
#include <infra/libs/yp_dns/yp_helpers/misc/token.h>

#include <yp/cpp/yp/data_model.h>
#include <yp/cpp/yp/token.h>

#include <yt/yt/client/tablet_client/public.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/retry/retry.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/watchdog/watchdog.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/generic/guid.h>
#include <util/generic/xrange.h>
#include <util/str_stl.h>
#include <util/string/cast.h>
#include <util/system/backtrace.h>
#include <util/system/hostname.h>

namespace NInfra::NYpDnsApi {

using namespace NEventlog::NBridge;
using NYP::NClient::NApi::NProto::TDnsRecordSetSpec;

namespace {

using namespace NYpDns;

constexpr ui64 STORAGE_FORMAT_VERSION = 8;

TDnsRecordSetSpec::TResourceRecord::EType GetType(NApi::ERecordType type) {
    switch (type) {
    case NApi::ERecordType::A:
        return TDnsRecordSetSpec::TResourceRecord::A;
    case NApi::ERecordType::NS:
        return TDnsRecordSetSpec::TResourceRecord::NS;
    case NApi::ERecordType::CNAME:
        return TDnsRecordSetSpec::TResourceRecord::CNAME;
    case NApi::ERecordType::SOA:
        return TDnsRecordSetSpec::TResourceRecord::SOA;
    case NApi::ERecordType::PTR:
        return TDnsRecordSetSpec::TResourceRecord::PTR;
    case NApi::ERecordType::MX:
        return TDnsRecordSetSpec::TResourceRecord::MX;
    case NApi::ERecordType::TXT:
        return TDnsRecordSetSpec::TResourceRecord::TXT;
    case NApi::ERecordType::AAAA:
        return TDnsRecordSetSpec::TResourceRecord::AAAA;
    case NApi::ERecordType::SRV:
        return TDnsRecordSetSpec::TResourceRecord::SRV;
    case NApi::ERecordType::RRSIG:
        return TDnsRecordSetSpec::TResourceRecord::RRSIG;
    case NApi::ERecordType::NSEC:
        return TDnsRecordSetSpec::TResourceRecord::NSEC;
    case NApi::ERecordType::DNSKEY:
        return TDnsRecordSetSpec::TResourceRecord::DNSKEY;
    case NApi::ERecordType::CAA:
        return TDnsRecordSetSpec::TResourceRecord::CAA;
    default:
        throw yexception() << "unknown record type";
    }
}

TMaybe<ui32> GetSerial(const TZoneConfig& zone, const ui64 maxYpTimestamp) {
    switch (zone.GetSerialGenerateMode()) {
        case NYpDns::ESerialNumberGenerateMode::ORIGINAL:
            return Nothing();
        case NYpDns::ESerialNumberGenerateMode::YP_TIMESTAMP_BASED:
        default: {
            return NYpDns::MakeSerialFromYpTimestamp(maxYpTimestamp);
        }
    }
}

NApi::TRecord MakeResultRecord(const TDnsRecordSetSpec::TResourceRecord& record, const TMaybe<ui32>& serial) {
    NApi::TRecord result;
    result.set_ttl(record.ttl());
    result.set_class_(record.class_());
    result.set_type(TDnsRecordSetSpec::TResourceRecord::EType_Name(record.type()));
    result.set_data(record.data());
    switch (record.type()) {
        case TDnsRecordSetSpec::TResourceRecord::SOA: {
            if (serial.Defined()) {
                NYpDns::TSOAData soaData(record.data());
                soaData.Serial = *serial;
                result.set_data(soaData.ToString());
            }
            break;
        }
        default:
            break;
    }
    return result;
}

NApi::TRecordSet MakeResultRecordSet(const TRecordSet& recordSet, const TMaybe<ui32>& serial) {
    NApi::TRecordSet result;
    result.set_id(recordSet.Meta().id());
    result.mutable_records()->Reserve(recordSet.Spec().records().size());;
    for (const auto& record : recordSet.Spec().records()) {
        *result.add_records() = MakeResultRecord(record, serial);
    }
    return result;
}

TString GetCluster(const NYP::NClient::TClientPtr client) {
    return TString{TStringBuf{client->Options().Address()}.Before('.')};
}

NProto::TRecordUpdateRequest MakeRecordUpdateRequest(const NApi::TReqUpdateRecord& request, const TZoneConfig* zoneConfig) {
    NProto::TRecordUpdateRequest result;
    result.set_type(NProto::TRecordUpdateRequest::UPDATE);

    NProto::TRecordUpdateRequest::TContent& content = *result.mutable_content();
    content.set_fqdn(request.fqdn());
    content.set_type(GetType(request.type()));
    content.set_data(request.data());

    ui64 ttl;
    if (request.ttl()) {
        ttl = request.ttl();
    } else {
        switch (request.type()) {
            case NApi::ERecordType::NS:
                ttl = zoneConfig->GetNSRecordDefaultTtl();
                break;
            default:
                ttl = zoneConfig->GetDefaultTtl();
                break;
        }
    }
    content.set_ttl(ttl);
    content.set_class_(request.class_());

    return result;
}

NProto::TRecordUpdateRequest MakeRecordUpdateRequest(const NApi::TReqRemoveRecord& request, const TZoneConfig* zoneConfig) {
    Y_UNUSED(zoneConfig);

    NProto::TRecordUpdateRequest result;
    result.set_type(NProto::TRecordUpdateRequest::REMOVE);

    NProto::TRecordUpdateRequest::TContent& content = *result.mutable_content();
    content.set_fqdn(request.fqdn());
    content.set_type(GetType(request.type()));
    content.set_data(request.data());

    return result;
}

TChange MakeChangeEntry(const TRecordRequest& recordRequest) {
    const auto& request = *recordRequest.Request;
    TChange result;

    result.set_replicated(false);
    result.set_uuid(CreateGuidAsString());
    result.set_timestamp(MicroSeconds());

    if (request.has_update()) {
        *result.mutable_record_update_request() = MakeRecordUpdateRequest(request.update(), recordRequest.ZoneConfig);
    } else if (request.has_remove()) {
        *result.mutable_record_update_request() = MakeRecordUpdateRequest(request.remove(), recordRequest.ZoneConfig);
    }
    return result;
}

class TZoneTableRule final : public NYP::NYPReplica::ITableRule<NYP::NYPReplica::TDnsRecordSetReplicaObject> {
public:
    constexpr TStringBuf GetID() const override {
        return ID;
    }

    constexpr bool IsStable() const override {
        return true;
    }

    virtual bool Filter(const NYP::NYPReplica::TDnsRecordSetReplicaObject& object) const override {
        return object.HasZone();
    }

    TString GetKey(const NYP::NYPReplica::TDnsRecordSetReplicaObject& object) const override {
        TStringBuf zone = object.GetZone().GetRef();
        zone.ChopSuffix(".");
        return TString::Join(zone, "#", object.GetKey());
    }

    TString GetTableName() const override {
        return TString();
    }

    NYP::NYPReplica::TColumnFamilyOptions GetColumnFamilyOptions() const override {
        NYP::NYPReplica::TColumnFamilyOptions options("");
        options.BlockTableConfig.SetIndexType(NYP::NYPReplica::TBlockTableConfig_EIndexType_BINARY_SEARCH);
        return options;
    }

    constexpr static TStringBuf ID = "ZONE";
};

TString GetFilterString(const TVector<TString>& ids, size_t from, size_t to) {
    TVector<TString> quotedIds;
    quotedIds.reserve(ids.size());
    for (size_t i = from; i < to; ++i) {
        quotedIds.push_back(TString::Join("\"", ids[i], "\""));
    }
    return TStringBuilder() << "[/meta/id] in (" << JoinSeq(", ", quotedIds) << ")";
}

TString GetSeekFilterString(const TListOptions& listOptions) {
    switch (listOptions.SeekType) {
        case NYP::NYPReplica::ESeekType::ToFirst:
            return {};
        case NYP::NYPReplica::ESeekType::AtOrNext:
            return TStringBuilder() << "[/meta/id] >= \"" << listOptions.SeekKey << "\"";
        case NYP::NYPReplica::ESeekType::Next:
            return TStringBuilder() << "[/meta/id] > \"" << listOptions.SeekKey << "\"";
        default:
            ythrow yexception() << "Seek type " << listOptions.SeekType << " is not supported";
    }
}

TString GetZoneFilterString(const TListOptions& listOptions, const TString& zone) {
    if (listOptions.TableInfo.RuleID == TZoneTableRule::ID) {
        return TStringBuilder() << "[/labels/zone] = \"" << zone << "\"";
    } else {
        return TStringBuilder() << "regex_partial_match('(^|\\.)" << zone << "\\.?$', [/meta/id])";
    }
}

TString GetFilterString(const TListOptions& listOptions, const TString& zone) {
    TString seekFilter = GetSeekFilterString(listOptions);
    TStringBuilder filter;
    if (seekFilter) {
        filter << "(" << seekFilter << ") and ";
    }
    return filter << GetZoneFilterString(listOptions, zone);
}

} // anonymous namespace

TService::TService(const TBridgeConfig& config)
    : ConfigHolder_(NUpdatableProtoConfig::CreateConfigHolder(config, config.GetUpdatableConfigOptions()))
    , Config_(ConfigHolder_->Accessor())
    , Logger_(CONFIG_SNAPSHOT_VALUE(Config_, GetLoggerConfig()))
    , SensorGroup_(NSensors::SERVICE_GROUP)
    , AdminHttpService_(
        CONFIG_SNAPSHOT_VALUE(Config_, GetAdminHttpServiceConfig()),
        CreateAdminRouter(*this)
    )
    , BridgeHttpService_(
        CONFIG_SNAPSHOT_VALUE(Config_, GetBridgeHttpServiceConfig()),
        CreateBridgeRouter(*this)
    )
    , GrpcServer_(
        CONFIG_SNAPSHOT_VALUE(Config_, GetGrpcServiceConfig()),
        *this
    )
    , DynamicZones_(MakeAtomicShared<TDynamicZones>())
    , BridgeStateServiceClient_(MakeAtomicShared<NYpDns::NDynamicZones::TBridgeServiceClient>(
        CONFIG_SNAPSHOT_VALUE(Config_, GetDynamicZonesConfig().GetBridgeServiceClientConfig()),
        [this] {
            TDynamicZonesPtr dynamicZones;
            {
                TReadGuard guard(DynamicZonesMutex_);
                dynamicZones = DynamicZones_;
            }

            if (!dynamicZones || dynamicZones->empty()) {
                ythrow yexception() << "Dynamic zones are not initialized";
            }

            NYpDns::NDynamicZones::NBridgeService::NApi::TReqReportConfiguration result;
            result.set_instance_id(HostName());
            result.mutable_zones()->Reserve(dynamicZones->size());
            for (const auto& [zoneId, zone] : *dynamicZones) {
                NYpDns::NDynamicZones::NBridgeService::NApi::TZone& zoneState = *result.add_zones();
                zoneState.set_id(TString{zoneId.toStringNoDot()});
                zoneState.set_status(NYpDns::NDynamicZones::NBridgeService::NApi::TZone::OK);
            }
            return result;
        }
    ))
    , ZonesManagerClient_(
        CONFIG_SNAPSHOT_VALUE(Config_, GetDynamicZonesConfig().GetZonesManagerClientConfig()),
        BridgeStateServiceClient_)
    , PollDynamicZonesLogger_(
        CONFIG_SNAPSHOT_VALUE(Config_, GetDynamicZonesConfig().GetPollConfig().GetLoggerConfig()))
    , DynamicZonesUpdater_(MakeHolder<TBackgroundThread>(
        [this] {
            NInfra::TLogFramePtr logFrame = PollDynamicZonesLogger_.SpawnFrame();
            INFRA_LOG_INFO(TPollDynamicZonesCycleStart());
            try {
                TVector<NYpDns::TZone> actualZones = ZonesManagerClient_.ListZones();
                TDynamicZonesPtr newDynamicZones = MakeAtomicShared<TDynamicZones>(actualZones.size());
                for (const NYpDns::TZone& zone : actualZones) {
                    DNSName zoneName(zone.GetName());
                    TZoneConfig config;
                    config.SetName(zone.GetName());
                    for (const TString& cluster : zone.Config().GetYPClusters()) {
                        config.AddClusters(cluster);
                    }
                    for (const TString& owner : zone.Config().GetOwners()) {
                        config.AddOwners(owner);
                    }
                    config.SetDefaultTtl(zone.Config().GetDefaultTtl());
                    config.SetNSRecordDefaultTtl(zone.Config().GetNSRecordTtl());
                    config.SetWriteToChangelist(true);
                    if (zone.Config().HasSerialGenerateMode()) {
                        config.SetSerialGenerateMode(zone.Config().GetSerialGenerateMode());
                    }
                    newDynamicZones->emplace(std::move(zoneName), std::move(config));
                }
                {
                    TWriteGuard guard(DynamicZonesMutex_);
                    DynamicZones_ = newDynamicZones;
                }
                INFRA_LOG_INFO(TPollDynamicZonesCycleSuccess());
            } catch (...) {
                INFRA_LOG_ERROR(TPollDynamicZonesCycleFailure(
                    CurrentExceptionMessage(),
                    TBackTrace::FromCurrentException().PrintToString()));
            }
        },
        TDuration::Seconds(1)
    ))
    , ZonesStateCoordinator_(MakeAtomicShared<NYpDns::NDynamicZones::TZonesStateCoordinator>(
        NYpDns::NDynamicZones::TZonesStateCoordinatorOptions(
            CONFIG_SNAPSHOT_VALUE(Config_, GetDynamicZonesConfig().GetZonesStateCoordinatorConfig()))
    ))
    , ZonesManagerService_(NYpDns::NDynamicZones::TZonesManagerServiceOptions(
        CONFIG_SNAPSHOT_VALUE(Config_, GetDynamicZonesConfig().GetZonesManagerServiceConfig()),
        ZonesStateCoordinator_
    ))
    , ClustersScoring_(TClustersScoringOptions{
        .ClusterIds = [config = Config_.Get()] { 
            TVector<TString> ids(Reserve(config->GetYpClientConfigs().size()));
            for (const auto& ypClientConfig : config->GetYpClientConfigs()) {
                ids.push_back(ypClientConfig.GetName());
            }
            return ids;
        }()
    })
    , YpClients_(CONFIG_SNAPSHOT_VALUE(Config_, GetYpClientConfigs().size()))
    , ReplicasManagementPool_(CreateThreadPool(CONFIG_SNAPSHOT_VALUE(Config_, GetYpClusterConfigs().size())))
    , ReplicaLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetReplicaLoggerConfig()))
    , BannedClusters_(Config_.Accessor<TBannedClustersConfig>("BannedClusters"))
{
    const auto initialConfig = Config_.Get();
    NMemoryLock::LockSelfMemory(initialConfig->GetMemoryLock(), Logger_.SpawnFrame(), SensorGroup_);
    const TString ypClientToken = NYpDns::FindYpClientToken();
    const TString ypReplicaToken = NYpDns::FindYpReplicaToken();
    Y_ENSURE(ypClientToken, "empty YP token for clients");
    Y_ENSURE(ypReplicaToken, "empty YP token for replicas");

    for (const TYpClientConfig& config : initialConfig->GetYpClientConfigs()) {
        NYP::NClient::TClientOptions clientOptions;
        clientOptions
            .SetAddress(config.GetAddress())
            .SetToken(ypClientToken)
            .SetEnableSsl(config.GetEnableSsl())
            .SetEnableBalancing(config.GetEnableBalancing())
            .SetTimeout(TDuration::Parse(config.GetTimeout()))
            .SetReadOnlyMode(config.GetReadOnlyMode());
        YpClients_.emplace(config.GetName(), NYP::NClient::CreateClient(clientOptions));
        YpTransactionFactories_.emplace(config.GetName(), NYP::NClient::CreateTransactionFactory(*YpClients_[config.GetName()]));
    }

    ZoneConfigs_.reserve(initialConfig->GetZoneConfigs().size());
    for (const TZoneConfig& zoneConfig : initialConfig->GetZoneConfigs()) {
        for (const TString& cluster : zoneConfig.GetClusters()) {
            Y_ENSURE(YpClients_.contains(cluster));
        }
        ZoneConfigs_.emplace_back(DNSName(zoneConfig.GetName()), zoneConfig);
    }

    InitReplicas(ypReplicaToken);
    InitSensors();
}

TService::~TService() {
    try {
        StopReplicas();
    } catch (...) {
    }
}

void TService::Start() {
    ServiceLogFrame_ = Logger_.SpawnFrame();

    ConfigHolder_->Start();
    AdminHttpService_.Start(ServiceLogFrame_);

    StartReplicas();

    ZonesManagerService_.Start();
    DynamicZonesUpdater_->Start();
    BridgeStateServiceClient_->RegisterInstance(ServiceLogFrame_);
    BridgeStateServiceClient_->StartReportConfiguration();
    ZonesStateCoordinator_->Start(ServiceLogFrame_);

    GrpcServer_.Start(ServiceLogFrame_);
    BridgeHttpService_.Start(ServiceLogFrame_);
    ServiceLogFrame_->LogEvent(TServiceStart());
}

void TService::Wait() {
    BridgeHttpService_.Wait(ServiceLogFrame_);
    GrpcServer_.Wait(ServiceLogFrame_);
    AdminHttpService_.Wait(ServiceLogFrame_);
    ZonesStateCoordinator_->Stop(ServiceLogFrame_);
}

void TService::Ping(TRequestPtr<NApi::TReqPing> request, TReplyPtr<NApi::TRspPing> reply) {
    Logger_.SpawnFrame()->LogEvent(TPing(AttributesToString(request->Attributes())));
    TRateSensor(SensorGroup_, NSensors::PING_REQUESTS).Inc();

    NApi::TRspPing result;
    result.SetData("pong");
    reply->Set(result);
}

void TService::ReopenLog(TRequestPtr<NApi::TReqReopenLog> request, TReplyPtr<NApi::TRspReopenLog> reply) {
    Logger_.SpawnFrame()->LogEvent(TReopenLog(AttributesToString(request->Attributes())));
    TRateSensor(SensorGroup_, NSensors::REOPEN_LOG_REQUESTS).Inc();

    Logger_.ReopenLog();
    ConfigHolder_->ReopenLog();
    ReplicaLogger_.ReopenLog();
    ZonesStateCoordinator_->ReopenLogs();
    ZonesManagerService_.ReopenLogs();
    PollDynamicZonesLogger_.ReopenLog();

    reply->Set(NApi::TRspReopenLog());
}

void TService::Shutdown(TRequestPtr<NApi::TReqShutdown> request, TReplyPtr<NApi::TRspShutdown>) {
    Logger_.SpawnFrame()->LogEvent(TShutdown(AttributesToString(request->Attributes())));
    TRateSensor(SensorGroup_, NSensors::SHUTDOWN_REQUESTS).Inc();

    static THolder<IWatchDog> abortWatchDog = THolder<IWatchDog>(CreateAbortByTimeoutWatchDog(CreateAbortByTimeoutWatchDogOptions(TDuration::Minutes(1)), "Ooops!"));
    BridgeHttpService_.ShutDown();
    GrpcServer_.Shutdown();
    AdminHttpService_.ShutDown();
}

void TService::Sensors(TRequestPtr<NApi::TReqSensors> request, TReplyPtr<NApi::TRspSensors> reply) {
    Logger_.SpawnFrame()->LogEvent(TSensors(AttributesToString(request->Attributes())));
    TRateSensor(SensorGroup_, NSensors::SENSORS_REQUESTS).Inc();

    NApi::TRspSensors result;
    TStringOutput stream(*result.MutableData());
    SensorRegistryPrint(stream, ESensorsSerializationType::SPACK_V1);
    reply->SetAttribute("Content-Type", "application/x-solomon-spack");
    reply->Set(result);
}

void TService::InitRecordSet(TMaybe<TRecordSet>& recordSet, const TRecordRequest& request) const {
    if (!recordSet.Defined()) {
        recordSet.ConstructInPlace();
        recordSet->MutableMeta()->set_id(TString{request.Fqdn.toStringNoDot()});
        if (!request.ZoneConfig->GetOwners().empty()) {
            NYP::NClient::NApi::NProto::TAccessControlEntry* ace = recordSet->MutableMeta()->add_acl();
            ace->set_action(NYP::NClient::NApi::NProto::EAccessControlAction::ACA_ALLOW);
            ace->add_permissions(NYP::NClient::NApi::NProto::EAccessControlPermission::ACP_READ);
            ace->add_permissions(NYP::NClient::NApi::NProto::EAccessControlPermission::ACA_WRITE);
            *ace->mutable_subjects() = request.ZoneConfig->GetOwners();
        }
    }
}

void TService::UpdateRecords(TRequestPtr<NApi::TReqUpdateRecords> request, TReplyPtr<NApi::TRspUpdateRecords> reply) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        TUpdateRecordsRequest(AttributesToString(request->Attributes())),
        TUpdateRecordsTime(),
        ResponseTimeHistograms_->GetLifetimeHistogram(NSensors::UPDATE_RECORDS)
    );

    TSensorGroup sensorGroup{SensorGroup_, NSensors::UPDATE_RECORDS};
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    const TDynamicZonesPtr dynamicZones = DynamicZones();

    TVector<THolder<TRecordRequest>> allRequests;
    THashMap<DNSName, TVector<TRecordRequest*>> requestsByFqdn;
    THashMap<TString, THashSet<DNSName>> fqdnsByCluster;
    for (const NApi::TRecordRequest& recordUpdateRequest : request->Get().requests()) {
        TRecordRequest& req = *allRequests.emplace_back(MakeHolder<TRecordRequest>(recordUpdateRequest, ZoneConfigs_, *dynamicZones));

        if (req.Request->has_update()) {
            logFrame->LogEvent(MakeUpdateRecordRequestEvent(req));
            TRateSensor{TSensorGroup{sensorGroup, NSensors::UPDATE_RECORD}, NSensors::REQUESTS}.Inc();
        } else if (req.Request->has_remove()) {
            logFrame->LogEvent(MakeRemoveRecordRequestEvent(req));
            TRateSensor{TSensorGroup{sensorGroup, NSensors::REMOVE_RECORD}, NSensors::REQUESTS}.Inc();
        }

        if (!req.ZoneConfig) {
            req.SetRspUnknownZoneStatus();
        } else if (!req.ZoneConfig->GetAllowUpdateRecords()) {
            req.SetRspZoneUpdatesDisabledStatus();
        } else if (const TString& hintCluster = req.Request->hints().cluster(); !hintCluster.empty() && !IsIn(req.ZoneConfig->GetClusters(), hintCluster)) {
            req.SetRspInvalidHintStatus(hintCluster);
        } else {
            requestsByFqdn[req.Fqdn].push_back(&req);
            if (!hintCluster.empty()) {
                fqdnsByCluster[hintCluster].insert(req.Fqdn);
            } else {
                for (const TString& cluster : req.ZoneConfig->GetClusters()) {
                    fqdnsByCluster[cluster].insert(req.Fqdn);
                }
            }
        }
    }

    const TVector<TClusterDescriptor> clusters = GetOrderedClusters(fqdnsByCluster);

    for (const TClusterDescriptor& clusterDescriptor : clusters) {
        const THashSet<DNSName>& fqdns = fqdnsByCluster[clusterDescriptor.ClusterId()];
        TVector<DNSName> unprocessedFqdns(Reserve(fqdns.size()));
        for (const DNSName& fqdn : fqdns) {
            if (!requestsByFqdn.at(fqdn).front()->Commited) {
                unprocessedFqdns.push_back(fqdn);
            }
        }
        UpdateRecords(requestsByFqdn, unprocessedFqdns, clusterDescriptor, logFrame, sensorGroup);
    }

    NApi::TRspUpdateRecords responses;
    responses.mutable_responses()->Reserve(allRequests.size());
    TSensorGroup statusSensorGroup{sensorGroup, NSensors::STATUS};
    bool success = true;
    for (const THolder<TRecordRequest>& request : allRequests) {
        if (request->Response.has_update()) {
            logFrame->LogEvent(MakeUpdateRecordResponseEvent(*request));
            TRateSensor{statusSensorGroup, NApi::TRspUpdateRecord::EUpdateRecordStatus_Name(request->Response.update().status())}.Inc();
            success &= request->Response.update().status() == NApi::TRspUpdateRecord::OK;
        } else if (request->Response.has_remove()) {
            logFrame->LogEvent(MakeRemoveRecordResponseEvent(*request));
            TRateSensor{statusSensorGroup, NApi::TRspRemoveRecord::ERemoveRecordStatus_Name(request->Response.remove().status())}.Inc();
            success &= request->Response.remove().status() == NApi::TRspRemoveRecord::OK;
        }
        *responses.mutable_responses()->Add() = request->Response;
    }
    if (success) {
        TRateSensor{sensorGroup, NSensors::SUCCESSES}.Inc();
    } else {
        TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
    }
    reply->Set(responses);
}

void TService::UpdateRecords(
    const THashMap<DNSName, TVector<TRecordRequest*>>& requestsByFqdn,
    const TVector<DNSName>& fqdns,
    const TClusterDescriptor& cluster,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    auto setStatus = [](const TVector<TRecordRequest*>& requests, auto setter, bool commited = false) {
        for (TRecordRequest* request : requests) {
            setter(request);
            if (commited) {
                request->Commited = commited;
            }
        }
    };

    auto setStatuses = [&setStatus, &fqdns, &requestsByFqdn](auto setter, bool commited = false) {
        for (const DNSName& fqdn : fqdns) {
            setStatus(requestsByFqdn.at(fqdn), setter, commited);
        }
    };

    INFRA_LOG_INFO(TUpdateRecords(cluster.ClusterId(), fqdns.size()));

    if (BannedClusters_.IsBannedForUpdateRecords(cluster.ClusterId())) {
        setStatuses([&cluster](TRecordRequest* requests){requests->SetRspClusterBannedStatus(cluster.ClusterId());});
        INFRA_LOG_INFO(TUpdateRecordsClusterBanned());
        return;
    }

    if (fqdns.empty()) {
        INFRA_LOG_INFO(TUpdateRecordsNothingToDo(cluster.ClusterId()));
        return;
    }

    for (const DNSName& fqdn : fqdns) {
        logFrame->LogEvent(TRecordSetFqdnToUpdate(TString{fqdn.toStringNoDot()}));
    }

    sensorGroup.AddLabel("yp_cluster", cluster.ClusterId());
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    TClusterUseReporter clusterUseReporter = cluster.GetUseReporter();

    TFindRecordSetsResult findRecordSetsResult;
    try {
        findRecordSetsResult = FindRecordSets(cluster.ClusterId(), fqdns, requestsByFqdn, logFrame, sensorGroup);
    } catch (const NYP::NClient::TResponseError& ex) {
        clusterUseReporter.Try();
        clusterUseReporter.Failed();
        setStatuses([errorMessage = ex.what(), &cluster](TRecordRequest* requests) {
            requests->SetRspYpErrorStatus(errorMessage, cluster.ClusterId());
        });
        INFRA_LOG_ERROR(TUpdateRecordsFailure(cluster.ClusterId(), ex.what()));
        TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
        return;
    }

    TVector<TRecordSet> creates;
    TVector<TRecordSet> updates;
    TVector<TRecordSet> removes;
    for (size_t i = 0; i < fqdns.size(); ++i) {
        const TVector<TRecordRequest*>& fqdnRequests = requestsByFqdn.at(fqdns[i]);
        if (!CheckChangelist(findRecordSetsResult.RecordSets[i], fqdnRequests, cluster, logFrame)) {
            continue;
        }
        try {
            UpdateRecordSet(std::move(findRecordSetsResult.RecordSets[i]), fqdnRequests, creates, updates, removes, logFrame, sensorGroup);
        } catch (...) {
            setStatus(fqdnRequests, [errorMessage = CurrentExceptionMessage(), &cluster](TRecordRequest* requests) {
                requests->SetRspValidationErrorStatus(errorMessage, cluster.ClusterId());
            });
            INFRA_LOG_ERROR(TUpdateRecordSetError(cluster.ClusterId(), TString{fqdns[i].toStringNoDot()}, CurrentExceptionMessage()));
        }
    }

    INFRA_LOG_INFO(TRecordSetsUpdatesStats(creates.size(), updates.size(), removes.size()));
    if (creates.empty() && updates.empty() && removes.empty()) {
        setStatuses([&cluster](TRecordRequest* requests){requests->SetRspOkStatus(cluster.ClusterId());}, /* commited */ true);
        INFRA_LOG_INFO(TUpdateRecordsNothingToDo(cluster.ClusterId()));
        TRateSensor{sensorGroup, NSensors::SUCCESSES}.Inc();
        return;
    }

    clusterUseReporter.Try();
    const NYP::NClient::TTransactionFactoryPtr transactionFactory = YpTransactionFactories_.at(cluster.ClusterId());
    try {
        NYP::NClient::TTransactionPtr transaction = CreateTransaction(transactionFactory, findRecordSetsResult.YpTimestamp, logFrame, sensorGroup);
        CreateRecordSets(transaction, creates, logFrame, sensorGroup);
        UpdateRecordSets(transaction, updates, logFrame, sensorGroup);
        RemoveRecordSets(transaction, removes, logFrame, sensorGroup);
        CommitTransaction(transaction, logFrame, sensorGroup);
    } catch (const NYP::NClient::TResponseError& ex) {
        clusterUseReporter.Failed();
        setStatuses([errorMessage = ex.what(), &cluster](TRecordRequest* requests) {
            requests->SetRspYpErrorStatus(errorMessage, cluster.ClusterId());
        });
        INFRA_LOG_ERROR(TUpdateRecordsFailure(cluster.ClusterId(), ex.what()));
        TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
        return;
    }

    clusterUseReporter.Succeeded();
    setStatuses([&cluster](TRecordRequest* requests){requests->SetRspOkStatus(cluster.ClusterId());}, /* commited */ true);
    INFRA_LOG_INFO(TUpdateRecordsSuccess(cluster.ClusterId()));
    TRateSensor{sensorGroup, NSensors::SUCCESSES}.Inc();
}

bool TService::CheckChangelist(const TMaybe<NYpDns::TRecordSet>& recordSet, const TVector<TRecordRequest*>& requests, const TClusterDescriptor& cluster, TLogFramePtr logFrame) const {
    if (recordSet.Empty() || !recordSet->HasChangelist() || requests.empty()) {
        return true;
    }

    const ui64 maxChangelistSize = requests[0]->ZoneConfig->GetMaxChangelistSize();
    if (maxChangelistSize && (ui64)recordSet->Changelist().changes().size() >= maxChangelistSize) {
        for (TRecordRequest* request : requests) {
            request->SetRspChangelistOverflowStatus(cluster.ClusterId(), recordSet->Changelist().changes().size());
        }
        INFRA_LOG_INFO(TCheckChangelistOverflow(cluster.ClusterId(), TString{requests[0]->Fqdn.toStringNoDot()}, recordSet->Changelist().changes().size(), maxChangelistSize));
        return false;
    }

    const ui64 maxChangesPerMinute = requests[0]->ZoneConfig->GetMaxNumberChangesInRecordSetPerMinute();

    if (maxChangesPerMinute) {
        ui64 currentTimestamp = MicroSeconds();
        ui64 lastMinuteQuantity = 0;

        for (const auto& change : recordSet->Changelist().changes()) {
            if (currentTimestamp >= change.timestamp() && TDuration::MicroSeconds(currentTimestamp - change.timestamp()) <= TDuration::Minutes(1)) {
                ++lastMinuteQuantity;
            }
        }

        if (lastMinuteQuantity >= maxChangesPerMinute) {
            for (TRecordRequest* request : requests) {
                request->SetRspRequestThrottledStatus(cluster.ClusterId(), lastMinuteQuantity);
            }
            INFRA_LOG_INFO(TCheckChangelistThrottled(cluster.ClusterId(), TString{requests[0]->Fqdn.toStringNoDot()}, lastMinuteQuantity, maxChangesPerMinute));
            return false;
        }
    }

    return true;
}

void TService::UpdateRecordSet(
    TMaybe<TRecordSet>&& recordSet,
    const TVector<TRecordRequest*>& requests,
    TVector<TRecordSet>& creates,
    TVector<TRecordSet>& updates,
    TVector<TRecordSet>& removes,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    const bool exists = recordSet.Defined();

    const TZoneConfig* const zoneConfig = requests.front()->ZoneConfig;
    sensorGroup = TSensorGroup{sensorGroup, NSensors::UPDATE_RECORD_SET};
    sensorGroup.AddLabel("zone", zoneConfig->GetName());

    if (exists) {
        logFrame->LogEvent(MakeRecordSetEvent<TRecordSetToUpdate>(*recordSet));
    } else {
        logFrame->LogEvent(TRecordSetToUpdate(TString{requests.front()->Fqdn.toStringNoDot()}, zoneConfig->GetName(), {}, {}));
    }

    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    bool changed = false;
    TRecordRequest* prevRequest = nullptr;
    for (TRecordRequest* request : requests) {
        if (prevRequest && google::protobuf::util::MessageDifferencer::Equals(*prevRequest->Request, *request->Request)) {
            continue;
        }
        if (request->Request->has_update()) {
            changed |= UpdateRecord(recordSet, *request, logFrame, sensorGroup);
        } else if (request->Request->has_remove()) {
            changed |= RemoveRecord(recordSet, *request, logFrame, sensorGroup);
        }
        prevRequest = request;
    }

    if (changed) {
        TRecordSetUpdateResult event;
        if (recordSet.Defined()) {
            event = MakeRecordSetEvent<TRecordSetUpdateResult>(*recordSet);
        } else {
            event = TRecordSetUpdateResult(TString{requests.front()->Fqdn.toStringNoDot()}, zoneConfig->GetName(), {}, {}, {});
        }

        const bool isChangelistEmpty = recordSet.Defined() && zoneConfig->GetWriteToChangelist()
            ? (!recordSet->HasChangelist() || recordSet->Changelist().changes().empty())
            : true;
        if (!exists && recordSet.Defined() && (!recordSet->Spec().records().empty() || !isChangelistEmpty)) {
            event.SetOperationType(TRecordSetUpdateResult::CREATE);
            creates.emplace_back(std::move(*recordSet));
        } else if (exists && recordSet->Spec().records().empty() && isChangelistEmpty) {
            event.SetOperationType(TRecordSetUpdateResult::REMOVE);
            removes.emplace_back(std::move(*recordSet));
        } else if (!recordSet->Spec().records().empty() || !isChangelistEmpty) {
            event.SetOperationType(TRecordSetUpdateResult::UPDATE);
            updates.emplace_back(std::move(*recordSet));
        } else {
            event.SetOperationType(TRecordSetUpdateResult::SKIP);
        }

        // TODO log if not changed too
        logFrame->LogEvent(event);
    }
}

bool TService::UpdateRecord(TMaybe<TRecordSet>& recordSet, TRecordRequest& request, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(MakeUpdateRecordRequestEvent(request));
    sensorGroup = TSensorGroup{sensorGroup, NSensors::UPDATE_RECORD};
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    if (!recordSet.Defined()) {
        InitRecordSet(recordSet, request);
    }

    recordSet->SetZone(request.ZoneConfig->GetName());

    const TChange change = MakeChangeEntry(request);
    bool recordSetChanged = ApplyUpdateRequest(recordSet, change, request.ZoneConfig->GetWriteToChangelist(),
        [&logFrame](const TDnsRecordSetSpec::TResourceRecord& record) {
            logFrame->LogEvent(TRecordBeforeUpdate(MakeRecordEvent(record)));
        },
        [&logFrame](const TDnsRecordSetSpec::TResourceRecord& record) {
            logFrame->LogEvent(TRecordAfterUpdate(MakeRecordEvent(record)));
        },
        [&logFrame]() {
            logFrame->LogEvent(TRecordNotFound());
        });

    return recordSetChanged;
}

bool TService::RemoveRecord(TMaybe<TRecordSet>& recordSet, TRecordRequest& request, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(MakeRemoveRecordRequestEvent(request));
    sensorGroup = TSensorGroup{sensorGroup, NSensors::REMOVE_RECORD};
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    if (!recordSet.Defined() && !request.ZoneConfig->GetWriteToChangelist()) {
        logFrame->LogEvent(TRecordNotFound());
        return false;
    }

    if (!recordSet.Defined()) {
        InitRecordSet(recordSet, request);
    }

    recordSet->SetZone(request.ZoneConfig->GetName());

    const TChange change = MakeChangeEntry(request);
    bool recordSetChanged = ApplyRemoveRequest(recordSet, change, request.ZoneConfig->GetWriteToChangelist(),
        [&logFrame](const TDnsRecordSetSpec::TResourceRecord& record) {
            logFrame->LogEvent(TRecordRemoved(MakeRecordEvent(record)));
        });

    if (!recordSetChanged) {
        logFrame->LogEvent(TRecordNotFound());
    }

    return recordSetChanged;
}

TVector<TClusterDescriptor> TService::GetOrderedClusters(
    const THashMap<TString, THashSet<DNSName>>& fqdnsByCluster
) const {
    const auto config = Config_.Get();

    TVector<TString> clusters(Reserve(fqdnsByCluster.size()));
    for (const auto& [cluster, fqdns] : fqdnsByCluster) {
        clusters.emplace_back(cluster);
    }

    return ClustersScoring_.Score(std::move(clusters), config->GetClustersBalancingConfig().GetAlgorithmConfig());
}

void TService::ListZoneRecordSets(TRequestPtr<NApi::TReqListZoneRecordSets> request, TReplyPtr<NApi::TRspListZoneRecordSets> reply) {
    ui64 responseSize = 0;

    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        MakeListZoneRecordSetsRequestEvent(request),
        TListElementsTime(),
        ResponseTimeHistograms_->GetLifetimeHistogram(NSensors::LIST_ZONE_RECORD_SETS, responseSize)
    );

    const TDynamicZonesPtr dynamicZones = DynamicZones();

    const NApi::TReqListZoneRecordSets& req = request->Get();

    NApi::TRspListZoneRecordSets rsp;

    TSensorGroup sensorGroup{SensorGroup_, NSensors::LIST_ZONE_RECORD_SETS};
    sensorGroup.AddLabel("zone", req.zone());

    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    TSensorGroup statusSensorGroup{sensorGroup, NSensors::STATUS};

    DNSName zoneName(req.zone());
    zoneName.makeUsLowerCase();
    NYpDns::TDnsNameView zoneNameView(req.zone());
    const TString keyPrefix = zoneName.toStringNoDot() + "#";

    const TZoneConfig* zoneConfig = FindZone(ZoneConfigs_, *dynamicZones, zoneName);
    if (!zoneConfig) {
        rsp.set_status(NApi::TRspListZoneRecordSets::UNKNOWN_ZONE);
        logFrame->LogEvent(MakeListZoneRecordSetsResponseEvent(rsp));
        reply->Set(rsp);

        TRateSensor{statusSensorGroup, NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status())}.Inc();
        return;
    }

    TVector<TClusterId> clusters(Reserve(zoneConfig->GetClusters().size()));
    for (const TString& cluster : zoneConfig->GetClusters()) {
        if (!BannedClusters_.IsBannedForListZoneRecordSets(cluster)) {
            clusters.emplace_back(cluster);
        } else {
            rsp.add_banned_clusters(cluster);
        }
    }

    if (clusters.empty()) {
        rsp.set_status(NApi::TRspListZoneRecordSets::DATA_UNAVAILABLE);
        logFrame->LogEvent(MakeListZoneRecordSetsResponseEvent(rsp));
        reply->Set(rsp);

        TRateSensor{statusSensorGroup, NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status())}.Inc();
        return;
    }

    TContinuationListOptions listOptions;
    listOptions.Limit = req.limit();

    if (req.continuation_token()) {
        listOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
        try {
            listOptions.SeekKey = Base64Decode(req.continuation_token());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, TDecodeContinuationTokenError(CurrentExceptionMessage()));

            rsp.set_status(NApi::TRspListZoneRecordSets::INVALID_CONTINUATION_TOKEN);
            logFrame->LogEvent(MakeListZoneRecordSetsResponseEvent(rsp));
            reply->Set(rsp);

            TRateSensor{statusSensorGroup, NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status())}.Inc();
            return;
        }
    } else {
        listOptions.SeekType = NYP::NYPReplica::ESeekType::ToFirst;
    }

    if (zoneConfig->GetRecordSetsLabeledWithZone()) {
        listOptions.TableInfo = NYP::NYPReplica::TTableInfo<NYP::NYPReplica::TDnsRecordSetReplicaObject>(TString(TZoneTableRule::ID));
        listOptions.Stop = [&keyPrefix](const TStringBuf key, const TVector<NYP::NYPReplica::TStorageElement<NYP::NYPReplica::TDnsRecordSetReplicaObject>>&) {
            return !key.StartsWith(keyPrefix);
        };
    } else {
        listOptions.TableInfo = NYP::NYPReplica::TTableInfo<NYP::NYPReplica::TDnsRecordSetReplicaObject>(
            TString(NYP::NYPReplica::GetDefaultRuleID<NYP::NYPReplica::TDnsRecordSetReplicaObject>()));
        listOptions.Filter = [&zoneNameView](const TStringBuf key, const TVector<NYP::NYPReplica::TStorageElement<NYP::NYPReplica::TDnsRecordSetReplicaObject>>&) {
            return NYpDns::TDnsNameView(key).IsPartOf(zoneNameView);
        };
    }

    TListRecordSetObjectsFromClustersResult listResult;

    listResult.ContinuationOptions = listOptions;
    listResult.ClustersContinuationOptions.reserve(zoneConfig->GetClusters().size());
    for (const TString& cluster : zoneConfig->GetClusters()) {
        TContinuationListOptions& clusterListOptions = listResult.ClustersContinuationOptions.emplace(cluster, listOptions).first->second;
        clusterListOptions.Snapshot = GetReplica(cluster).GetReplicaSnapshot();
    }

    while ((!req.limit() || listResult.RecordSetObjects.size() < req.limit()) && !listResult.ContinuationOptions.HasReachedEnd) {
        TListRecordSetObjectsFromClustersResult listResultBatch = ListZoneRecordSetsFromClusters(
            zoneName,
            *zoneConfig,
            clusters,
            listResult.ContinuationOptions,
            listResult.ClustersContinuationOptions,
            logFrame,
            sensorGroup);

        const size_t mergeSize = req.limit()
            ? Min(req.limit() - listResult.RecordSetObjects.size(), listResultBatch.RecordSetObjects.size())
            : listResultBatch.RecordSetObjects.size();
        listResult.RecordSetObjects.reserve(listResult.RecordSetObjects.size() + mergeSize);
        std::move(
            listResultBatch.RecordSetObjects.begin(),
            listResultBatch.RecordSetObjects.begin() + mergeSize,
            std::back_inserter(listResult.RecordSetObjects));

        listResult.ContinuationOptions = std::move(listResultBatch.ContinuationOptions);
        if (!listResult.RecordSetObjects.empty() && mergeSize < listResultBatch.RecordSetObjects.size()) {
            listResult.ContinuationOptions.SeekKey = listResult.RecordSetObjects.back().Meta().id();
        }

        listResult.ClustersContinuationOptions = std::move(listResultBatch.ClustersContinuationOptions);
    }

    ui64 maxYpTimestamp = 0;
    rsp.set_status(NApi::TRspListZoneRecordSets::OK);
    for (const auto& [cluster, continuationOptions] : listResult.ClustersContinuationOptions) {
        maxYpTimestamp = Max(maxYpTimestamp, continuationOptions.YpTimestamp);
        (*rsp.mutable_yp_timestamps())[cluster] = continuationOptions.YpTimestamp;
    }
    const TMaybe<ui32>& serial = GetSerial(*zoneConfig, maxYpTimestamp);
    rsp.mutable_record_sets()->Reserve(listResult.RecordSetObjects.size());
    for (const TRecordSet& recordSetObject : listResult.RecordSetObjects) {
        const auto& recordSet = *rsp.add_record_sets() = MakeResultRecordSet(recordSetObject, serial);
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, MakeListZoneRecordSetResponseEvent(recordSet));
    }
    responseSize += listResult.RecordSetObjects.size();
    if (!listResult.RecordSetObjects.empty()) {
        rsp.set_continuation_token(Base64Encode(listResult.RecordSetObjects.back().Meta().id()));
    }

    logFrame->LogEvent(MakeListZoneRecordSetsResponseEvent(rsp));
    TRateSensor{statusSensorGroup, NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status())}.Inc();

    reply->Set(rsp);
}

void TService::ListZones(NInfra::TRequestPtr<NYpDns::NDynamicZones::NApi::TReqListZones> request, NInfra::TReplyPtr<NYpDns::NDynamicZones::NApi::TRspListZones> reply) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        TListZonesStart(),
        TListZonesTime(),
        ResponseTimeHistograms_->GetLifetimeHistogram(NSensors::LIST_ZONES)
    );
    ZonesManagerService_.ListZones(request, reply);
}

void TService::CreateZone(NInfra::TRequestPtr<NYpDns::NDynamicZones::NApi::TReqCreateZone> request, NInfra::TReplyPtr<NYpDns::NDynamicZones::NApi::TRspCreateZone> reply) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        TCreateZoneStart(),
        TCreateZoneTime(),
        ResponseTimeHistograms_->GetLifetimeHistogram(NSensors::CREATE_ZONE)
    );
    ZonesManagerService_.CreateZone(request, reply);
}

void TService::RemoveZone(NInfra::TRequestPtr<NYpDns::NDynamicZones::NApi::TReqRemoveZone> request, NInfra::TReplyPtr<NYpDns::NDynamicZones::NApi::TRspRemoveZone> reply) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        TRemoveZoneStart(),
        TRemoveZoneTime(),
        ResponseTimeHistograms_->GetLifetimeHistogram(NSensors::REMOVE_ZONE)
    );
    ZonesManagerService_.RemoveZone(request, reply);
}

ui64 TService::GenerateYpTimestamp(NYP::NClient::TClientPtr client, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TYpGenerateTimestampRequest());
    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_GENERATE_TIMESTAMP};
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();
    try {
        const ui64 result = ExecuteYpRequest<ui64>(
            [client] {
                return client->GenerateTimestamp().GetValue(client->Options().Timeout() * 2);
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpGenerateTimestampError(ex.what()));
                TRateSensor{sensorGroup, NSensors::ERRORS}.Inc();
            }
        );
        logFrame->LogEvent(TYpGenerateTimestampSuccess(result));
        TRateSensor{sensorGroup, NSensors::SUCCESSES}.Inc();
        return result;
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpGenerateTimestampFailure(ex.what()));
        TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
        throw;
    }
}

NYP::NClient::TTransactionPtr TService::CreateTransaction(NYP::NClient::TTransactionFactoryPtr transactionFactory, ui64 timestamp, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TYpCreateTransactionRequest(timestamp));
    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_START_TRANSACTION};
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();
    try {
        auto result = ExecuteYpRequest<NYP::NClient::TTransactionPtr>(
            [transactionFactory, timestamp, logFrame, &sensorGroup] {
                NYP::NClient::TTransactionPtr transaction = transactionFactory->CreateTransaction(timestamp);
                logFrame->LogEvent(TYpCreateTransactionSuccess(transaction->TransactionId()));
                TRateSensor{sensorGroup, NSensors::SUCCESSES}.Inc();
                return transaction;
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpCreateTransactionError(ex.what()));
                TRateSensor{sensorGroup, NSensors::ERRORS}.Inc();
            }
        );
        return result;
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpCreateTransactionFailure(ex.what()));
        TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
        throw;
    }
}

void TService::CommitTransaction(NYP::NClient::TTransactionPtr transaction, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TYpCommitTransactionRequest(transaction->TransactionId()));
    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_COMMIT_TRANSACTION};
    TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();
    try {
        const ui64 commitTimestamp = ExecuteYpRequest<ui64>(
            [transaction, logFrame] {
                return transaction->Commit().GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpCommitTransactionError(ex.what()));
                TRateSensor{sensorGroup, NSensors::ERRORS}.Inc();
                const int errorCode = static_cast<int>(ex.Error().GetNonTrivialCode());
                switch (NYT::NTabletClient::EErrorCode(errorCode)) {
                    case NYT::NTabletClient::EErrorCode::TransactionLockConflict:
                        throw ex;
                    default:
                        break;
                }
                switch (NYP::NClient::NApi::EErrorCode(errorCode)) {
                    case NYP::NClient::NApi::EErrorCode::NoSuchTransaction:
                    case NYP::NClient::NApi::EErrorCode::AuthorizationError:
                        throw ex;
                    default:
                        break;
                }
            }
        );
        logFrame->LogEvent(TYpCommitTransactionSuccess(commitTimestamp));
        TRateSensor{sensorGroup, NSensors::SUCCESSES}.Inc();
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpCommitTransactionFailure(ex.what()));
        TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
        throw;
    }
}

void TService::CreateRecordSets(NYP::NClient::TTransactionPtr transaction, const TVector<TRecordSet>& recordSets, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TYpCreateRecordSetsRequest(recordSets.size()));
    if (recordSets.empty()) {
        logFrame->LogEvent(TYpEmptyCreateRecordSetsList());
        return;
    }

    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_CREATE_RECORD_SETS};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TVector<NYP::NClient::TCreateObjectRequest> requests;
    requests.reserve(recordSets.size());
    for (const TRecordSet& recordSet : recordSets) {
        logFrame->LogEvent(MakeRecordSetEvent<TYpCreateRecordSetContent>(recordSet));
        requests.emplace_back(recordSet.MakeYpObject());
    }

    try {
        ExecuteYpRequest(
            [transaction, &requests] {
                transaction->CreateObjects(requests).GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpCreateRecordSetsError(ex.what()));
                TRateSensor(sensorGroup, NSensors::ERRORS).Inc();
            }
        );
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpCreateRecordSetsFailure(ex.what()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }

    logFrame->LogEvent(TYpCreateRecordSetsSuccess());
}

void TService::UpdateRecordSets(NYP::NClient::TTransactionPtr transaction, const TVector<TRecordSet>& recordSets, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TYpUpdateRecordSetsRequest(recordSets.size()));
    if (recordSets.empty()) {
        logFrame->LogEvent(TYpEmptyUpdateRecordSetsList());
        return;
    }

    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_UPDATE_RECORD_SETS};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TVector<NYP::NClient::TUpdateRequest> requests;
    requests.reserve(recordSets.size());
    for (const TRecordSet& recordSet : recordSets) {
        logFrame->LogEvent(MakeRecordSetEvent<TYpUpdateRecordSetContent>(recordSet));

        TVector<NYP::NClient::TSetRequest> setRequests;
        TVector<NYP::NClient::TRemoveRequest> removeRequests;

        setRequests.emplace_back("/spec/records", recordSet.Spec().records());
        if (const TMaybe<TString>& zone = recordSet.Zone(); zone.Defined()) {
            setRequests.emplace_back("/labels/zone", *zone);
        } else {
            removeRequests.emplace_back("/labels/zone");
        }
        if (recordSet.HasChangelist()) {
            setRequests.emplace_back("/labels/changelist", NYT::NodeFromJsonValue(recordSet.Changelist().ToJson()));
        } else {
            removeRequests.emplace_back("/labels/changelist");
        }
        requests.emplace_back(NYP::NClient::TDnsRecordSet::ObjectType, recordSet.Meta().id(), std::move(setRequests), std::move(removeRequests));
    }

    try {
        ExecuteYpRequest(
            [transaction, &requests] {
                transaction->UpdateObjects(requests).GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpUpdateRecordSetsError(ex.what()));
                TRateSensor(sensorGroup, NSensors::ERRORS).Inc();
            }
        );
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpUpdateRecordSetsFailure(ex.what()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }

    logFrame->LogEvent(TYpUpdateRecordSetsSuccess());
}

void TService::RemoveRecordSets(NYP::NClient::TTransactionPtr transaction, const TVector<TRecordSet>& recordSets, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TYpRemoveRecordSetsRequest(recordSets.size()));
    if (recordSets.empty()) {
        logFrame->LogEvent(TYpEmptyRemoveRecordSetsList());
        return;
    }

    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_REMOVE_RECORD_SETS};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TVector<NYP::NClient::TRemoveObjectRequest> requests;
    requests.reserve(recordSets.size());
    for (const TRecordSet& recordSet : recordSets) {
        logFrame->LogEvent(MakeRecordSetEvent<TYpRemoveRecordSetContent>(recordSet));
        requests.emplace_back(NYP::NClient::TDnsRecordSet::ObjectType, recordSet.Meta().id());
    }

    try {
        ExecuteYpRequest(
            [transaction, &requests] {
                transaction->RemoveObjects(requests).GetValue(transaction->ClientOptions().Timeout() * 2);
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpRemoveRecordSetsError(ex.what()));
                TRateSensor(sensorGroup, NSensors::ERRORS).Inc();
            }
        );
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpRemoveRecordSetsFailure(ex.what()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }

    logFrame->LogEvent(TYpRemoveRecordSetsSuccess());
}

TVector<NYP::NClient::TSelectorResult> TService::SelectZoneRecordSets(
    NYP::NClient::TClientPtr client,
    const ui64 ypTimestamp,
    const DNSName& zone,
    const TListOptions& listOptions,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    static const TVector<TString> selectors = {"/meta/id", "/spec", "/labels/zone", "/labels/changelist"};

    logFrame->LogEvent(MakeYpSelectZoneRecordSetsRequestEvent(ypTimestamp, zone, listOptions));

    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_SELECT_ZONE_RECORD_SETS};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    NYP::NClient::TSelectObjectsResult result;
    const TString zoneName(zone.toStringNoDot());
    try {
        result = ExecuteYpRequest<NYP::NClient::TSelectObjectsResult>(
            [client, &zoneName, &listOptions, ypTimestamp] {
                return client->SelectObjects<NYP::NClient::TDnsRecordSet>(
                    selectors,
                    GetFilterString(listOptions, zoneName),
                    NYP::NClient::TSelectObjectsOptions().SetLimit(listOptions.Limit),
                    ypTimestamp
                ).GetValue(client->Options().Timeout() * 2);
            },
            [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpSelectZoneRecordSetsError(ex.what()));
                TRateSensor(sensorGroup, NSensors::ERRORS).Inc();
            }
        );

        logFrame->LogEvent(TYpSelectZoneRecordSetsSuccess(result.Results.size()));

        return std::move(result.Results);
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpSelectZoneRecordSetsFailure(ex.what()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }
}

TListRecordSetObjectsResult TService::ListZoneRecordSetsFromReplica(const TYPReplica& replica, const DNSName& zone, TContinuationListOptions listOptions, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    sensorGroup = TSensorGroup{sensorGroup, NSensors::FROM_REPLICA};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TString initialSeekKey = listOptions.SeekKey;
    if (listOptions.TableInfo.RuleID == TZoneTableRule::ID) {
        if (listOptions.SeekType == NYP::NYPReplica::ESeekType::ToFirst) {
            listOptions.SeekType = NYP::NYPReplica::ESeekType::AtOrNext;
            listOptions.SeekKey = zone.toStringNoDot() + "#";
        } else {
            listOptions.SeekKey = TString::Join(zone.toStringNoDot(), "#", initialSeekKey);
        }
    }

    logFrame->LogEvent(MakeListZoneRecordSetsFromReplicaRequestEvent(replica, zone, listOptions));

    NYP::NYPReplica::TListElementsResult<NYP::NYPReplica::TDnsRecordSetReplicaObject> elements = replica.ListElements<NYP::NYPReplica::TDnsRecordSetReplicaObject>(listOptions);

    TListRecordSetObjectsResult result;
    result.YpTimestamp = listOptions.YpTimestamp = *replica.GetYpTimestamp(listOptions.Snapshot);
    result.RecordSetObjects.reserve(elements.size());
    for (auto&& [key, values] : elements) {
        Y_ENSURE(values.size() == 1);
        result.RecordSetObjects.emplace_back(std::move(values.front().ReplicaObject));
    }

    if (!result.RecordSetObjects.empty()) {
        listOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
        listOptions.SeekKey = result.RecordSetObjects.back().GetObject().Meta().id();
    } else {
        listOptions.SeekType = NYP::NYPReplica::ESeekType::ToFirst;
        listOptions.SeekKey = std::move(initialSeekKey);
    }
    listOptions.ListFrom = TContinuationListOptions::EListFrom::REPLICA;
    listOptions.HasReachedEnd = !listOptions.Limit || result.RecordSetObjects.size() < listOptions.Limit;
    result.ContinuationOptions = std::move(listOptions);

    logFrame->LogEvent(TListZoneRecordSetsFromReplicaSuccess::FromFields(result.YpTimestamp, result.RecordSetObjects.size()));
    TRateSensor(sensorGroup, NSensors::SUCCESSES).Inc();

    return result;
}

TListRecordSetObjectsResult TService::ListZoneRecordSetsFromYP(
    const NYP::NClient::TClientPtr client,
    const DNSName& zone,
    TContinuationListOptions listOptions,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    logFrame->LogEvent(MakeListZoneRecordSetsFromYPRequestEvent(GetCluster(client), zone, listOptions));

    sensorGroup = TSensorGroup{sensorGroup, NSensors::FROM_YP};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TListRecordSetObjectsResult result;
    try {
        if (!listOptions.YpTimestamp) {
            listOptions.YpTimestamp = GenerateYpTimestamp(client, logFrame, sensorGroup);
        }
        result.YpTimestamp = listOptions.YpTimestamp;
        const TVector<NYP::NClient::TSelectorResult> selectionResults = SelectZoneRecordSets(client, result.YpTimestamp, zone, listOptions, logFrame, sensorGroup);

        result.RecordSetObjects.reserve(selectionResults.size());
        for (const NYP::NClient::TSelectorResult& selectorResult : selectionResults) {
            TRecordSet recordSet;
            NYT::TNode zone;
            selectorResult.Fill(
                recordSet.MutableMeta()->mutable_id(),
                recordSet.MutableSpec(),
                &zone,
                recordSet.MutableChangelist()
            );
            if (zone.HasValue()) {
                recordSet.SetZone(zone.AsString());
            }
            result.RecordSetObjects.emplace_back(std::move(recordSet));
        }

        if (!result.RecordSetObjects.empty()) {
            listOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
            listOptions.SeekKey = result.RecordSetObjects.back().GetObject().Meta().id();
        }
        listOptions.ListFrom = TContinuationListOptions::EListFrom::YP;
        listOptions.HasReachedEnd = !listOptions.Limit || result.RecordSetObjects.size() < listOptions.Limit;
        result.ContinuationOptions = std::move(listOptions);
    } catch (...) {
        logFrame->LogEvent(TListZoneRecordSetsFromYPFailure(CurrentExceptionMessage()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }

    logFrame->LogEvent(TListZoneRecordSetsFromYPSuccess::FromFields(result.YpTimestamp, result.RecordSetObjects.size()));
    TRateSensor(sensorGroup, NSensors::SUCCESSES).Inc();

    return result;
}

TListRecordSetObjectsFromClustersResult TService::ListZoneRecordSetsFromClusters(
    const DNSName& zone,
    const TZoneConfig& zoneConfig,
    const TVector<TClusterId>& clusters,
    TContinuationListOptions listOptions,
    THashMap<TString, TContinuationListOptions> clustersListOptions,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    THashMap<TString, TListRecordSetObjectsResult> listResults;
    for (const TString& cluster : clusters) {
        TContinuationListOptions& clusterListOptions = clustersListOptions[cluster];
        clusterListOptions.Limit = listOptions.Limit ? (listOptions.Limit + clusters.size() - 1) : 0;
        clusterListOptions.SeekType = listOptions.SeekType;
        clusterListOptions.SeekKey = listOptions.SeekKey;

        const TYPReplica& replica = GetReplica(cluster);
        NYP::NClient::TClientPtr ypClient = YpClients_.at(cluster);

        listResults[cluster] = ListZoneRecordSetsFromCluster(cluster, replica, ypClient, zone, zoneConfig, clusterListOptions, logFrame, sensorGroup);
    }

    THashMap<TString, TVector<TSerializedRecordSet>> recordSets(listResults.size());
    for (auto& [cluster, listResult] : listResults) {
        recordSets[cluster] = std::move(listResult.RecordSetObjects);
    }

    TListRecordSetObjectsFromClustersResult result;
    TString lastId;
    bool hasReachedLimit = false;
    Iterate(recordSets, [&result, &listOptions = std::as_const(listOptions), &lastId, &hasReachedLimit](const TRecordSetReplicas& recordSetReplicas) {
        if (listOptions.Limit && result.RecordSetObjects.size() == listOptions.Limit) {
            hasReachedLimit = true;
            return false;
        }

        TMaybe<TRecordSet> mergedRecordSet = MergeInOne(
            recordSetReplicas.Replicas,
            TMergeOptions()
                .SetFormChangelist(false)
                .SetMergeAcls(false)
        );
        if (mergedRecordSet.Defined()) {
            result.RecordSetObjects.emplace_back(std::move(*mergedRecordSet));
        }

        lastId = recordSetReplicas.Fqdn;
        return true;
    });

    listOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
    listOptions.SeekKey = lastId;
    listOptions.HasReachedEnd = !hasReachedLimit && AllOf(listResults, [&](const auto& it) { return it.second.ContinuationOptions.HasReachedEnd; });
    result.ContinuationOptions = std::move(listOptions);

    result.ClustersContinuationOptions.reserve(listResults.size());
    for (auto&& [cluster, listResult] : listResults) {
        listResult.ContinuationOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
        listResult.ContinuationOptions.SeekKey = lastId;
        result.ClustersContinuationOptions[cluster] = std::move(listResult.ContinuationOptions);
    }

    return result;
}

TListRecordSetObjectsResult TService::ListZoneRecordSetsFromCluster(
    const TString& cluster,
    const TYPReplica& replica,
    NYP::NClient::TClientPtr ypClient,
    const DNSName& zone,
    const TZoneConfig& zoneConfig,
    const TContinuationListOptions& listOptions,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    if (listOptions.ListFrom.Defined()) {
        switch (*listOptions.ListFrom) {
            case TContinuationListOptions::EListFrom::REPLICA: {
                return ListZoneRecordSetsFromReplica(replica, zone, listOptions, logFrame, sensorGroup);
            }
            case TContinuationListOptions::EListFrom::YP: {
                return ListZoneRecordSetsFromYP(ypClient, zone, listOptions, logFrame, sensorGroup);
            }
            default:
                break;
        }
    }

    const TMaybe<TDuration> age = replica.Age(listOptions.Snapshot);
    const TDuration maxAcceptableAge = TDuration::Parse(zoneConfig.GetMaxReplicaAgeForListing());
    logFrame->LogEvent(TReplicaAge(cluster, age.Defined() ? age->ToString() : "<UNDEFINED>", maxAcceptableAge.ToString()));

    TListRecordSetObjectsResult listResult;
    if (age.Defined() && age < maxAcceptableAge) {
        listResult = ListZoneRecordSetsFromReplica(replica, zone, listOptions, logFrame, sensorGroup);
    } else {
        TRateSensor{sensorGroup, NSensors::FALLBACK_TO_YP}.Inc();

        try {
            listResult = ListZoneRecordSetsFromYP(ypClient, zone, listOptions, logFrame, sensorGroup);
        } catch (...) {
            logFrame->LogEvent(TFallbackToListZoneRecordSetsFromReplica());
            TRateSensor{TSensorGroup{sensorGroup, NSensors::FALLBACK_TO_YP}, NSensors::FALLBACK_TO_REPLICA}.Inc();
            listResult = ListZoneRecordSetsFromReplica(replica, zone, listOptions, logFrame, sensorGroup);
        }
    }

    return listResult;
}

TVector<NYP::NClient::TSelectorResult> TService::GetRecordSets(NYP::NClient::TClientPtr client, const TVector<TString>& ids, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    static const TVector<TString> selectors = {
        "/meta/id",
        "/spec",
        "/labels/zone",
        "/labels/changelist",
    };
    logFrame->LogEvent(MakeYpGetRecordSetsRequestEvent(client, ids, selectors));
    if (ids.empty()) {
        logFrame->LogEvent(TYpEmptyGetRecordSetsList());
        return {};
    }

    sensorGroup = TSensorGroup{sensorGroup, NSensors::YP_GET_RECORD_SETS};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TVector<NYP::NClient::TSelectorResult> result;
    result.reserve(ids.size());
    try {
        const size_t batchSize = 100;
        for (size_t i = 0; i < ids.size(); i += batchSize) {
            auto subResult = ExecuteYpRequest<NYP::NClient::TSelectObjectsResult>(
                [client, &ids, i] {
                    return client->SelectObjects<NYP::NClient::TDnsRecordSet>(
                        selectors,
                        GetFilterString(ids, i, Min(i + batchSize, ids.size())),
                        NYP::NClient::TSelectObjectsOptions()
                    ).GetValue(client->Options().Timeout() * 2);
                },
                [logFrame, &sensorGroup] (const NYP::NClient::TResponseError& ex) {
                    logFrame->LogEvent(ELogPriority::TLOG_WARNING, TYpGetRecordSetsError(ex.what()));
                    TRateSensor(sensorGroup, NSensors::ERRORS).Inc();
                }
            );
            std::move(subResult.Results.begin(), subResult.Results.end(), std::back_inserter(result));
        }
    } catch (const NYP::NClient::TResponseError& ex) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TYpGetRecordSetsFailure(ex.what()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }

    logFrame->LogEvent(TYpGetRecordSetsSuccess());
    TRateSensor(sensorGroup, NSensors::SUCCESSES).Inc();

    return result;
}

TGetRecordSetsResult TService::GetRecordSetsFromReplica(const TYPReplica& replica, const TYPReplicaSnapshot& snapshot, const TVector<TString>& ids, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TGetRecordSetsFromReplicaRequest(replica.Name(), ids.size()));

    sensorGroup = TSensorGroup{sensorGroup, NSensors::FROM_REPLICA};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TGetRecordSetsResult result;
    result.YpTimestamp = *replica.GetYpTimestamp(snapshot);
    result.RecordSets.reserve(ids.size());
    for (const TString& id : ids) {
        const TMaybe<NYP::NYPReplica::TReplicaSelectionResult<NYP::NClient::TDnsRecordSet>> lookupResult = replica.GetByKey<NYP::NYPReplica::TDnsRecordSetReplicaObject>(id, snapshot);
        if (lookupResult.Defined()) {
            Y_ENSURE(lookupResult->Objects.size() == 1);
            Y_ENSURE(lookupResult->YpTimestamp == result.YpTimestamp);
            result.RecordSets.emplace_back(lookupResult->Objects.front());
        } else {
            result.RecordSets.emplace_back(Nothing());
        }
    }

    logFrame->LogEvent(TGetRecordSetsFromReplicaSuccess(result.YpTimestamp));
    TRateSensor(sensorGroup, NSensors::SUCCESSES).Inc();

    return result;
}

TGetRecordSetsResult TService::GetRecordSetsFromYP(NYP::NClient::TClientPtr client, const TVector<TString>& ids, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    logFrame->LogEvent(TGetRecordSetsFromYPRequest(GetCluster(client), ids.size()));

    sensorGroup = TSensorGroup{sensorGroup, NSensors::FROM_YP};
    TRateSensor(sensorGroup, NSensors::REQUESTS).Inc();

    TGetRecordSetsResult result;
    try {
        result.YpTimestamp = GenerateYpTimestamp(client, logFrame, sensorGroup);

        NYP::NClient::TClientOptions options = client->Options();
        options.SetSnapshotTimestamp(result.YpTimestamp);
        client = NYP::NClient::CreateClient(options);

        TVector<NYP::NClient::TSelectorResult> selectionResults = GetRecordSets(client, ids, logFrame, sensorGroup);

        result.RecordSets.reserve(selectionResults.size());
        for (const NYP::NClient::TSelectorResult& selectorResult : selectionResults) {
            if (selectorResult.Values().value_payloads().empty()) {
                result.RecordSets.emplace_back(Nothing());
                continue;
            }

            TRecordSet recordSet;
            TString zone;
            NJson::TJsonValue changelistJson;
            selectorResult.Fill(
                recordSet.MutableMeta()->mutable_id(),
                recordSet.MutableSpec(),
                &zone,
                &changelistJson
            );
            recordSet.SetZone(zone);
            if (changelistJson.IsMap()) {
                recordSet.MutableChangelist()->FromJson(changelistJson);
            }
            result.RecordSets.emplace_back(std::move(recordSet));
        }
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TGetRecordSetsFromYPFailure(CurrentExceptionMessage()));
        TRateSensor(sensorGroup, NSensors::FAILURES).Inc();
        throw;
    }

    logFrame->LogEvent(TGetRecordSetsFromYPSuccess(result.YpTimestamp));
    TRateSensor(sensorGroup, NSensors::SUCCESSES).Inc();

    return result;
}

TGetRecordSetsResult TService::GetRecordSets(const TStringBuf cluster, const TVector<TString>& ids, const TDuration& maxAcceptableReplicaAge, TLogFramePtr logFrame, TSensorGroup sensorGroup) const {
    sensorGroup = TSensorGroup{sensorGroup, NSensors::GET_RECORD_SETS};

    const TYPReplica& replica = *Replicas_.at(cluster);
    const TYPReplicaSnapshot snapshot = replica.GetReplicaSnapshot();

    const TMaybe<TDuration> age = replica.Age(snapshot);
    logFrame->LogEvent(TReplicaAge(TString{cluster}, age.Defined() ? age->ToString() : "<UNDEFINED>", maxAcceptableReplicaAge.ToString()));

    if (age.Defined() && age < maxAcceptableReplicaAge) {
        return GetRecordSetsFromReplica(replica, snapshot, ids, logFrame, sensorGroup);
    } else {
        TRateSensor{sensorGroup, NSensors::FALLBACK_TO_YP}.Inc();

        NYP::NClient::TClientPtr client = YpClients_.at(cluster);
        return GetRecordSetsFromYP(client, ids, logFrame, sensorGroup);
    }
}

TFindRecordSetsResult TService::FindRecordSets(
    const TStringBuf cluster,
    const TVector<DNSName>& fqdns,
    const THashMap<DNSName, TVector<TRecordRequest*>>& requestsByFqdn,
    TLogFramePtr logFrame,
    TSensorGroup sensorGroup
) const {
    sensorGroup = TSensorGroup{sensorGroup, NSensors::FIND_RECORD_SETS};

    TVector<TString> ids;
    ids.reserve(2 * fqdns.size());

    TDuration maxAcceptableReplicaAge = TDuration::Max();
    for (const DNSName& fqdn : fqdns) {
        logFrame->LogEvent(TFindRecordSet(TString{fqdn.toStringNoDot()}));

        ids.emplace_back(fqdn.toString());
        ids.emplace_back(fqdn.toStringNoDot());

        const TZoneConfig* zoneConfig = requestsByFqdn.at(fqdn).front()->ZoneConfig;
        maxAcceptableReplicaAge = Min(maxAcceptableReplicaAge, TDuration::Parse(zoneConfig->GetMaxReplicaAgeForLookup()));
    }

    const TGetRecordSetsResult getRecordSetsResult = GetRecordSets(cluster, ids, maxAcceptableReplicaAge, logFrame, sensorGroup);

    THashMap<TString, TRecordSet> idToRecordSets;
    for (const TMaybe<TRecordSet>& recordSet : getRecordSetsResult.RecordSets) {
        if (!recordSet.Defined()) {
            continue;
        }

        TString id = recordSet->Meta().id();
        idToRecordSets.emplace(std::move(id), std::move(*recordSet));
    }

    TFindRecordSetsResult result;
    result.YpTimestamp = getRecordSetsResult.YpTimestamp;
    result.RecordSets.reserve(fqdns.size());
    for (size_t i = 0; i < fqdns.size(); ++i) {
        const TRecordSet* recordSet = nullptr;
        for (size_t j = 0; j < 2; ++j) {
            if (auto* it = idToRecordSets.FindPtr(ids[2 * i + j])) {
                recordSet = it;
                break;
            }
        }

        if (!recordSet) {
            logFrame->LogEvent(TRecordSetNotFound(TString{fqdns[i].toStringNoDot()}));
            result.RecordSets.push_back(Nothing());
        } else {
            logFrame->LogEvent(MakeRecordSetEvent<TRecordSetFound>(*recordSet));
            result.RecordSets.emplace_back(*recordSet);
        }
    }

    return result;
}

const TYPReplica& TService::GetReplica(const TString& cluster) const {
    const auto replicaPtr = Replicas_.FindPtr(cluster);
    Y_ENSURE(replicaPtr && replicaPtr->Get());
    return *replicaPtr->Get();
}

TDynamicZonesPtr TService::DynamicZones() const {
    TReadGuard guard(DynamicZonesMutex_);
    return DynamicZones_;
}

NYP::NYPReplica::TTableRulesHolder<NYP::NYPReplica::TDnsRecordSetReplicaObject> CreateReplicaRules() {
    NYP::NYPReplica::TTableRulesHolder<NYP::NYPReplica::TDnsRecordSetReplicaObject> rulesHolder;
    rulesHolder.AddRule<NYP::NYPReplica::TDnsRecordSetReplicaObject>(MakeHolder<TZoneTableRule>());
    return rulesHolder;
}

void TService::InitReplicas(const TString& ypToken) {
    const auto config = Config_.Get();
    for (const auto& clusterConfig : config->GetYpClusterConfigs()) {
        Replicas_.emplace(clusterConfig.GetName(), MakeHolder<TYPReplica>(
            Config_.Accessor<NYP::NYPReplica::TYPReplicaConfig>("YpReplicaConfig"),
            Config_.Accessor<NYP::NYPReplica::TYPClusterConfig>({"YpClusterConfigs", clusterConfig.GetName()}),
            CreateReplicaRules(),
            ypToken,
            ReplicaLogger_,
            STORAGE_FORMAT_VERSION
        ));
    }
}

void TService::StartReplicas() {
    TVector<NThreading::TFuture<void>> replicasStartFutures;
    for (auto& [cluster, replica] : Replicas_) {
        replicasStartFutures.push_back(NThreading::Async(
            [replicaPtr = replica.Get()] {
                replicaPtr->Start();
            },
            *ReplicasManagementPool_
        ));
    }
    NThreading::WaitExceptionOrAll(replicasStartFutures).GetValueSync();
}

void TService::StopReplicas() {
    TVector<NThreading::TFuture<void>> replicasStopFutures;
    for (auto& [cluster, replica] : Replicas_) {
        replicasStopFutures.push_back(NThreading::Async(
            [replicaPtr = replica.Get()] {
                replicaPtr->Stop();
            },
            *ReplicasManagementPool_
        ));
    }
    NThreading::WaitAll(replicasStopFutures).GetValueSync();
}

void TService::InitSensors() {
    const auto listProtoEnumNames = [](const NProtoBuf::EnumDescriptor* descriptor) -> TVector<TString> {
        TVector<TString> result(Reserve(descriptor->value_count()));
        for (size_t idx : xrange(descriptor->value_count())) {
            const NProtoBuf::EnumValueDescriptor* valueDescriptor = descriptor->value(idx);
            result.push_back(valueDescriptor->name());
        }
        return result;
    };

    /* update_records sensors */
    {
        TSensorGroup sensorGroup{SensorGroup_, NSensors::UPDATE_RECORDS};
        TRateSensor{sensorGroup, NSensors::REQUESTS};
        TRateSensor{TSensorGroup{sensorGroup, NSensors::UPDATE_RECORD}, NSensors::REQUESTS};
        TRateSensor{TSensorGroup{sensorGroup, NSensors::REMOVE_RECORD}, NSensors::REQUESTS};

        TSensorGroup statusSensorGroup{sensorGroup, NSensors::STATUS};
        for (const TString& status : listProtoEnumNames(NApi::TRspUpdateRecord::EUpdateRecordStatus_descriptor())) {
            TRateSensor{statusSensorGroup, status};
        }
        for (const TString& status : listProtoEnumNames(NApi::TRspRemoveRecord::ERemoveRecordStatus_descriptor())) {
            TRateSensor{statusSensorGroup, status};
        }
        TRateSensor{sensorGroup, NSensors::SUCCESSES};
        TRateSensor{sensorGroup, NSensors::FAILURES};
    }

    /* list_zone_record_sets sensors */
    {
        for (const auto& [zoneName, zoneConfig] : ZoneConfigs_) {
            TSensorGroup sensorGroup(SensorGroup_, NSensors::LIST_ZONE_RECORD_SETS);
            sensorGroup.AddLabel("zone", TString{zoneName.toStringNoDot()});

            TRateSensor{sensorGroup, NSensors::REQUESTS};

            TSensorGroup statusSensorGroup{sensorGroup, NSensors::STATUS};
            for (const TString& status : listProtoEnumNames(NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_descriptor())) {
                TRateSensor{statusSensorGroup, status};
            }
        }
    }

    /* histograms */
    TVector<NLifetimeHistogram::TResponseTimeHistogramOptions> histogramOptions;
    histogramOptions.reserve(NSensors::RESPONSE_TIME_HISTOGRAM_INIT_PARAMETERS.size());
    for (const auto& [operationType, bucketBounds, responseSizeTypes] : NSensors::RESPONSE_TIME_HISTOGRAM_INIT_PARAMETERS) {
        histogramOptions.push_back({operationType, NMonitoring::TBucketBounds(bucketBounds), TVector<std::pair<NLifetimeHistogram::EResponseSizeType, ui64>>(responseSizeTypes)});
    }
    ResponseTimeHistograms_.Reset(new NLifetimeHistogram::TResponseTimeHistogramsHolder(SensorGroup_, histogramOptions));
}

TLogger* TService::GetLogger() {
    return &Logger_;
}

} // namespace NInfra::NYpDnsApi
