#pragma once

#include <infra/yp_dns_api/bridge/api/api.pb.h>
#include <infra/yp_dns_api/bridge/api/internal.pb.h>
#include <infra/yp_dns_api/bridge/cluster_scoring/scoring.h>
#include <infra/yp_dns_api/bridge/config/config.pb.h>
#include <infra/yp_dns_api/bridge/grpc/server.h>
#include <infra/yp_dns_api/bridge/misc/banned_clusters.h>
#include <infra/yp_dns_api/bridge/misc/record_request.h>
#include <infra/yp_dns_api/bridge/sensors/sensors.h>
#include <infra/yp_dns_api/bridge/service_iface/service_iface.h>

#include <infra/libs/http_service/service.h>
#include <infra/libs/logger/logger.h>
#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/updatable_proto_config/accessor.h>
#include <infra/libs/updatable_proto_config/holder.h>
#include <infra/libs/yp_dns/dynamic_zones/client.h>
#include <infra/libs/yp_dns/dynamic_zones/zones_manager_service/service/service.h>
#include <infra/libs/yp_dns/dynamic_zones/zones_state_coordinator.h>
#include <infra/libs/yp_dns/record_set/record_set.h>
#include <infra/libs/yp_replica/yp_replica.h>

#include <yp/cpp/yp/client.h>
#include <yp/cpp/yp/data_model.h>
#include <yp/cpp/yp/fwd.h>

#include <util/generic/maybe.h>
#include <util/system/rwlock.h>

namespace NInfra::NYpDnsApi {

using TYPReplica = NYP::NYPReplica::TYPReplica<NYP::NYPReplica::TDnsRecordSetReplicaObject>;
using TYPReplicaSnapshot = TYPReplica::TReplicaSnapshot;
using TListOptions = TYPReplica::TListOptions<NYP::NYPReplica::TDnsRecordSetReplicaObject>;
using TBridgeConfigPtr = TAtomicSharedPtr<const TBridgeConfig>;

struct TGetRecordSetsResult {
    ui64 YpTimestamp = 0;
    TVector<TMaybe<NYpDns::TRecordSet>> RecordSets;
};

struct TFindRecordSetsResult {
    ui64 YpTimestamp = 0;
    TVector<TMaybe<NYpDns::TRecordSet>> RecordSets;
};

struct TContinuationListOptions : public TListOptions {
public:
    TContinuationListOptions() = default;
    TContinuationListOptions(const TContinuationListOptions&) = default;
    TContinuationListOptions(TContinuationListOptions&&) = default;
    TContinuationListOptions(TListOptions listOptions)
        : TListOptions(std::move(listOptions))
    {
    }

    TContinuationListOptions& operator=(const TContinuationListOptions&) = default;
    TContinuationListOptions& operator=(TContinuationListOptions&&) = default;

public:
    enum class EListFrom {
        REPLICA = 0,
        YP = 1,
    };

    TMaybe<EListFrom> ListFrom;
    ui64 YpTimestamp = 0;
    bool HasReachedEnd = false;
};

struct TListRecordSetObjectsResult {
    ui64 YpTimestamp = 0;
    TVector<NYpDns::TSerializedRecordSet> RecordSetObjects;
    TContinuationListOptions ContinuationOptions;
};

struct TListRecordSetObjectsFromClustersResult {
    TVector<NYpDns::TRecordSet> RecordSetObjects;
    TContinuationListOptions ContinuationOptions;
    THashMap<TString, TContinuationListOptions> ClustersContinuationOptions;
};

class TService: public IService {
public:
    TService(const TBridgeConfig& config);

    ~TService();

    void Start() override;

    void Wait() override;

    void Ping(TRequestPtr<NApi::TReqPing> request, TReplyPtr<NApi::TRspPing> reply) override;

    void ReopenLog(TRequestPtr<NApi::TReqReopenLog> request, TReplyPtr<NApi::TRspReopenLog> reply) override;

    void Shutdown(TRequestPtr<NApi::TReqShutdown> request, TReplyPtr<NApi::TRspShutdown> reply) override;

    void Sensors(TRequestPtr<NApi::TReqSensors> request, TReplyPtr<NApi::TRspSensors> reply) override;

    void UpdateRecords(TRequestPtr<NApi::TReqUpdateRecords> request, TReplyPtr<NApi::TRspUpdateRecords> reply) override;

    void ListZoneRecordSets(TRequestPtr<NApi::TReqListZoneRecordSets> request, TReplyPtr<NApi::TRspListZoneRecordSets> reply) override;

    void ListZones(NInfra::TRequestPtr<NYpDns::NDynamicZones::NApi::TReqListZones> request, NInfra::TReplyPtr<NYpDns::NDynamicZones::NApi::TRspListZones> reply) override;

    void CreateZone(NInfra::TRequestPtr<NYpDns::NDynamicZones::NApi::TReqCreateZone> request, NInfra::TReplyPtr<NYpDns::NDynamicZones::NApi::TRspCreateZone> reply) override;

    void RemoveZone(NInfra::TRequestPtr<NYpDns::NDynamicZones::NApi::TReqRemoveZone> request, NInfra::TReplyPtr<NYpDns::NDynamicZones::NApi::TRspRemoveZone> reply) override;

private:
    void InitRecordSet(TMaybe<NYpDns::TRecordSet>& recordSet, const TRecordRequest& request) const;

    void UpdateRecords(
        const THashMap<DNSName, TVector<TRecordRequest*>>& requestsByFqdn,
        const TVector<DNSName>& fqdns,
        const TClusterDescriptor& cluster,
        TLogFramePtr logFrame,
        TSensorGroup sensorGroup
    ) const;

    bool CheckChangelist(const TMaybe<NYpDns::TRecordSet>& recordSet, const TVector<TRecordRequest*>& requests, const TClusterDescriptor& cluster, TLogFramePtr logFrame) const;

    void UpdateRecordSet(
        TMaybe<NYpDns::TRecordSet>&& recordSet,
        const TVector<TRecordRequest*>& requests,
        TVector<NYpDns::TRecordSet>& creates,
        TVector<NYpDns::TRecordSet>& updates,
        TVector<NYpDns::TRecordSet>& removes,
        TLogFramePtr logFrame,
        TSensorGroup sensorGroup
    ) const;

    bool UpdateRecord(TMaybe<NYpDns::TRecordSet>& recordSet, TRecordRequest& request, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;
    bool RemoveRecord(TMaybe<NYpDns::TRecordSet>& recordSet, TRecordRequest& request, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TVector<TClusterDescriptor> GetOrderedClusters(const THashMap<TString, THashSet<DNSName>>& fqdnsByCluster) const;

    ui64 GenerateYpTimestamp(NYP::NClient::TClientPtr client, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    NYP::NClient::TTransactionPtr CreateTransaction(NYP::NClient::TTransactionFactoryPtr transactionFactory, ui64 timestamp, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;
    void CommitTransaction(NYP::NClient::TTransactionPtr transaction, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    void CreateRecordSets(NYP::NClient::TTransactionPtr transaction, const TVector<NYpDns::TRecordSet>& recordSet, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;
    void UpdateRecordSets(NYP::NClient::TTransactionPtr transaction, const TVector<NYpDns::TRecordSet>& recordSet, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;
    void RemoveRecordSets(NYP::NClient::TTransactionPtr transaction, const TVector<NYpDns::TRecordSet>& recordSet, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TVector<NYP::NClient::TSelectorResult> SelectZoneRecordSets(NYP::NClient::TClientPtr client, const ui64 ypTimestamp, const DNSName& zone, const TListOptions& listOptions, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TListRecordSetObjectsResult ListZoneRecordSetsFromReplica(const TYPReplica& replica, const DNSName& zone, TContinuationListOptions listOptions, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TListRecordSetObjectsResult ListZoneRecordSetsFromYP(const NYP::NClient::TClientPtr client, const DNSName& zone, TContinuationListOptions listOptions, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TListRecordSetObjectsFromClustersResult ListZoneRecordSetsFromClusters(
        const DNSName& zone,
        const TZoneConfig& zoneConfig,
        const TVector<TClusterId>& clusters,
        TContinuationListOptions listOptions,
        THashMap<TString, TContinuationListOptions> clustersListOptions,
        TLogFramePtr logFrame,
        TSensorGroup sensorGroup
    ) const;

    TListRecordSetObjectsResult ListZoneRecordSetsFromCluster(
        const TString& cluster,
        const TYPReplica& replica,
        NYP::NClient::TClientPtr ypClient,
        const DNSName& zone,
        const TZoneConfig& zoneConfig,
        const TContinuationListOptions& listOptions,
        TLogFramePtr logFrame,
        TSensorGroup sensorGroup
    ) const;

    TVector<NYP::NClient::TSelectorResult> GetRecordSets(NYP::NClient::TClientPtr client, const TVector<TString>& ids, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TGetRecordSetsResult GetRecordSetsFromReplica(const TYPReplica& replica, const TYPReplicaSnapshot& snapshot, const TVector<TString>& ids, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TGetRecordSetsResult GetRecordSetsFromYP(NYP::NClient::TClientPtr client, const TVector<TString>& ids, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TGetRecordSetsResult GetRecordSets(const TStringBuf cluster, const TVector<TString>& ids, const TDuration& maxAcceptableReplicaAge, TLogFramePtr logFrame, TSensorGroup sensorGroup) const;

    TFindRecordSetsResult FindRecordSets(
        const TStringBuf cluster,
        const TVector<DNSName>& fqdns,
        const THashMap<DNSName, TVector<TRecordRequest*>>& requestsByFqdn,
        TLogFramePtr logFrame,
        TSensorGroup sensorGroup
    ) const;

    const TYPReplica& GetReplica(const TString& cluster) const;

    TDynamicZonesPtr DynamicZones() const;

private:
    void InitReplicas(const TString& ypToken);
    void StartReplicas();
    void StopReplicas();

    void InitSensors();

private:
    TLogger* GetLogger() override;

private:
    NUpdatableProtoConfig::TConfigHolderPtr<TBridgeConfig> ConfigHolder_;
    NUpdatableProtoConfig::TAccessor<TBridgeConfig> Config_;

    TLogger Logger_;
    TLogFramePtr ServiceLogFrame_;

    const TSensorGroup SensorGroup_;

    THolder<NLifetimeHistogram::TResponseTimeHistogramsHolder> ResponseTimeHistograms_;

    NInfra::THttpService AdminHttpService_;
    NInfra::THttpService BridgeHttpService_;
    TGrpcServer GrpcServer_;

    TRWMutex DynamicZonesMutex_;
    TDynamicZonesPtr DynamicZones_;

    TAtomicSharedPtr<NYpDns::NDynamicZones::TBridgeServiceClient> BridgeStateServiceClient_;
    NYpDns::NDynamicZones::TClient ZonesManagerClient_;

    TLogger PollDynamicZonesLogger_;
    THolder<IBackgroundThread> DynamicZonesUpdater_;

    TAtomicSharedPtr<NYpDns::NDynamicZones::TZonesStateCoordinator> ZonesStateCoordinator_;
    NYpDns::NDynamicZones::TZonesManagerService ZonesManagerService_;

    TClustersScoring ClustersScoring_;

    TZoneConfigs ZoneConfigs_;
    THashMap<TString, NYP::NClient::TClientPtr> YpClients_;
    THashMap<TString, NYP::NClient::TTransactionFactoryPtr> YpTransactionFactories_;

    THolder<IThreadPool> ReplicasManagementPool_;
    TLogger ReplicaLogger_;
    THashMap<TString, THolder<TYPReplica>> Replicas_;

    TBannedClusters BannedClusters_;
};

} // namespace NInfra::NYpDnsApi
