#include "clients_registry.h"

#include "execute.h"

namespace NInfra::NYpDnsApi {

////////////////////////////////////////////////////////////////////////////////

namespace {

NYP::NClient::TClientPtr MakeSnapshotClient(NYP::NClient::TClientPtr client, ui64 timestamp) {
    NYP::NClient::TClientOptions options = client->Options();
    options.SetSnapshotTimestamp(timestamp);
    return NYP::NClient::CreateClient(options);
}

} // anonymous namespace

////////////////////////////////////////////////////////////////////////////////

TYpClientsRegistry::TYpClientsRegistry(NYP::NClient::TClientOptions defaultClientOptions)
    : DefaultClientOptions_(std::move(defaultClientOptions))
{
}

std::pair<NYP::NClient::TClientPtr, bool> TYpClientsRegistry::Register(const TString& key, NYP::NClient::TClientPtr client) {
    const auto [it, inserted] = emplace(key, client);
    return {it->second, inserted};
}

std::pair<NYP::NClient::TClientPtr, bool> TYpClientsRegistry::Register(const TClusterOptions& clusterOptions) {
    NYP::NClient::TClientOptions clientOptions = DefaultClientOptions_;
    clientOptions.SetAddress(clusterOptions.Address)
                 .SetEnableSsl(clusterOptions.EnableSsl)
                 .SetReadOnlyMode(clusterOptions.ReadOnly)
                 .SetTimeout(clusterOptions.Timeout)
                 .SetToken(clusterOptions.Token);
    return Register(clusterOptions.Name, NYP::NClient::CreateClient(clientOptions));
}

NYP::NClient::TClientPtr TYpClientsRegistry::Get(const TString& key) const {
    const auto it = find(key);
    if (it == end()) {
        ythrow yexception() << "no client with key \"" << key << "\"";
    }

    return it->second;
}

TYpClientsRegistry TYpClientsRegistry::CreateSnapshot(const THashMap<TString, ui64>& timestamps) const {
    TYpClientsRegistry result(DefaultClientOptions_);
    for (const auto& [key, client] : *this) {
        const auto* it = timestamps.FindPtr(key);
        const ui64 timestamp = it ? *it : GenerateTimestamp(client);
        result.Register(key, MakeSnapshotClient(client, timestamp));
    }
    return result;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra::NYpDnsApi
