#include "execute.h"

#include <infra/yp_dns_api/libs/yp/protos/events/events_decl.ev.pb.h>

namespace NInfra::NYpDnsApi {

namespace {

NEventlog::TSelectObjectsOptions MakeSelectObjectsOptionsEvent(const NYP::NClient::TSelectObjectsOptions& options) {
    NEventlog::TSelectObjectsOptions event;
    event.SetLimit(options.Limit());
    event.SetOffset(options.Offset());
    event.SetContinuationToken(options.ContinuationToken());
    return event;
}

NEventlog::TSelectObjectsRequest MakeSelectObjectsRequestEvent(
    NYP::NClient::TClientPtr client,
    NYP::NClient::NApi::NProto::EObjectType objectType,
    const TVector<TString>& selectors,
    const TString& filter,
    const NYP::NClient::TSelectObjectsOptions& options,
    const ui64 timestamp
) {
    NEventlog::TSelectObjectsRequest event;
    event.SetObjectType(NYP::NClient::NApi::NProto::EObjectType_Name(objectType));
    event.MutableSelectors()->Reserve(selectors.size());
    for (const TString& selector : selectors) {
        *event.AddSelectors() = selector;
    }
    event.SetFilter(filter);
    *event.MutableOptions() = MakeSelectObjectsOptionsEvent(options);
    event.SetTimestamp(timestamp ? timestamp : client->Options().SnapshotTimestamp());
    return event;
}

NEventlog::TAggregateObjectsRequest MakeAggregateObjectsRequestEvent(
    NYP::NClient::TClientPtr client,
    NYP::NClient::NApi::NProto::EObjectType objectType,
    const TVector<TString>& groupByExpressions,
    const TVector<TString>& aggregateExpressions,
    const TString& filter,
    const ui64 timestamp
) {
    NEventlog::TAggregateObjectsRequest event;
    event.SetObjectType(NYP::NClient::NApi::NProto::EObjectType_Name(objectType));
    event.MutableGroupByExpressions()->Reserve(groupByExpressions.size());
    for (const TString& expr : groupByExpressions) {
        event.AddGroupByExpressions(expr);
    }
    event.MutableAggregateExpressions()->Reserve(aggregateExpressions.size());
    for (const TString& expr : aggregateExpressions) {
        event.AddAggregateExpressions(expr);
    }
    event.SetFilter(filter);
    event.SetTimestamp(timestamp ? timestamp : client->Options().SnapshotTimestamp());
    return event;
}

} // anonymous namespace

////////////////////////////////////////////////////////////////////////////////

void ExecuteYpRequest(const std::function<void()>& requestFunc, const std::function<void(const NYP::NClient::TResponseError&)>& onError) {
    DoWithRetry<NYP::NClient::TResponseError>(
        requestFunc,
        onError,
        TRetryOptions().WithCount(3)
                       .WithSleep(TDuration::MilliSeconds(250))
                       .WithIncrement(TDuration::MilliSeconds(250)),
        /* throwLast = */ true
    );
}

////////////////////////////////////////////////////////////////////////////////

ui64 GenerateTimestamp(NYP::NClient::TClientPtr client) {
    return ExecuteYpRequest<ui64>(
        [client] {
            return client->GenerateTimestamp().GetValue(client->Options().Timeout() * 2);
        },
        [](const NYP::NClient::TResponseError&) {}
    );
}

////////////////////////////////////////////////////////////////////////////////

NYP::NClient::TSelectObjectsResult SelectObjects(
    NYP::NClient::TClientPtr client,
    NYP::NClient::NApi::NProto::EObjectType objectType,
    const TVector<TString>& selectors,
    const TString& filter,
    const NYP::NClient::TSelectObjectsOptions& options,
    const ui64 timestamp,
    NInfra::TLogFramePtr logFrame
) {
    try {
        return ExecuteYpRequest<NYP::NClient::TSelectObjectsResult>(
            [client, objectType, &selectors, &filter, &options, timestamp, logFrame] {
                if (logFrame) {
                    logFrame->LogEvent(MakeSelectObjectsRequestEvent(client, objectType, selectors, filter, options, timestamp));
                }
                return client->SelectObjects(objectType, selectors, filter, options, timestamp).GetValue(client->Options().Timeout() * 2);
            },
            [logFrame](const NYP::NClient::TResponseError& ex) {
                if (logFrame) {
                    logFrame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TSelectObjectsError(ex.what()));
                }
            }
        );
    } catch (...) {
        if (logFrame) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TSelectObjectsFailure(CurrentExceptionMessage()));
        }
        throw;
    }
}

////////////////////////////////////////////////////////////////////////////////

TVector<NYP::NClient::TSelectorResult> AggregateObjects(
    NYP::NClient::TClientPtr client,
    NYP::NClient::NApi::NProto::EObjectType objectType,
    const TVector<TString>& groupByExpressions,
    const TVector<TString>& aggregateExpressions,
    const TString& filter,
    const ui64 timestamp,
    NInfra::TLogFramePtr logFrame
) {
    try {
        return ExecuteYpRequest<TVector<NYP::NClient::TSelectorResult>>(
            [client, objectType, &groupByExpressions, &aggregateExpressions, &filter, timestamp, logFrame] {
                if (logFrame) {
                    logFrame->LogEvent(MakeAggregateObjectsRequestEvent(client, objectType, groupByExpressions, aggregateExpressions, filter, timestamp));
                }
                return client->AggregateObjects(objectType, groupByExpressions, aggregateExpressions, filter, timestamp).GetValue(client->Options().Timeout() * 2);
            },
            [logFrame](const NYP::NClient::TResponseError& ex) {
                if (logFrame) {
                    logFrame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TAggregateObjectsRequestError(ex.what()));
                }
            }
        );
    } catch (...) {
        if (logFrame) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TAggregateObjectsRequestFailure(CurrentExceptionMessage()));
        }
        throw;
    }
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra::NYpDnsApi
