#include "get_objects.h"

#include "execute.h"

namespace NInfra::NYpDnsApi {

constexpr ui64 DEFAULT_MAX_SELECT_LIMIT = 2000;

////////////////////////////////////////////////////////////////////////////////

NEventlog::TGetObjectsArgument MakeGetObjectsArgumentEvent(const NController::IObjectManager::TSelectArgument& arg) {
    NEventlog::TGetObjectsArgument result;
    if (arg.ClusterName.Defined()) {
        result.SetCluster(*arg.ClusterName);
    }
    result.SetObjectType(NYP::NClient::NApi::NProto::EObjectType_Name(arg.ObjectType));
    result.MutableSelectors()->Reserve(arg.Selector.size());
    for (const TString& selector : arg.Selector) {
        *result.AddSelectors() = selector;
    }
    result.SetFilter(arg.Filter);
    result.SetLimit(arg.Options.Limit());
    if (arg.TotalLimit.Defined()) {
        result.SetTotalLimit(*arg.TotalLimit);
    }
    return result;
}

NEventlog::TGetObjectsArgument MakeGetObjectsArgumentEvent(const TGetObjectsArgument& arg) {
    NEventlog::TGetObjectsArgument result;
    result.SetCluster(arg.Cluster);
    result.SetObjectType(NYP::NClient::NApi::NProto::EObjectType_Name(arg.ObjectType));
    result.MutableSelectors()->Reserve(arg.Selectors.size());
    for (const TString& selector : arg.Selectors) {
        *result.AddSelectors() = selector;
    }
    result.SetFilter(arg.Filter);
    if (arg.Limit.Defined()) {
        result.SetLimit(*arg.Limit);
    }
    if (arg.TotalLimit.Defined()) {
        result.SetTotalLimit(*arg.TotalLimit);
    }
    return result;
}

////////////////////////////////////////////////////////////////////////////////

TGetObjectsResult GetObjects(const NYP::NClient::TClientPtr client, const TGetObjectsArgument& arg, NInfra::TLogFramePtr logFrame) {
    Y_ENSURE(client->Options().SnapshotTimestamp() > 0);

    logFrame->LogEvent(NEventlog::TGetObjects(client->Options().Address(), MakeGetObjectsArgumentEvent(arg)));

    TGetObjectsResult result;

    NYP::NClient::TSelectObjectsOptions selectOptions;
    selectOptions.SetLimit(arg.Limit.GetOrElse(DEFAULT_MAX_SELECT_LIMIT));
    while (arg.TotalLimit != 0u) {
        if (arg.TotalLimit.Defined() && arg.Limit != 0u) {
            selectOptions.SetLimit(Min(*arg.TotalLimit - result.Results.size(), selectOptions.Limit()));
        }
        NYP::NClient::TSelectObjectsResult selectResult = SelectObjects(
            client,
            arg.ObjectType,
            arg.Selectors,
            arg.Filter,
            selectOptions,
            /* timestamp */ 0,
            logFrame
        );

        if (result.Timestamp == 0) {
            result.Timestamp = selectResult.Timestamp;
        } else {
            Y_ENSURE(result.Timestamp == selectResult.Timestamp);
        }
        std::move(selectResult.Results.begin(), selectResult.Results.end(), std::back_inserter(result.Results));

        if (!selectOptions.Limit() || selectResult.Results.size() < selectOptions.Limit() || (arg.TotalLimit.Defined() && result.Results.size() >= arg.TotalLimit)) {
            break;
        }

        selectOptions.SetContinuationToken(selectResult.ContinuationToken);
    }

    logFrame->LogEvent(NEventlog::TGetObjectsResult(client->Options().Address(), result.Results.size()));

    return result;
}

TGetObjectsResults GetObjects(const TYpClientsRegistry& clientsRegistry, const TGetObjectsArguments& args, NInfra::TLogFramePtr logFrame) {
    TGetObjectsResults result;
    result.reserve(args.size());
    for (const TGetObjectsArgument& arg : args) {
        result.emplace_back(GetObjects(clientsRegistry.Get(arg.Cluster), arg, logFrame));
    }
    return result;
}

} // namespace NInfra::NYpDnsApi
