import argparse
import json
import juggler_sdk
import logging
import sys
import typing as tp

from infra.yp_service_discovery.monitoring.solomon.src.util.solomon_api import SolomonApi


JUGGLER_NAMESPACE = 'yp.dns.api'
JUGGLER_COMMON_HOST = 'yp-dns-api'

SOLOMON_PROJECT_ID = 'yp_dns_api'

YP_CLUSTERS = [
    'sas-test',
    'man-pre',
    'sas',
    'man',
    'vla',
    'myt',
    'iva',
    'xdc',
]

DEBUG = 'debug'
WARN = 'warn'
CRIT = 'crit'

DEFAULT_NOTIFY_PARAMS = {
    'logins': {
        DEBUG: [
        ],
        WARN: [
            '@svc_yp_dns:yp-dns-duty',
            'dns-monitoring',
        ],
        CRIT: [
            '@svc_yp_dns:yp-dns-duty',
            'dns-monitoring',
        ],
    },
    'escalations': {
        DEBUG: [
        ],
        WARN: [
        ],
        CRIT: [
        ],
    },
}


def microseconds(seconds=0, millis=0):
    return seconds * 1000 * 1000 + millis * 1000


def milliseconds(seconds=0, minutes=0):
    return seconds * 1000 + minutes * 60 * 1000


def normalize_name(name: str) -> str:
    return name.lower().replace('_', '-')


def create_selectors(**kwargs):
    def create_selector(key, value):
        if value.startswith('!'):
            return '{}!="{}"'.format(key, value[1:])
        return '{}="{}"'.format(key, value)

    selectors = ', '.join(map(lambda kv: create_selector(*kv), kwargs.items()))
    return '{' + selectors + '}'


def get_solomon_alert_name(name: str, solomon_cluster: dict):
    full_name = ''
    if 'alert_name_prefix' in solomon_cluster['solomon']:
        full_name += solomon_cluster['solomon']['alert_name_prefix'] + ' '
    full_name += name
    return full_name


def get_solomon_threshold_alert(name: str, title_name: str, selectors: dict,
                                solomon_cluster: dict, description: tp.Optional[str] = None,
                                group_by_labels: list = [],
                                time_aggregation: str = 'SUM', predicate: str = 'GT',
                                threshold: float = 0, period_ms: int = milliseconds(minutes=3),
                                delay_s: int = 30):
    if description is None:
        description = title_name

    service = name
    alert_id = f"{solomon_cluster['name']}.{service}"

    default_selectors = {
        'cluster': solomon_cluster['solomon']['cluster'],
        'service': solomon_cluster['solomon']['service'],
    }
    if 'host' not in group_by_labels:
        default_selectors['host'] = 'all'

    return {
        'id': alert_id,
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_solomon_alert_name(title_name, solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': create_selectors(**(default_selectors | selectors)),
                'timeAggregation': time_aggregation,
                'predicate': predicate,
                'threshold': threshold,
            }
        },
        'groupByLabels': group_by_labels,
        'periodMillis': period_ms,
        'delaySeconds': delay_s,
        'annotations': {
            'service': service,
            'host': f"{solomon_cluster['name']}.{JUGGLER_COMMON_HOST}",
            'alert_description': description,
        },
    }


def get_solomon_expression_alert(name: str, title_name: str,
                                 program: str, check_expression: str,
                                 solomon_cluster: dict, description: tp.Optional[str] = None,
                                 annotations: dict = {},
                                 group_by_labels: list = [],
                                 period_ms: int = milliseconds(minutes=3),
                                 delay_s: int = 30):
    if description is None:
        description = title_name

    service = name
    alert_id = f"{solomon_cluster['name']}.{service}"

    annotations |= {
        'service': service,
        'host': f"{solomon_cluster['name']}.{JUGGLER_COMMON_HOST}",
        'alert_description': description,
    }

    return {
        'id': alert_id,
        'projectId': SOLOMON_PROJECT_ID,
        'name': get_solomon_alert_name(title_name, solomon_cluster),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'expression': {
                'program': program,
                'checkExpression': check_expression,
            }
        },
        'groupByLabels': group_by_labels,
        'periodMillis': period_ms,
        'delaySeconds': delay_s,
        'annotations': annotations,
    }


def get_yp_unreach_checks(*yp_clusters):
    return ['yp-{}.yp.yandex.net:yp.infra.auto_downtime_'.format(yp_cluster) for yp_cluster in yp_clusters]


def get_nanny_service_url(service):
    return {
        'url': 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}'.format(service),
        'title': 'Nanny: {}'.format(service),
        'type': 'nanny',
    }


def get_docs_url():
    return {
        'url': 'https://wiki.yandex-team.ru/ypdns/monitoring/#opisaniealertov',
        'title': 'Описание алертов',
        'type': 'wiki',
    }


def get_source_url():
    return {
        'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/infra/yp_dns_api/monitoring/alerts',
        'title': 'Alerts generation source code',
    }


def get_standard_urls():
    return [
        get_nanny_service_url('yp-dns-api-bridge'),
        get_nanny_service_url('yp-dns-api-replicator'),
        get_docs_url(),
        get_source_url(),
    ]


def notification(status, logins, notify_methods):
    template_kwargs = {
        "status": status,
        "login": logins,
        "method": notify_methods,
    }

    if (isinstance(status, str) and status == "CRIT") or (isinstance(status, dict) and status["to"] == "CRIT"):
        template_kwargs["repeat"] = 1800  # 30 min

    return juggler_sdk.NotificationOptions(
        template_name="on_status_change",
        template_kwargs=template_kwargs,
    )


def get_notifications_config(notify_config, level=CRIT):
    logins = notify_config['logins'][level]
    escalation_logins = notify_config['escalations'][level]

    result = []
    if logins:
        result.extend([
            notification(
                status={'from': 'OK', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ]
            ),
            notification(
                status={'from': 'WARN', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ]
            ),
            notification(
                status={'from': 'CRIT', 'to': 'OK'},
                logins=logins,
                notify_methods=[
                    'telegram',
                ]
            ),
        ])

    if escalation_logins:
        result.extend([
            juggler_sdk.NotificationOptions(
                template_name='phone_escalation',
                template_kwargs={
                    'delay': 60,
                    'logins': escalation_logins,
                }
            ),
        ])

    return result


def get_juggler_check(*, host=None, service=None, solomon_alert=None, notification_level=CRIT,
                      unreach_checks=[], notify_params=DEFAULT_NOTIFY_PARAMS, flaps_config=None):
    if solomon_alert is not None:
        host = solomon_alert['annotations']['host']
        service = solomon_alert['annotations']['service']

    check = juggler_sdk.Check(
        host=host,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        aggregator='logic_or',
        aggregator_kwargs={
            'unreach_checks': unreach_checks,
        },
        notifications=get_notifications_config(notify_params, notification_level),
        flaps_config=flaps_config,
    )

    check.meta['urls'] = get_standard_urls()

    return check


def update_juggler_check(juggler_client, check, force=False):
    if isinstance(check, juggler_sdk.CheckFilter):
        result = juggler_client.remove_checks([check]).results[0]
        logging.info('Removed juggler check {}:{}: {}'.format(check.host, check.service, 'Success' if result.success else 'Failed'))
    else:
        diff = juggler_client.upsert_check(check, force=force).diff.to_dict()
        logging.info('Updated juggler check {}:{}'.format(check.host, check.service))
        logging.info('Diff:\n{}'.format(json.dumps(diff, indent=2)))


def update_multicluster_updates_monitoring_alerts(juggler_client):
    def get_unexpected_state_check():
        return get_juggler_check(
            host='multicluster_updates.yp_dns_api',
            service='unexpected_state',
            unreach_checks=get_yp_unreach_checks('sas-test', 'man-pre')
        )

    def get_incorrect_dns_answer_check():
        return get_juggler_check(
            host='multicluster_updates.yp_dns_api',
            service='incorrect_dns_answer',
            unreach_checks=get_yp_unreach_checks('sas-test', 'man-pre')
        )

    update_juggler_check(juggler_client, get_unexpected_state_check())
    update_juggler_check(juggler_client, get_incorrect_dns_answer_check())


def get_update_records_status_alert(solomon_cluster: dict, status: str) -> tuple[dict, juggler_sdk.Check]:
    name = normalize_name(f"update-records-{status}")
    solomon_alert = get_solomon_threshold_alert(
        name=name,
        title_name=f"Update records with status {status}",
        solomon_cluster=solomon_cluster,
        selectors={
            'sensor': f'bridge.service.update_records.status.{status}',
        },
        threshold=10,
    )
    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        flaps_config=juggler_sdk.FlapOptions(stable=300, critical=900, boost=0),
    )
    return solomon_alert, juggler_check


def get_list_zone_record_sets_status_alert(solomon_cluster: dict, status: str) -> tuple[dict, juggler_sdk.Check]:
    name = normalize_name(f"list-zone-record-sets-{status}")
    solomon_alert = get_solomon_threshold_alert(
        name=name,
        title_name=f"Record sets listings with status {status}",
        description=f"Record sets listings for zone {{{{labels.zone}}}} with status {status}",
        solomon_cluster=solomon_cluster,
        selectors={
            'sensor': f'bridge.service.list_zone_record_sets.status.{status}',
        },
        group_by_labels=[
            'zone',
        ],
    )
    juggler_check = get_juggler_check(solomon_alert=solomon_alert)
    return solomon_alert, juggler_check


def get_lock_acquired_alert(solomon_cluster: dict) -> tuple[dict, juggler_sdk.Check]:
    name = normalize_name("lock-is-not-acquired")
    solomon_alert = get_solomon_threshold_alert(
        name=name,
        title_name="Failed to acquire YT lock",
        solomon_cluster=solomon_cluster,
        selectors={
            'sensor': 'controller.lock_acquired',
        },
        time_aggregation='LAST_NON_NAN',
        predicate='EQ',
        threshold=0,
    )
    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        flaps_config=juggler_sdk.FlapOptions(stable=300, critical=900, boost=0),
    )
    return solomon_alert, juggler_check


def get_no_successful_sync_cycles_alert(solomon_cluster: dict, replication_group_and_number_of_shards: tuple[str, int]) -> tuple[dict, juggler_sdk.Check]:
    replication_group, number_of_shards = replication_group_and_number_of_shards
    def build_shard_sensor(shard_id: int) -> str:
        return f"let sensors_shard_{shard_id} = group_lines('sum', \
                {{service='replicator', \
                sensor='{replication_group}_shard_id_{shard_id}.{replication_group}_shard_id_{shard_id}_successful_sync_cycles', \
                host=all}});"

    if number_of_shards == 1:
        name = normalize_name(f"no-successful-sync-cycles-{replication_group}")
        solomon_alert = get_solomon_threshold_alert(
            name=name,
            title_name=f"No successful sync cycles for group {replication_group}",
            solomon_cluster=solomon_cluster,
            selectors={
                'sensor': f'controller.{replication_group}_successful_sync_cycles',
            },
            time_aggregation='SUM',
            period_ms=milliseconds(minutes=10),
            predicate='EQ',
            threshold=0,
        )
    else:
        name = normalize_name(f"no-successful-sync-cycles-{replication_group}-on-shards")
        solomon_alert = get_solomon_expression_alert(
            name=name,
            title_name=f"No successful sync cycles on shards for group {replication_group}",
            solomon_cluster=solomon_cluster,
            program='\n'.join((build_shard_sensor(i) for i in range(number_of_shards))),
            check_expression=' || '.join(f'sum(sensors_shard_{i}) == 0' for i in range(number_of_shards)),
            annotations={
                "service": f"no-successful-sync-cycles-on-shards-{replication_group}",
                "host": "replicator.yp-dns-api",
                "alert_description": f"No successful sync cycles on shards for group {replication_group}"
            },
        )

    juggler_check = get_juggler_check(solomon_alert=solomon_alert)
    return solomon_alert, juggler_check


def get_failed_sync_cycles_alert(solomon_cluster: dict, replication_group_and_number_of_shards: tuple[str, int]) -> tuple[dict, juggler_sdk.Check]:
    replication_group, number_of_shards = replication_group_and_number_of_shards
    def build_shard_sensor(shard_id: int) -> str:
        return f"let sensors_shard_{shard_id} = group_lines('sum', \
                {{service='replicator', \
                sensor='{replication_group}_shard_id_{shard_id}.{replication_group}_shard_id_{shard_id}_failed_sync_cycles', \
                host=all}});"

    if number_of_shards == 1:
        name = normalize_name(f"failed-sync-cycles-{replication_group}")
        solomon_alert = get_solomon_threshold_alert(
            name=name,
            title_name=f"Failed sync cycles for group {replication_group}",
            solomon_cluster=solomon_cluster,
            selectors={
                'sensor': f'controller.{replication_group}_failed_sync_cycles',
            },
            threshold=10,
        )
    else:
        name = normalize_name(f"failed-sync-cycles-{replication_group}-on-shards")
        solomon_alert = get_solomon_expression_alert(
            name=name,
            title_name=f"Failed sync cycles on shards for group {replication_group}",
            solomon_cluster=solomon_cluster,
            program='\n'.join((build_shard_sensor(i) for i in range(number_of_shards))),
            check_expression=' || '.join(f'sum(sensors_shard_{i}) > 10' for i in range(number_of_shards)),
            annotations={
                "service": f"failed-sync-cycles-on-shards-{replication_group}",
                "host": "replicator.yp-dns-api",
                "alert_description": f"Failed sync cycles for group {replication_group}"
            },
        )

    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        flaps_config=juggler_sdk.FlapOptions(stable=600, critical=1800, boost=0),
        notification_level=WARN,
    )
    return solomon_alert, juggler_check


def get_failed_transactions_alert(solomon_cluster: dict, replication_group_and_number_of_shards: tuple[str, int]) -> tuple[dict, juggler_sdk.Check]:
    replication_group, number_of_shards = replication_group_and_number_of_shards
    def build_shard_sensor(shard_id: int) -> str:
        return f"let sensors_shard_{shard_id} = group_lines('sum', \
                {{service='replicator', \
                sensor='{replication_group}_shard_id_{shard_id}.failed_transactions', \
                host=all}});"

    if number_of_shards == 1:
        name = normalize_name("failed-yp-transactions")
        solomon_alert = get_solomon_threshold_alert(
            name=name,
            title_name="Failed YP transactions",
            solomon_cluster=solomon_cluster,
            selectors={
                'sensor': 'controller.failed_transactions',
            },
            threshold=10,
        )
    else:
        name = normalize_name(f"failed-yp-transactions-{replication_group}-on-shards")
        solomon_alert = get_solomon_expression_alert(
            name=name,
            title_name=f"Failed YP transactions on shards for group {replication_group}",
            solomon_cluster=solomon_cluster,
            program='\n'.join((build_shard_sensor(i) for i in range(number_of_shards))),
            check_expression=' || '.join(f'sum(sensors_shard_{i}) > 10' for i in range(number_of_shards)),
            annotations={
                "service": f"failed-yp-transactions-on-shards-{replication_group}",
                "host": "replicator.yp-dns-api",
                "alert_description": f"Failed YP transactions on shards for group {replication_group}"
            },
        )

    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        flaps_config=juggler_sdk.FlapOptions(stable=600, critical=1800, boost=0),
        notification_level=DEBUG,
    )
    return solomon_alert, juggler_check


def get_logger_failed_writes_alert(solomon_cluster: dict) -> tuple[dict, juggler_sdk.Check]:
    name = normalize_name("logger-failed-writes")
    solomon_alert = get_solomon_threshold_alert(
        name=name,
        title_name="Logger failed writes",
        solomon_cluster=solomon_cluster,
        selectors={
            'sensor': 'logger.write_failed',
        },
    )
    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        notification_level=WARN,
    )
    return solomon_alert, juggler_check


def get_memory_lock_alert(solomon_cluster: dict, sensor_prefix: str) -> tuple[dict, juggler_sdk.Check]:
    name = normalize_name("memory-is-not-locked")
    solomon_alert = get_solomon_threshold_alert(
        name=name,
        title_name="Memory is not locked",
        description="Memory is not locked on host {{labels.host}}",
        solomon_cluster=solomon_cluster,
        selectors={
            'sensor': f'{sensor_prefix}.memory_lock',
            'type': 'Startup',
        },
        time_aggregation='MAX',
        predicate='EQ',
        threshold=0,
        group_by_labels=[
            'host',
        ],
    )
    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        notification_level=DEBUG,
    )
    return solomon_alert, juggler_check


def get_response_time_alert(solomon_cluster: dict, operation_info: dict) -> tuple[dict, juggler_sdk.Check]:
    operation_type = operation_info["operation_type"]
    perc = operation_info["perc"]
    threshold = operation_info["threshold"]
    response_size = "NOSIZE"

    if "response_size" in operation_info:
        response_size = operation_info["response_size"]

    operation = f"{operation_type}-{response_size}"
    name = normalize_name(f"{operation}-response-time-q{perc}")

    selectors={
        'cluster': solomon_cluster['solomon']['cluster'],
        'service': solomon_cluster['solomon']['service'],
        'sensor': 'bridge.service.response_time',
        'operation_type': operation_type,
        'response_size': response_size,
        'host': 'all'
    }

    solomon_alert = get_solomon_expression_alert(
        name=name,
        title_name=f"Response time {operation} (max q{perc}) exceeds the threshold value",
        solomon_cluster=solomon_cluster,
        program='\n'.join([
            f'let lines = {create_selectors(**selectors)};',
            f'let perc = histogram_percentile({perc}, lines);',
            'let response_time = max(perc);',
            f'let threshold = {threshold};',
            'let response_time_pretty_s = to_fixed(response_time / 1000000, 3);',
            'let threshols_pretty_s = to_fixed(threshold / 1000000, 3);',
        ]),
        check_expression='response_time > threshold',
        annotations={
            "response_time": "{{expression.response_time_pretty_s}}s",
            "threshold:": "{{expression.threshols_pretty_s}}s"
        },
        period_ms=milliseconds(minutes=1),
    )
    juggler_check = get_juggler_check(
        solomon_alert=solomon_alert,
        flaps_config=juggler_sdk.FlapOptions(stable=600, critical=1800, boost=0),
        notification_level=WARN,
    )
    return solomon_alert, juggler_check


def update_alert(solomon_client=None, juggler_client=None, solomon_alert=None, juggler_check=None):
    if solomon_alert:
        solomon_client.put_item('alerts', solomon_alert['id'], solomon_alert)

    if juggler_check:
        update_juggler_check(juggler_client, juggler_check)


def update_changelist_size_monitoring_alerts(juggler_client):
    def get_max_changelist_size_check(yp_cluster):
        return get_juggler_check(
            host='{}.check_changelist_size.yp_dns_api'.format(yp_cluster),
            service='max_changelist_size',
            unreach_checks=get_yp_unreach_checks(yp_cluster)
        )

    def get_record_sets_with_nonempty_changelist_check(yp_cluster):
        return get_juggler_check(
            host='{}.check_changelist_size.yp_dns_api'.format(yp_cluster),
            service='record_sets_with_nonempty_changelist',
            unreach_checks=get_yp_unreach_checks(yp_cluster)
        )

    for yp_cluster in YP_CLUSTERS:
        update_juggler_check(juggler_client, get_max_changelist_size_check(yp_cluster))
        update_juggler_check(juggler_client, get_record_sets_with_nonempty_changelist_check(yp_cluster))


REPLICATION_GROUPS = [
    ("dynamic", 2),
    ("reverse-ip6", 2),
    ("testing-r-zone", 2),
]

SOLOMON_CLUSTERS = [
    {
        "name": "bridge",
        "solomon": {
            "cluster": "bridge",
            "service": "bridge",
            "alert_name_prefix": "[Bridge]",
        },
        "alerts": {
            "by_status": [
                {
                    'func': get_update_records_status_alert,
                    'status': [
                        'UNDEFINED',
                        'UNKNOWN_ZONE',
                        'YP_ERROR',
                        'VALIDATION_ERROR',
                        'INVALID_HINT',
                        'CLUSTER_BANNED',
                    ],
                },
                {
                    'func': get_list_zone_record_sets_status_alert,
                    'status': [
                        'UNDEFINED',
                        'UNKNOWN_ZONE',
                        'INVALID_CONTINUATION_TOKEN',
                        'DATA_UNAVAILABLE',
                    ],
                },
            ],
            "by_sensor_prefix": [
                {
                    "func": get_memory_lock_alert,
                    "sensor_prefix": [
                        "bridge.service",
                    ],
                },
            ],
            "by_operation_info": [
                {
                    "func": get_response_time_alert,
                    "operation_info": [
                        {
                            "operation_type": "update_records",
                            "perc": 50,
                            "threshold": microseconds(millis=500),
                        },
                        {
                            "operation_type": "update_records",
                            "perc": 99,
                            "threshold": microseconds(seconds=3),
                        },
                        {
                            "operation_type": "list_zones",
                            "perc": 50,
                            "threshold": microseconds(seconds=1),
                        },
                        {
                            "operation_type": "list_zones",
                            "perc": 99,
                            "threshold": microseconds(seconds=3),
                        },
                        {
                            "operation_type": "create_zone",
                            "perc": 50,
                            "threshold": microseconds(millis=500),
                        },
                        {
                            "operation_type": "create_zone",
                            "perc": 99,
                            "threshold": microseconds(seconds=1),
                        },
                        {
                            "operation_type": "remove_zone",
                            "perc": 50,
                            "threshold": microseconds(millis=500),
                        },
                        {
                            "operation_type": "remove_zone",
                            "perc": 99,
                            "threshold": microseconds(seconds=1),
                        },
                        {
                            "operation_type": "list_zone_record_sets",
                            "response_size": "SMALL",
                            "perc": 50,
                            "threshold": microseconds(seconds=1),
                        },
                        {
                            "operation_type": "list_zone_record_sets",
                            "response_size": "SMALL",
                            "perc": 99,
                            "threshold": microseconds(seconds=5),
                        },
                        {
                            "operation_type": "list_zone_record_sets",
                            "response_size": "MEDIUM",
                            "perc": 50,
                            "threshold": microseconds(seconds=3),
                        },
                        {
                            "operation_type": "list_zone_record_sets",
                            "response_size": "MEDIUM",
                            "perc": 99,
                            "threshold": microseconds(seconds=10),
                        },
                        {
                            "operation_type": "list_zone_record_sets",
                            "response_size": "LARGE",
                            "perc": 50,
                            "threshold": microseconds(seconds=5),
                        },
                        {
                            "operation_type": "list_zone_record_sets",
                            "response_size": "LARGE",
                            "perc": 99,
                            "threshold": microseconds(seconds=20),
                        },
                    ],
                }
            ]
        },
    },
    {
        "name": "replicator",
        "solomon": {
            "cluster": "replicator",
            "service": "replicator",
            "alert_name_prefix": "[Replicator]",
        },
        "alerts": {
            "all": [
                {"func": get_lock_acquired_alert},
                {"func": get_logger_failed_writes_alert},
            ],
            "by_replication_group_and_number_of_shards": [
                {
                    "func": get_no_successful_sync_cycles_alert,
                    "replication_group_and_number_of_shards": REPLICATION_GROUPS,
                },
                {
                    "func": get_failed_transactions_alert,
                    "replication_group_and_number_of_shards": REPLICATION_GROUPS,
                },
            ],
            "by_replication_group": [
                {
                    "func": get_failed_sync_cycles_alert,
                    "replication_group": REPLICATION_GROUPS,
                },
            ],
            "by_sensor_prefix": [
                {
                    "func": get_memory_lock_alert,
                    "sensor_prefix": [
                        "controller",
                    ],
                },
            ],
        },
    }
]


def update_bridge_alerts(solomon_client: SolomonApi, juggler_client: juggler_sdk.JugglerApi):
    for cluster in SOLOMON_CLUSTERS:
        for key, alerts in cluster['alerts'].items():
            if key.startswith('by_'):
                by_key = key[len('by_'):]
                for alert in alerts:
                    for key_value in alert[by_key]:
                        if not isinstance(key_value, list):
                            key_value = [key_value]
                        update_alert(solomon_client, juggler_client, *alert['func'](cluster, *key_value))
            else:
                for alert in alerts:
                    update_alert(solomon_client, juggler_client, *alert['func'](cluster))


def update_alerts(args):
    solomon_client = SolomonApi(SOLOMON_PROJECT_ID, args.solomon_token)
    with juggler_sdk.JugglerApi('http://juggler-api.search.yandex.net', oauth_token=args.juggler_token, mark='yp-dns-api-alerts') as juggler_client:
        # update_multicluster_updates_monitoring_alerts(juggler_client)
        update_changelist_size_monitoring_alerts(juggler_client)

        update_bridge_alerts(solomon_client, juggler_client)


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--solomon-token', required=True, help='solomon token. https://solomon.yandex-team.ru/api/internal/auth')
    parser.add_argument('--juggler-token', required=True, help='juggler token. https://juggler.yandex-team.ru/')
    parser.add_argument('--verbose', action="store_true", help='Enable verbose mode')
    return parser.parse_args(argv)


def main(argv):
    args = parse_args(argv)
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO))

    update_alerts(args)


if __name__ == '__main__':
    main(sys.argv[1:])
