import argparse
import logging
import requests
import sys

from yp.client import YpClient, find_token


logger = logging.getLogger()
sh = logging.StreamHandler()
sh.setFormatter(logging.Formatter('[%(asctime)s] [%(levelname)s] %(message)s'))
logger.addHandler(sh)


EXCEPTION_ERROR_MESSAGE_TMPL = 'An error occurred while running check:\n{}'


def juggler_notify(host_prefix, service, status, description):
    logger.info('Push juggler event.\nStatus: {}\nDescription:\n{}'.format(status, description))
    r = requests.post(
        "http://juggler-push.search.yandex.net/events",
        json={
            "source": "check_changelist_size",
            "events": [
                {
                    "status": status,
                    "description": description,
                    "host": "{}.check_changelist_size.yp_dns_api".format(host_prefix),
                    "service": service,
                    "instance": "",
                }
            ]
        },
        timeout=10
    )
    r.raise_for_status()


def get_zones_with_max_changelist_size(cluster):
    yp_client = YpClient(cluster, config={'token': find_token()})

    def list_zones_with_changelist_size(changelist_min_size):
        if changelist_min_size == 0:
            return ['all']

        result = yp_client.aggregate_objects(
            'dns_record_set',
            group_by=[
                'string([/labels/zone])',
            ],
            aggregators=[
                'sum(1u)'
            ],
            filter='[/labels/changelist/changes/{}] != #'.format(changelist_min_size - 1)
        )

        logging.debug('Aggregate objects result:\n{}'.format(result))
        return list(map(lambda r: r[0], result))

    rb = 1
    while len(list_zones_with_changelist_size(rb)) > 0:
        rb *= 2
    lb = rb // 2

    while lb + 1 < rb:
        mb = (lb + rb) // 2
        if len(list_zones_with_changelist_size(mb)) > 0:
            lb = mb
        else:
            rb = mb

    return {
        'zones': list_zones_with_changelist_size(lb),
        'max_changelist_size': lb,
    }


def get_zones_with_nonempty_changelist(cluster):
    yp_client = YpClient(cluster, config={'token': find_token()})

    aggr_result = yp_client.aggregate_objects(
        'dns_record_set',
        group_by=[
            'string([/labels/zone])',
        ],
        aggregators=[
            'sum(1u)'
        ],
        filter='[/labels/changelist/changes/0] != #'
    )

    logging.debug('Aggregate objects result:\n{}'.format(aggr_result))

    return {
        zone: record_sets_number
        for zone, record_sets_number in aggr_result
    }


def run_max_changelist_size_check(cluster, threshold):
    juggler_service = 'max_changelist_size'
    max_zones_to_print = 4

    try:
        result = get_zones_with_max_changelist_size(cluster)
    except Exception as e:
        juggler_status = 'NO_DATA'
        juggler_description = EXCEPTION_ERROR_MESSAGE_TMPL.format(e)
    else:
        if result['max_changelist_size'] > threshold:
            juggler_status = 'CRIT'
        else:
            juggler_status = 'OK'

        juggler_description_tmpl = '\n'.join([
            f"Max changelist size in YP-{cluster} is {result['max_changelist_size']} (threshold = {threshold})",
            "Zones: {zones}",
            "See these record sets:"
            f" yp select dns_record_set --no-tabular --address {cluster} --selector /meta/id --selector /spec --selector /labels"
            f" --filter \'[/labels/changelist/changes/{result['max_changelist_size'] - 1}] != #\'"
        ])

        if len(result['zones']) <= max_zones_to_print:
            zones_to_print = result['zones']
        else:
            zones_to_print = result['zones'][:max_zones_to_print] + ['...']
        juggler_description = juggler_description_tmpl.format(
            zones=', '.join(zones_to_print)
        )

    juggler_notify(
        host_prefix=cluster,
        service=juggler_service,
        status=juggler_status,
        description=juggler_description
    )


def run_record_sets_with_nonempty_changelist_check(cluster, threshold):
    juggler_service = 'record_sets_with_nonempty_changelist'
    max_zones_to_print = 4

    try:
        result = get_zones_with_nonempty_changelist(cluster)
    except Exception as e:
        juggler_status = 'NO_DATA'
        juggler_description = EXCEPTION_ERROR_MESSAGE_TMPL.format(e)
    else:
        record_sets_with_nonempty_cl = sum(result.values())
        if record_sets_with_nonempty_cl > threshold:
            juggler_status = 'CRIT'
        else:
            juggler_status = 'OK'

        juggler_description_tmpl = '\n'.join([
            f"Record sets with non-empty changelist number in YP-{cluster} is {record_sets_with_nonempty_cl} (threshold = {threshold})",
            "Top zones: {zones}",
        ])

        sorted_zones = list(map(
            lambda kv: '{} ({})'.format(kv[0], kv[1]),
            sorted(result.items(), key=lambda v: v[1], reverse=True)
        ))
        if len(sorted_zones) <= max_zones_to_print:
            zones_to_print = sorted_zones
        else:
            zones_to_print = sorted_zones[:max_zones_to_print] + ['...']
        juggler_description = juggler_description_tmpl.format(
            zones=', '.join(zones_to_print)
        )

    juggler_notify(
        host_prefix=cluster,
        service=juggler_service,
        status=juggler_status,
        description=juggler_description
    )


def try_run(check_func, *args):
    try:
        check_func(*args)
    except Exception as e:
        logger.error('An error occurred while running check')
        logger.exception(e)
        pass


def run_checks(args):
    try_run(run_max_changelist_size_check, args.cluster, args.max_cl_size)
    try_run(run_record_sets_with_nonempty_changelist_check, args.cluster, args.max_nonempty_cl)


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('-c', '--cluster', required=True,
                        help='YP cluster name')
    parser.add_argument('--max-cl-size', type=int, default=10,
                        help='Max changelist size')
    parser.add_argument('--max-nonempty-cl', type=int, default=10,
                        help='Max record sets with nonempty changelist')

    return parser.parse_args(argv)


def main(argv):
    args = parse_args(argv)
    logger.setLevel(logging.DEBUG)
    run_checks(args)


if __name__ == '__main__':
    main(sys.argv[1:])
