#include "config_validator.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NInfra::NYpDnsApi::NReplicator;

Y_UNIT_TEST_SUITE(ConfigValidatorTest) {
    auto expectException = [](const TShardsZonesDistributionConfig& config) {
        try {
            NConfigValidator::ValidateShardsZonesDistribution(config);
        } catch (const yexception& ex) {
            return;
        } catch (...) {
            throw "Unexpected exception";
        }

        throw "Did not throw exception";
    };

    Y_UNIT_TEST(TestEmptyConfig) {
        auto config = TShardsZonesDistributionConfig();

        NConfigValidator::ValidateShardsZonesDistribution(config);
    }

    Y_UNIT_TEST(TestAllCommonShardsConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(4);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(4);
        }

        NConfigValidator::ValidateShardsZonesDistribution(config);
    }

    Y_UNIT_TEST(TestAllCommonShardsWrongNumbersConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(4);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(2);
        }

        expectException(config);
    }

    Y_UNIT_TEST(TestNoShardsDescribedConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(4);
        }

        expectException(config);
    }

    Y_UNIT_TEST(TestAllSpecificZonesConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(2);
            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("a");
            firstShard->AddListOfZones("b");
            auto secondShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            secondShard->AddListOfZones("c");
            secondShard->AddListOfZones("d");
        }

        expectException(config);
    }

    Y_UNIT_TEST(TestSpecificZonesGivenConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(3);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(1);

            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("a");
            firstShard->AddListOfZones("b");
            auto secondShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            secondShard->AddListOfZones("c");
            secondShard->AddListOfZones("d");
        }

        NConfigValidator::ValidateShardsZonesDistribution(config);
    }

    Y_UNIT_TEST(TestTooBigNumberOfShardsConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(4);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(1);

            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("a");
            firstShard->AddListOfZones("b");
            auto secondShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            secondShard->AddListOfZones("c");
            secondShard->AddListOfZones("d");
        }

        expectException(config);
    }

    Y_UNIT_TEST(TestSpecificZoneADuplicateConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(3);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(1);

            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("a");
            firstShard->AddListOfZones("a");
            auto secondShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            secondShard->AddListOfZones("c");
            secondShard->AddListOfZones("d");
        }

        expectException(config);
    }

    Y_UNIT_TEST(TestSpecificShardsIntersectConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(3);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(1);

            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("a");
            firstShard->AddListOfZones("c");
            auto secondShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            secondShard->AddListOfZones("c");
            secondShard->AddListOfZones("d");
        }

        expectException(config);
    }

    Y_UNIT_TEST(TestFirstProdConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(2);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(1);

            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("cloud.yandex.net");
            firstShard->AddListOfZones("db.yandex.net");
            firstShard->AddListOfZones("search.yandex.net");
            firstShard->AddListOfZones("yandexcloud.net");
        }

        NConfigValidator::ValidateShardsZonesDistribution(config);
    }

    Y_UNIT_TEST(TestSecondProdConfig) {
        auto config = TShardsZonesDistributionConfig();

        {
            config.MutableShardsConfig()->SetNumberOfShards(5);
            config.MutableZonesCoverage()->SetNumberOfCommonShards(1);

            auto firstShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            firstShard->AddListOfZones("cloud.yandex.net");
            auto secondShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            secondShard->AddListOfZones("db.yandex.net");
            auto thirdShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            thirdShard->AddListOfZones("search.yandex.net");
            auto fourthShard = config.MutableZonesCoverage()->AddSpecificZonesShards();
            fourthShard->AddListOfZones("yandexcloud.net");
        }

        NConfigValidator::ValidateShardsZonesDistribution(config);
    }
}
