#include "main.h"

#include <infra/yp_dns_api/replicator/config/config.pb.h>
#include <infra/yp_dns_api/replicator/logger/events/events_decl.ev.pb.h>
#include <infra/yp_dns_api/replicator/zone_replicator/factory.h>
#include <infra/yp_dns_api/replicator/zone_replicator/protos/config/config.pb.h>

#include <infra/libs/controller/standalone_controller/standalone_controller.h>
#include <infra/libs/logger/logger.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/proto_config/load.h>

#include <util/generic/hash_set.h>

namespace NInfra::NYpDnsApi::NReplicator {

struct TOptions {
    TString Zone;
    THashSet<TString> Clusters;
    TString LogPath = "stderr";
    bool RunOnce = false;
    bool ReadOnly = false;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;

    TOptions result;

    opts.SetFreeArgsNum(0);

    opts.AddHelpOption();

    opts.AddLongOption('z', "zone", "Zone")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.Zone);

    opts.AddLongOption('c', "clusters", "Cluster where the zone is stored")
        .Required()
        .RequiredArgument()
        .SplitHandler(&result.Clusters, ',');

    opts.AddLongOption('l', "log-path", "Path log")
        .Optional()
        .RequiredArgument()
        .DefaultValue(result.LogPath)
        .StoreResult(&result.LogPath);

    opts.AddLongOption('o', "run-once", "Run sync iteration only once")
        .Optional()
        .StoreTrue(&result.RunOnce);

    opts.AddLongOption('r', "read-only", "Run in read only mode")
        .Optional()
        .StoreTrue(&result.ReadOnly);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

int SyncZone(int argc, const char* argv[]) {
    TOptions options = ParseOptions(argc, argv);

    TLoggerConfig loggerConfig;
    if (options.LogPath == "stderr") {
        loggerConfig.SetBackend(TLoggerConfig::STDERR);
    } else if (options.LogPath == "stdout") {
        loggerConfig.SetBackend(TLoggerConfig::STDOUT);
    } else {
        loggerConfig.SetBackend(TLoggerConfig::FILE);
        loggerConfig.SetPath(options.LogPath);
    }
    loggerConfig.SetLevel("DEBUG");
    loggerConfig.SetQueueSize(1'000'000);

    TZonesReplicationConfig factoryConfig;
    factoryConfig.SetGroupName("sync-" + options.Zone);
    for (const TString& cluster : options.Clusters) {
        NController::TClientConfig clusterConfig;
        clusterConfig.SetClusterName(cluster);
        clusterConfig.SetAddress(cluster + ".yp.yandex.net:8090");
        clusterConfig.SetTimeout("80s");
        clusterConfig.SetReadOnlyMode(options.ReadOnly);
        clusterConfig.SetMaxReceiveMessageSize(299062097);
        *factoryConfig.AddClusterConfigs() = std::move(clusterConfig);
    }
    TZoneReplicatorConfig zoneReplicatorConfig;
    zoneReplicatorConfig.SetZone(options.Zone);
    for (const TString& cluster : options.Clusters) {
        zoneReplicatorConfig.AddClusters(cluster);
    }
    zoneReplicatorConfig.SetMaxUpdatesPerCluster(1000);
    zoneReplicatorConfig.SetTotalSelectLimit(10000);
    zoneReplicatorConfig.SetSelectLimit(10000);
    zoneReplicatorConfig.SetRecordSetsType(TZoneReplicatorConfig::WITH_CHANGES);
    *factoryConfig.AddZoneReplicatorConfigs() = std::move(zoneReplicatorConfig);

    TReplicatorConfig serviceConfig;
    NProtoConfig::TLoadConfigOptions loadConfigOptions;
    loadConfigOptions.Resource = "/proto_config/replicator_config.json";
    NProtoConfig::LoadWithOptions(serviceConfig, loadConfigOptions);
    *serviceConfig.MutableController()->MutableLogger() = std::move(loggerConfig);
    serviceConfig.MutableController()->MutableLeadingInvader()->SetType(NInfra::NLeadingInvader::TConfig::MOCK_LEADER);

    NController::TSharding shardFactory(serviceConfig.GetController().GetLeadingInvader());
    NController::TStandaloneController controller(
        serviceConfig.GetController()
        , {NController::TObjectManagersFactoryPtr(new TZoneReplicatorsFactory(
            shardFactory.GetShard(0) /* Shard */
            , std::move(factoryConfig) /* ZonesReplicationConfig */
            , TZoneReplicatorsFactory::EShardType::COMMON_SHARD /* ShardType */
            , 1 /* NumberOfCommonShards */
            , {} /* SpecificZones */
            , {} /* AllShardsSpecificZones */
        ))}
    );

    while (true) {
        try {
            controller.Sync();
        } catch (...) {
            Cerr << "Sync error: " << CurrentExceptionMessage() << Endl;
        }
        if (options.RunOnce) {
            break;
        }
    }

    return EXIT_SUCCESS;
}

} // namespace NInfra::NYpDnsApi::NReplicator
