#include "factory.h"

#include <infra/yp_dns_api/replicator/zone_replicator/protos/config/config.pb.h>

#include <infra/yp_dns_api/replicator/logger/events/events_decl.ev.pb.h>

#include <infra/yp_dns_api/libs/yp/aggregate_objects.h>

#include <infra/libs/yp_dns/record_set/record_set.h>

#include <yp/cpp/yp/token.h>

#include <library/cpp/yt/farmhash/farm_hash.h>

namespace NInfra::NYpDnsApi::NReplicator {

namespace {

TString BuildGetObjectsFilter(const TString& zoneFilter, const TString& recordSetTypeFilter) {
    Y_ENSURE(!zoneFilter.empty(), "Zone filter should not be empty for GetObjects/SelectObjects arguments");
    TStringBuilder result = TStringBuilder() << "(" << zoneFilter << ")";
    if (!recordSetTypeFilter.empty()) {
        result << " and (" << recordSetTypeFilter << ")";
    }
    return result;
}

TString BuildFilterByRecordSetsType(const TZoneReplicatorConfig::ERecordSetsType& recordSetsType) {
    switch (recordSetsType) {
        case TZoneReplicatorConfig::WITH_CHANGES:
            return "[/labels/changelist/changes/0] != # or [/spec/records] = # or [/spec/records/0] = #";
        case TZoneReplicatorConfig::WITHOUT_CHANGELIST_ALL:
        case TZoneReplicatorConfig::WITHOUT_CHANGELIST_ANY:
            return "[/labels/changelist] = #";
        case TZoneReplicatorConfig::ALL:
        default:
            return "";
    }
}

THashSet<TString> GetSpecificZonesForAllShards(
    const TShardsZonesDistributionConfig::TZonesCoverageConfig& zonesCoverageConfig
) {
    THashSet<TString> specificZones;
    for (const auto& specificZonesShard : zonesCoverageConfig.GetSpecificZonesShards()) {
        for (const auto& zone : specificZonesShard.GetListOfZones()) {
            specificZones.emplace(zone);
        }
    }

    return specificZones;
}

TZoneReplicatorsFactory::EShardType GetShardType(
    const TShardsZonesDistributionConfig::TZonesCoverageConfig& zonesConverageConfig,
    size_t shardId
) {
    return shardId < zonesConverageConfig.GetNumberOfCommonShards()
        ? TZoneReplicatorsFactory::EShardType::COMMON_SHARD
        : TZoneReplicatorsFactory::EShardType::SPECIFIC_ZONES_SHARD;
}

THashSet<TString> GetSpecificZones(
    const TShardsZonesDistributionConfig::TZonesCoverageConfig& zonesCoverageConfig,
    size_t shardId
) {
    if (shardId < zonesCoverageConfig.GetNumberOfCommonShards()) {
        return {};
    }

    THashSet<TString> specificZones;
    size_t specificZonesShardIndex = shardId - zonesCoverageConfig.GetNumberOfCommonShards();

    for (const auto& zone : zonesCoverageConfig.GetSpecificZonesShards().at(specificZonesShardIndex).GetListOfZones()) {
        specificZones.emplace(zone);
    }
    return specificZones;
}

} // anonymous namespace

TZoneReplicatorsFactory::TZoneReplicatorsFactory(
    NController::TShardPtr shard
    , TZonesReplicationConfig config
    , const EShardType& shardType
    , size_t numberOfCommonShards
    , const THashSet<TString>& specificZones
    , const THashSet<TString>& allShardsSpecificZones
)
    : IObjectManagersFactory(config.GetGroupName(), shard)
    , Config_(std::move(config))
    , ZonesManagerClient_(
        Config_.GetDynamicZonesConfig().GetEnable()
            ? MakeHolder<NYpDns::NDynamicZones::TClient>(
                Config_.GetDynamicZonesConfig().GetZonesManagerClientConfig(),
                MakeAtomicShared<NYpDns::NDynamicZones::TReplicatorServiceClient>()
            )
            : nullptr
    )
    , StaticZoneReplicatorConfigs_(Config_.GetZoneReplicatorConfigs().begin(), Config_.GetZoneReplicatorConfigs().end())
    , RandomGenerator_(Seed())
    , ShardType_(shardType)
    , NumberOfCommonShards_(numberOfCommonShards)
    , CurrentSpecificZones_(specificZones)
    , AllShardsSpecificZones_(allShardsSpecificZones)
{
    FilterOutZonesByShard(StaticZoneReplicatorConfigs_);

    FillAggregateArguments(StaticZoneReplicatorConfigs_, StaticAggregateObjectsArguments_);
    FillSelectArguments(StaticZoneReplicatorConfigs_, StaticGetObjectsArguments_);
}

TMaybe<TVector<NController::TClientConfig>> TZoneReplicatorsFactory::GetYpClientConfigs() const {
    if (Config_.GetClusterConfigs().empty()) {
        return Nothing();
    }
    TVector<NController::TClientConfig> result;
    for (const NController::TClientConfig& clusterConfig : Config_.GetClusterConfigs()) {
        result.push_back(clusterConfig);
    }
    return result;
}

TVector<NController::IObjectManagersFactory::TAggregateArgument>
TZoneReplicatorsFactory::GetAggregateArguments(NInfra::TLogFramePtr logFrame) const {
    Y_UNUSED(logFrame);

    if (Config_.GetDynamicZonesConfig().GetEnable()) {
        Y_ENSURE(ZonesManagerClient_);
        TVector<NYpDns::TZone> dynamicZones = ZonesManagerClient_->ListZones();
        ShuffleRange(dynamicZones, RandomGenerator_);

        TVector<TZoneReplicatorConfig> replicatorConfigs;
        replicatorConfigs.reserve(dynamicZones.size());
        for (const NYpDns::TZone& zone : dynamicZones) {
            TZoneReplicatorConfig& config = replicatorConfigs.emplace_back();
            config.SetZone(zone.GetName());
            for (const TString& cluster : zone.Config().GetYPClusters()) {
                config.AddClusters(cluster);
            }
            config.SetMaxUpdatesPerCluster(500);
            config.SetTotalSelectLimit(1000);
            config.SetSelectLimit(1000);
        }

        FilterOutZonesByShard(replicatorConfigs);

        AggregateObjectsArguments_ = StaticAggregateObjectsArguments_;
        GetObjectsArguments_ = StaticGetObjectsArguments_;
        FillAggregateArguments(replicatorConfigs, AggregateObjectsArguments_);
        FillSelectArguments(replicatorConfigs, GetObjectsArguments_);

        ZoneReplicatorConfigs_ = StaticZoneReplicatorConfigs_;
        ZoneReplicatorConfigs_.reserve(ZoneReplicatorConfigs_.size() + replicatorConfigs.size());
        std::move(replicatorConfigs.begin(), replicatorConfigs.end(), std::back_inserter(ZoneReplicatorConfigs_));
    } else {
        ZoneReplicatorConfigs_ = StaticZoneReplicatorConfigs_;
        AggregateObjectsArguments_ = StaticAggregateObjectsArguments_;
        GetObjectsArguments_ = StaticGetObjectsArguments_;
    }

    TVector<NController::IObjectManagersFactory::TAggregateArgument> result;
    result.reserve(AggregateObjectsArguments_.size());
    for (const auto& [key, arg] : AggregateObjectsArguments_) {
        result.push_back(arg);
    }

    return result;
}

TVector<NController::IObjectManager::TSelectArgument> TZoneReplicatorsFactory::GetSelectArguments(
    const TVector<TVector<NController::TSelectorResultPtr>>& aggregateResults
    , NInfra::TLogFramePtr logFrame
) const {
    Y_ENSURE(AggregateObjectsArguments_.size() == aggregateResults.size());

    TMap<TAggregateResultKey, ui64> recordSetsNumber;
    auto aggregateArgumentsIt = AggregateObjectsArguments_.cbegin();
    for (size_t i = 0; i < aggregateResults.size(); ++i, ++aggregateArgumentsIt) {
        const auto& [argumentKey, argument] = *aggregateArgumentsIt;
        for (size_t j = 0; j < aggregateResults[i].size(); ++j) {
            TString zone;
            ui64 objectsNumber;
            aggregateResults[i][j]->Fill(
                &zone,
                &objectsNumber
            );
            logFrame->LogEvent(NEventlog::TAggregationResult(
                argument.ClusterName,
                zone,
                argument.Filter,
                objectsNumber
            ));

            TAggregateResultKey resultKey{
                argument.ClusterName,
                zone,
                argumentKey.RecordSetsType
            };
            recordSetsNumber.emplace(std::move(resultKey), objectsNumber);
        }
    }

    TVector<NController::IObjectManager::TSelectArgument> result;
    result.reserve(GetObjectsArguments_.size());
    auto argIt = GetObjectsArguments_.cbegin();

    ZoneReplicatorsSkipMask_.assign(ZoneReplicatorConfigs_.size(), false);
    ui32 zoneReplicatorsToRunNumber = 0;
    for (size_t i = 0; i < ZoneReplicatorConfigs_.size(); ++i) {
        const TZoneReplicatorConfig& zoneReplicatorConfig = ZoneReplicatorConfigs_.at(i);

        // Check whether we need to skip zone
        auto hasChanges = [&]() -> bool {
            auto argJt = argIt;
            for (const TString& cluster : zoneReplicatorConfig.GetClusters()) {
                Y_ENSURE(argJt != GetObjectsArguments_.end());
                Y_ENSURE(cluster == argJt->ClusterName);

                TAggregateResultKey aggregateKey{
                    cluster,
                    zoneReplicatorConfig.GetZone(),
                    zoneReplicatorConfig.GetRecordSetsType()
                };

                if (recordSetsNumber.Value(aggregateKey, 0) > 0) {
                    return true;
                }

                ++argJt;
            }
            return false;
        };

        bool skipZone = false;
        TString skipReason;
        if (Config_.GetMaxZoneReplicatorsToRun() > 0 &&
            zoneReplicatorsToRunNumber >= Config_.GetMaxZoneReplicatorsToRun())
        {
            skipZone = true;
            skipReason = "Reached limit of zone replicators to run";
        } else if (!hasChanges()) {
            skipZone = true;
            skipReason = "No changes";
        }

        if (skipZone) {
            logFrame->LogEvent(NEventlog::TSkipZone(zoneReplicatorConfig.GetZone(), skipReason));
            ZoneReplicatorsSkipMask_[i] = true;
            std::advance(argIt, zoneReplicatorConfig.GetClusters().size());
            continue;
        }

        ++zoneReplicatorsToRunNumber;

        // Update arguments for select-objects requests to YP
        for (const TString& cluster : zoneReplicatorConfig.GetClusters()) {
            auto& resultArg = result.emplace_back(*argIt);

            Y_ENSURE(argIt != GetObjectsArguments_.end());
            Y_ENSURE(cluster == resultArg.ClusterName);

            TAggregateResultKey aggregateKey{
                cluster,
                zoneReplicatorConfig.GetZone(),
                zoneReplicatorConfig.GetRecordSetsType()
            };
            ui64 objectsNumber = recordSetsNumber.Value(aggregateKey, 0);

            if (objectsNumber < resultArg.Options.Limit()) { // earlier first condition was if Limit is defined, but this way it should work the same
                logFrame->LogEvent(NEventlog::TSetLimitForGetObjects(ToString(resultArg.Options.Limit()), objectsNumber, "0"));
                resultArg.Options.SetLimit(0);
            }

            {
                TMaybe<ui64> totalLimitBefore = resultArg.TotalLimit;
                if (resultArg.TotalLimit.Defined()) {
                    resultArg.TotalLimit = Min(*resultArg.TotalLimit, objectsNumber);
                } else {
                    resultArg.TotalLimit = objectsNumber;
                }
                logFrame->LogEvent(NEventlog::TSetTotalLimitForGetObjects(ToString(totalLimitBefore), objectsNumber, ToString(resultArg.TotalLimit)));
            }

            logFrame->LogEvent(MakeGetObjectsArgumentEvent(resultArg));

            ++argIt;
        }
    }
    Y_ENSURE(argIt == GetObjectsArguments_.end());

    return result;
}

TVector<TExpected<NController::TObjectManagerPtr, NController::IObjectManagersFactory::TValidationError>>
TZoneReplicatorsFactory::GetObjectManagers(
    const TVector<NController::TSelectObjectsResultPtr>& selectorResults
    , NInfra::TLogFramePtr logFrame
) const {
    Y_UNUSED(logFrame);
    TVector<TExpected<NController::TObjectManagerPtr, NController::IObjectManagersFactory::TValidationError>> result;
    result.reserve(ZoneReplicatorConfigs_.size());

    auto getObjectsResultsIt = selectorResults.cbegin();
    for (size_t i = 0; i < ZoneReplicatorConfigs_.size(); ++i) {
        if (ZoneReplicatorsSkipMask_.at(i)) {
            continue;
        }

        const TZoneReplicatorConfig& config = ZoneReplicatorConfigs_.at(i);

        THashMap<TString, TVector<NYpDns::TSerializedRecordSet>> recordSets;
        recordSets.reserve(config.GetClusters().size());
        for (const TString& cluster : config.GetClusters()) {
            TVector<NYpDns::TSerializedRecordSet>& clusterRecordSets = recordSets[cluster];
            clusterRecordSets.reserve((*getObjectsResultsIt)->Results.size());

            bool recordSetsSorted = true;
            TString prevRecordSetId;
            for (const NInfra::NController::TSelectorResultPtr& selectorResult : (*getObjectsResultsIt)->Results) {
                NYpDns::TRecordSet recordSet;

                recordSet.SetYpTimestamp((*getObjectsResultsIt)->Timestamp);
                TString zone;
                NJson::TJsonValue changelistJson;
                selectorResult->Fill(
                    recordSet.MutableMeta()->mutable_id(),
                    recordSet.MutableMeta()->mutable_acl(),
                    recordSet.MutableSpec()->mutable_records(),
                    &zone,
                    &changelistJson
                );
                recordSet.SetZone(zone);
                if (changelistJson.IsMap()) {
                    recordSet.MutableChangelist()->FromJson(changelistJson);
                }

                recordSetsSorted &= prevRecordSetId < recordSet.Meta().id();
                prevRecordSetId = recordSet.Meta().id();

                clusterRecordSets.emplace_back(std::move(recordSet));
            }

            if (!recordSetsSorted) {
                SortBy(clusterRecordSets, [](const NYpDns::TSerializedRecordSet& recordSet) {
                    return recordSet.GetObject().Meta().id();
                });
            }

            ++getObjectsResultsIt;
        }

        result.emplace_back(new TZoneReplicator(config, std::move(recordSets)));
    }
    Y_ENSURE(getObjectsResultsIt == selectorResults.cend());

    return result;
}

void TZoneReplicatorsFactory::FillAggregateArguments(
    const TVector<TZoneReplicatorConfig>& replicatorConfigs,
    TMap<TAggregateArgumentKey, NController::IObjectManagersFactory::TAggregateArgument>& aggregateArguments
) const {
    NController::IObjectManagersFactory::TAggregateArgument aggregateObjectsArg;
    aggregateObjectsArg.ObjectType = NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET;

    aggregateObjectsArg.GroupByExpressions = {
        "string([/labels/zone])",
    };
    aggregateObjectsArg.AggregateExpressions = {
        "sum(1u)",
    };

    for (const TZoneReplicatorConfig& zoneReplicatorConfig : replicatorConfigs) {
        const TString recordSetTypeFilter = BuildFilterByRecordSetsType(zoneReplicatorConfig.GetRecordSetsType());

        // Fill AggregateObjects argument
        aggregateObjectsArg.Filter = recordSetTypeFilter;

        for (const TString& cluster : zoneReplicatorConfig.GetClusters()) {
            aggregateObjectsArg.ClusterName = cluster;
            TAggregateArgumentKey key{cluster, zoneReplicatorConfig.GetRecordSetsType()};
            aggregateArguments.emplace(std::move(key), aggregateObjectsArg);
        }
    }
}

void TZoneReplicatorsFactory::FillSelectArguments(
    const TVector<TZoneReplicatorConfig>& replicatorConfigs,
    TVector<NController::IObjectManager::TSelectArgument>& getObjectsArguments
) const {
    for (const TZoneReplicatorConfig& zoneReplicatorConfig : replicatorConfigs) {
        // Fill GetObjects argument
        NYP::NClient::NApi::NProto::EObjectType objectType = NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET;
        TVector<TString> selectors = {
            "/meta/id",
            "/meta/acl",
            "/spec/records",
            "/labels/zone",
            "/labels/changelist",
        };

        const TString zoneFilter = TStringBuilder() << "[/labels/zone] = \"" << zoneReplicatorConfig.GetZone() << "\"";
        const TString recordSetTypeFilter = BuildFilterByRecordSetsType(zoneReplicatorConfig.GetRecordSetsType());
        const TString getObjectsFilter = BuildGetObjectsFilter(zoneFilter, recordSetTypeFilter);

        ui64 totalLimit = zoneReplicatorConfig.GetTotalSelectLimit();
        NYP::NClient::TSelectObjectsOptions options;
        options.SetLimit(zoneReplicatorConfig.GetSelectLimit());

        // Add arguments
        for (const TString& cluster : zoneReplicatorConfig.GetClusters()) {
            getObjectsArguments.emplace_back(
                objectType,
                selectors,
                getObjectsFilter,
                options,
                NController::TClientFilterConfig{},
                NController::TOverrideYpReqLimitsConfig{},
                /* selectAll */ true,
                cluster,
                totalLimit
            );
        }
    }
}

bool TZoneReplicatorsFactory::CheckZoneCompatibility(
    const TString& zone
) const {
    switch (ShardType_) {
        case TZoneReplicatorsFactory::EShardType::COMMON_SHARD:
            return !AllShardsSpecificZones_.contains(zone) && NYT::FarmFingerprint(zone) % NumberOfCommonShards_ == GetShard()->GetShardId();
        case TZoneReplicatorsFactory::EShardType::SPECIFIC_ZONES_SHARD:
            return CurrentSpecificZones_.contains(zone);
    }
}

void TZoneReplicatorsFactory::FilterOutZonesByShard(
    TVector<TZoneReplicatorConfig>& zoneReplicatorConfigs
) const {
    zoneReplicatorConfigs.erase(
        std::remove_if(
            zoneReplicatorConfigs.begin(),
            zoneReplicatorConfigs.end(),
            [this] (const TZoneReplicatorConfig& zoneReplicatorConfig) {
                return !CheckZoneCompatibility(zoneReplicatorConfig.GetZone());
            }),
        zoneReplicatorConfigs.end()
    );
}

TVector<NController::TObjectManagersFactoryPtr> CreateZoneReplicatorsFactories(
    TVector<TZonesReplicationConfig> configs
    , NController::TSharding& shardFactory
    , const TShardsZonesDistributionConfig::TZonesCoverageConfig& zonesCoverageConfig
) {
    TVector<NController::TObjectManagersFactoryPtr> result;
    result.reserve(configs.size() * shardFactory.GetNumberOfShards());

    THashSet<TString> allShardsSpecificZones = GetSpecificZonesForAllShards(zonesCoverageConfig);
    for (size_t i = 0; i < shardFactory.GetNumberOfShards(); ++i) {
        TZoneReplicatorsFactory::EShardType shardType = GetShardType(zonesCoverageConfig, i);
        THashSet<TString> specificZones = GetSpecificZones(zonesCoverageConfig, i);

        for (TZonesReplicationConfig& config : configs) {
            result.emplace_back(CreateManagersFactory<TZoneReplicatorsFactory>(
                shardFactory.GetShard(i)
                , config
                , shardType
                , zonesCoverageConfig.GetNumberOfCommonShards()
                , specificZones
                , allShardsSpecificZones
            ));
        }
    }
    return result;
}

} // namespace NInfra::NYpDnsApi::NReplicator
