#pragma once

#include "replicator.h"

#include <infra/yp_dns_api/replicator/config/config.pb.h>

#include <infra/yp_dns_api/libs/yp/aggregate_objects.h>
#include <infra/yp_dns_api/libs/yp/clients_registry.h>
#include <infra/yp_dns_api/libs/yp/get_objects.h>

#include <infra/libs/yp_dns/dynamic_zones/client.h>

#include <infra/libs/controller/object_manager/object_manager.h>
#include <infra/libs/logger/log_frame.h>

#include <util/random/shuffle.h>

namespace NInfra::NYpDnsApi::NReplicator {

struct TAggregateArgumentKey {
    TString Cluster;
    TZoneReplicatorConfig::ERecordSetsType RecordSetsType;

    bool operator<(const TAggregateArgumentKey& rhs) const {
        return std::tie(Cluster, RecordSetsType) <
               std::tie(rhs.Cluster, rhs.RecordSetsType);
    }
};

struct TAggregateResultKey {
    TString Cluster;
    TString Zone;
    TZoneReplicatorConfig::ERecordSetsType RecordSetsType;

    bool operator<(const TAggregateResultKey& rhs) const {
        return std::tie(Cluster, Zone, RecordSetsType) <
               std::tie(rhs.Cluster, rhs.Zone, rhs.RecordSetsType);
    }
};

class TZoneReplicatorsFactory: public NController::IObjectManagersFactory {
public:
    enum class EShardType {
        COMMON_SHARD = 0,
        SPECIFIC_ZONES_SHARD = 1,
    };

    TZoneReplicatorsFactory(
        NController::TShardPtr shard
        , TZonesReplicationConfig config
        , const EShardType& shardType
        , size_t numberOfCommonShards
        , const THashSet<TString>& specificZones
        , const THashSet<TString>& allShardsSpecificZones
    );

    TMaybe<TVector<NController::TClientConfig>> GetYpClientConfigs() const override;

    TVector<NController::IObjectManagersFactory::TAggregateArgument> GetAggregateArguments(NInfra::TLogFramePtr logFrame) const override;

    TVector<NController::IObjectManager::TSelectArgument> GetSelectArguments(const TVector<TVector<NController::TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const override;

    TVector<TExpected<NController::TObjectManagerPtr, NController::IObjectManagersFactory::TValidationError>> GetObjectManagers(
        const TVector<NController::TSelectObjectsResultPtr>& selectorResults
        , NInfra::TLogFramePtr logFrame
    ) const override;

private:
    void FillSelectArguments(
        const TVector<TZoneReplicatorConfig>& replicatorConfigs,
        TVector<NController::IObjectManager::TSelectArgument>& getObjectsArguments
    ) const;

    void FillAggregateArguments(
        const TVector<TZoneReplicatorConfig>& replicatorConfigs,
        TMap<TAggregateArgumentKey, NController::IObjectManagersFactory::TAggregateArgument>& aggregateArguments
    ) const;

    bool CheckZoneCompatibility(
        const TString& zone
    ) const;

    void FilterOutZonesByShard(
        TVector<TZoneReplicatorConfig>& zones
    ) const;

private:
    const TZonesReplicationConfig Config_;

    const THolder<NYpDns::NDynamicZones::TClient> ZonesManagerClient_;

    TVector<TZoneReplicatorConfig> StaticZoneReplicatorConfigs_;
    TMap<TAggregateArgumentKey, NController::IObjectManagersFactory::TAggregateArgument> StaticAggregateObjectsArguments_;
    TVector<NController::IObjectManager::TSelectArgument> StaticGetObjectsArguments_;

    mutable TReallyFastRng32 RandomGenerator_;

    const EShardType ShardType_;
    const size_t NumberOfCommonShards_;
    THashSet<TString> CurrentSpecificZones_;
    THashSet<TString> AllShardsSpecificZones_;

    mutable TVector<TZoneReplicatorConfig> ZoneReplicatorConfigs_;
    mutable TVector<bool> ZoneReplicatorsSkipMask_;
    mutable TMap<TAggregateArgumentKey, NController::IObjectManagersFactory::TAggregateArgument> AggregateObjectsArguments_;
    mutable TVector<NController::IObjectManager::TSelectArgument> GetObjectsArguments_;
};

template <typename TManagersFactory, typename... Args>
NController::TObjectManagersFactoryPtr CreateManagersFactory(Args&&... args) {
    return MakeAtomicShared<TManagersFactory>(std::forward<Args>(args)...);
}

TVector<NController::TObjectManagersFactoryPtr> CreateZoneReplicatorsFactories(
    TVector<TZonesReplicationConfig> configs,
    NController::TSharding& shardFactory,
    const TShardsZonesDistributionConfig::TZonesCoverageConfig& zonesCoverageConfig
);

} // namespace NInfra::NYpDnsApi::NReplicator
