#include "replicator.h"

#include <infra/yp_dns_api/replicator/logger/events/events_decl.ev.pb.h>

#include <infra/libs/yp_dns/replication/iterate.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/generic/set.h>

namespace NInfra::NYpDnsApi::NReplicator {

namespace {

NEventlog::TDnsRecord MakeDnsRecordEvent(const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& record) {
    NEventlog::TDnsRecord event;
    event.SetTtl(record.ttl());
    event.SetClass(record.class_());
    event.SetType(NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType_Name(record.type()));
    event.SetData(record.data());
    return event;
}

NEventlog::TDnsRecordSet MakeDnsRecordSetEvent(const NYpDns::TRecordSet& recordSet) {
    NEventlog::TDnsRecordSet event;
    event.SetFqdn(recordSet.Meta().id());
    event.MutableRecords()->Reserve(recordSet.Spec().records().size());
    for (const auto& record : recordSet.Spec().records()) {
        *event.AddRecords() = MakeDnsRecordEvent(record);
    }
    if (recordSet.HasChangelist()) {
        event.SetChangelist(NJson::WriteJson(recordSet.Changelist().ToJson(), /* formatOutput */ false));
    }
    for (const NYP::NClient::NApi::NProto::TAccessControlEntry& ace : recordSet.Meta().acl()) {
        event.AddAcl(NProtobufJson::Proto2Json(ace, NProtobufJson::TProto2JsonConfig().SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName)));
    }
    return event;
}

NEventlog::TZoneReplicatorRecordSetReplica MakeZoneReplicatorRecordSetReplicaEvent(const TString& cluster, const TMaybe<NYpDns::TRecordSet>& recordSetReplica) {
    NEventlog::TZoneReplicatorRecordSetReplica event;
    event.SetCluster(cluster);
    if (recordSetReplica) {
        *event.MutableRecordSet() = MakeDnsRecordSetEvent(*recordSetReplica);
    }
    return event;
}

NEventlog::TZoneReplicatorRecordSetReplicasMergeResult MakeZoneReplicatorRecordSetReplicasMergeResultEvent(const TString& cluster, const TMaybe<NYpDns::TRecordSet>& mergedRecordSet) {
    NEventlog::TZoneReplicatorRecordSetReplicasMergeResult event;
    event.SetCluster(cluster);
    if (mergedRecordSet) {
        *event.MutableRecordSet() = MakeDnsRecordSetEvent(*mergedRecordSet);
    }
    return event;
}

NEventlog::TZoneReplicatorRecordSetReplicaOperation MakeZoneReplicatorRecordSetReplicaOperationEvent(
    const NEventlog::TZoneReplicatorRecordSetReplicaOperation::EAction action,
    const TString& cluster,
    const TMaybe<NYpDns::TRecordSet>& recordSetReplica = {}
) {
    NEventlog::TZoneReplicatorRecordSetReplicaOperation event;
    event.SetAction(action);
    event.SetCluster(cluster);
    if (recordSetReplica) {
        *event.MutableRecordSet() = MakeDnsRecordSetEvent(*recordSetReplica);
    }
    return event;
}


} // anonymous namespace

TZoneReplicator::TZoneReplicator(TZoneReplicatorConfig config, THashMap<TString, TVector<NYpDns::TSerializedRecordSet>> zoneRecordSets)
    : Config_(std::move(config))
    , ZoneRecordSets_(std::move(zoneRecordSets))
{
    for (const auto& [cluster, recordSets] : ZoneRecordSets_) {
        Y_ENSURE(IsSortedBy(recordSets.begin(), recordSets.end(), [](const auto& recordSet) {
            return recordSet.GetObject().Meta().id();
        }));
    }

    THashMap<TString, TVector<TString>> fqdnsToAsk;
    NYpDns::Iterate(ZoneRecordSets_, [&fqdnsToAsk](const NYpDns::TRecordSetReplicas& recordSetReplicas) {
        for (const auto& [cluster, recordSetReplica] : recordSetReplicas.Replicas) {
            if (!recordSetReplica) {
                fqdnsToAsk[cluster].push_back(recordSetReplicas.Fqdn);
            }
        }
        return true;
    });

    constexpr size_t BATCH_SIZE = 20;

    size_t reserveSize = 0;
    for (const auto& [cluster, fqdns] : fqdnsToAsk) {
        reserveSize += (fqdns.size() + BATCH_SIZE - 1) / BATCH_SIZE;
    }
    GetDependentObjectsArguments_.reserve(reserveSize);
    for (const auto& [cluster, fqdns] : fqdnsToAsk) {
        for (size_t i = 0; i < fqdns.size(); i += BATCH_SIZE) {
            const size_t j = Min(i + BATCH_SIZE, fqdns.size());

            NYP::NClient::NApi::NProto::EObjectType objectType = NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET;
            TVector<TString> selectors = {
                "/meta/id",
                "/meta/acl",
                "/spec/records",
                "/labels/zone",
                "/labels/changelist",
            };
            TStringBuilder filter = TStringBuilder() << "[/meta/id] in (";
            for (size_t k = i; k < j; ++k) {
                if (k != i) {
                    filter << ",";
                }
                filter << "\"" << fqdns[k] << "\"";
            }
            filter << ")";

            GetDependentObjectsArguments_.emplace_back(
                objectType
                , selectors
                , std::move(filter)
                , NYP::NClient::TSelectObjectsOptions{}
                , NController::TClientFilterConfig{}
                , NController::TOverrideYpReqLimitsConfig{}
                , /* selectAll */ true
                , cluster);
        }
    }
}

TString TZoneReplicator::GetObjectId() const {
    return Config_.GetZone();
}

TVector<NController::IObjectManager::TSelectArgument> TZoneReplicator::GetDependentObjectsSelectArguments() const {
    return GetDependentObjectsArguments_;
}

void TZoneReplicator::GenerateYpUpdates(
    const TDependentObjects& dependentObjects
    , THashMap<TString, TVector<TRequest>>& requests
    , TLogFramePtr frame
) const {
    frame->LogEvent(NEventlog::TZoneReplicatorGenerateYpUpdates(Config_.GetZone()));

    THashMap<TString, TVector<NYpDns::TSerializedRecordSet>> allRecordSets = ZoneRecordSets_;
    for (const TString& cluster : Config_.GetClusters()) {
        allRecordSets.try_emplace(cluster);
    }
    Y_ENSURE(dependentObjects.SelectedObjects.size() == GetDependentObjectsArguments_.size());
    THashSet<TString> idsToSkip;
    for (size_t i = 0; i < dependentObjects.SelectedObjects.size(); ++i) {
        for (const NController::TSelectorResultPtr& selectorResult : dependentObjects.SelectedObjects[i]->Results) {
            NYpDns::TRecordSet recordSet;

            recordSet.SetYpTimestamp(dependentObjects.SelectedObjects[i]->Timestamp);
            TString zone;
            NJson::TJsonValue changelistJson;
            selectorResult->Fill(
                recordSet.MutableMeta()->mutable_id(),
                recordSet.MutableMeta()->mutable_acl(),
                recordSet.MutableSpec()->mutable_records(),
                &zone,
                &changelistJson
            );
            recordSet.SetZone(zone);
            if (changelistJson.IsDefined()) {
                recordSet.MutableChangelist()->FromJson(changelistJson);
            }

            switch (Config_.GetRecordSetsType()) {
                case TZoneReplicatorConfig::WITHOUT_CHANGELIST_ALL: {
                    if (recordSet.HasChangelist()) {
                        idsToSkip.insert(recordSet.Meta().id());
                    }
                    break;
                }
                case TZoneReplicatorConfig::WITHOUT_CHANGELIST_ANY:
                case TZoneReplicatorConfig::WITH_CHANGES:
                case TZoneReplicatorConfig::ALL:
                default:
                    break;
            }

            allRecordSets[*GetDependentObjectsArguments_[i].ClusterName].emplace_back(std::move(recordSet));
        }
    }
    for (auto& [cluster, recordSets] : allRecordSets) {
        SortBy(recordSets, [](const NYpDns::TSerializedRecordSet& recordSet) {
            return recordSet.GetObject().Meta().id();
        });
    }

    TSet<TString> clustersReachedUpdatesLimit;
    NYpDns::Iterate(allRecordSets, [&config = Config_, &clustersReachedUpdatesLimit, &idsToSkip, &requests, frame](const NYpDns::TRecordSetReplicas& recordSetReplicas) {
        if (clustersReachedUpdatesLimit.size() == static_cast<size_t>(config.GetClusters().size())) {
            frame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TZoneReplicatorMaxUpdatesReachedForAllClusters());
            return false;
        }
        if (idsToSkip.contains(recordSetReplicas.Fqdn)) {
            return true;
        }

        THashMap<TString, TMaybe<NYpDns::TRecordSet>> recordSets;
        recordSets.reserve(recordSetReplicas.Replicas.size());

        frame->LogEvent(NEventlog::TZoneReplicatorHandleRecordSet(recordSetReplicas.Fqdn));
        for (auto&& [cluster, recordSetReplica] : recordSetReplicas.Replicas) {
            frame->LogEvent(MakeZoneReplicatorRecordSetReplicaEvent(cluster, recordSetReplica));
            Y_ENSURE(recordSets.emplace(cluster, std::move(recordSetReplica)).second);
        }

        const THashMap<TString, TMaybe<NYpDns::TRecordSet>> targetRecordSets = NYpDns::ReplicateChanges(recordSets);
        THashMap<TString, ui64> createUpdates;
        THashMap<TString, ui64> removeUpdates;
        THashMap<TString, ui64> updateUpdates;
        for (const auto& [cluster, recordSet] : targetRecordSets) {
            frame->LogEvent(MakeZoneReplicatorRecordSetReplicasMergeResultEvent(cluster, recordSet));
        }
        for (const auto& [cluster, _] : recordSetReplicas.Replicas) {
            TVector<TRequest>& clusterUpdates = requests[cluster];

            const ui64 totalUpdates = clusterUpdates.size();
            if (config.GetMaxUpdatesPerCluster() > 0 && totalUpdates>= config.GetMaxUpdatesPerCluster()) {
                frame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TZoneReplicatorMaxUpdatesPerClusterReached(
                    cluster,
                    config.GetMaxUpdatesPerCluster(),
                    createUpdates[cluster],
                    removeUpdates[cluster],
                    updateUpdates[cluster],
                    totalUpdates
                ));
                clustersReachedUpdatesLimit.insert(cluster);
                continue;
            }

            const auto& baseRecordSet = recordSets.at(cluster);
            const auto& targetRecordSet = targetRecordSets.at(cluster);
            if (targetRecordSet != baseRecordSet) {
                if (!targetRecordSet) {
                    Y_ENSURE(baseRecordSet && baseRecordSet->Meta().id() == recordSetReplicas.Fqdn);
                    frame->LogEvent(MakeZoneReplicatorRecordSetReplicaOperationEvent(
                        NEventlog::TZoneReplicatorRecordSetReplicaOperation::REMOVE,
                        cluster
                    ));
                    clusterUpdates.emplace_back(NYP::NClient::TRemoveObjectRequest(
                        NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET,
                        baseRecordSet->Meta().id()
                    ));
                    ++removeUpdates[cluster];
                } else if (!baseRecordSet) {
                    frame->LogEvent(MakeZoneReplicatorRecordSetReplicaOperationEvent(
                        NEventlog::TZoneReplicatorRecordSetReplicaOperation::CREATE,
                        cluster,
                        targetRecordSet
                    ));
                    clusterUpdates.emplace_back(NYP::NClient::TCreateObjectRequest(
                        targetRecordSet->MakeYpObject()
                    ));
                    ++createUpdates[cluster];
                } else {
                    Y_ENSURE(baseRecordSet && baseRecordSet->Meta().id() == recordSetReplicas.Fqdn);
                    frame->LogEvent(MakeZoneReplicatorRecordSetReplicaOperationEvent(
                        NEventlog::TZoneReplicatorRecordSetReplicaOperation::UPDATE,
                        cluster,
                        targetRecordSet
                    ));

                    TVector<NYP::NClient::TSetRequest> setRequests = {
                        NYP::NClient::TSetRequest{"/meta/acl", targetRecordSet->Meta().acl()},
                        NYP::NClient::TSetRequest{"/spec/records", targetRecordSet->Spec().records()},
                    };
                    if (targetRecordSet->HasChangelist()) {
                        setRequests.emplace_back("/labels/changelist", NYT::NodeFromJsonValue(targetRecordSet->Changelist().ToJson()));;
                    }
                    clusterUpdates.emplace_back(NYP::NClient::TUpdateRequest{
                        NYP::NClient::NApi::NProto::OT_DNS_RECORD_SET,
                        baseRecordSet->Meta().id(),
                        setRequests,
                        /* removeRequests */ {
                        }
                    });
                    ++updateUpdates[cluster];
                }
            } else {
                frame->LogEvent(MakeZoneReplicatorRecordSetReplicaOperationEvent(
                    NEventlog::TZoneReplicatorRecordSetReplicaOperation::SKIP,
                    cluster
                ));
            }
        }

        return true;
    });
}

} // namespace NInfra::NYpDnsApi::NReplicator
