import argparse
import logging
import requests
import sys
import time


DEFAULT_NAMESERVERS = [
    'ns1.testns.yandex.net',
]

DEFAULT_YP_CLUSTERS = [
    'sas-test',
    'man-pre',
]

ZONE_NAME_TMPL = '{subdomain}.zonemanager.test.yandex.net'


def create_zone(address, zone):
    logging.info(f"{zone}: creating")
    r = requests.post(
        f"http://{address}/zone/create/json",
        json={
            "zone": {
                "config": {
                    "Name": zone,
                    "PrimaryNameserver": DEFAULT_NAMESERVERS[0],
                    "Nameservers": DEFAULT_NAMESERVERS,
                    "YPClusters": DEFAULT_YP_CLUSTERS,
                },
            },
        },
        timeout=60
    )
    r.raise_for_status()
    logging.info(f"{zone}: create response: {r.json()}")
    assert r.json()["status"] == "OK"
    create_time = time.time()
    logging.info(f"{zone}: created in {r.elapsed.total_seconds()}s")

    logging.info(f"{zone}: waiting for appearance in listing")
    while True:
        try:
            r = requests.post(
                f"http://{address}/zones/list/json",
                json={
                    "service_type": "DNS-API",
                },
                timeout=120
            )

            r.raise_for_status()

            assert r.json()["status"] == "OK"
            if any(map(lambda z: z["config"]["Name"] == zone, r.json()["zones"])):
                appeared_listing_time = time.time()
                logging.info(f"{zone}: found in listing after {appeared_listing_time - create_time:.2f}s")
                break

        except Exception:
            logging.exception("Failed to list zones")
            pass


def remove_zone(address, zone):
    logging.info(f"{zone}: removing")
    r = requests.post(
        f"http://{address}/zone/remove/json",
        json={
            "zone_id": zone,
        },
        timeout=20
    )
    r.raise_for_status()
    logging.info(f"{zone}: remove response: {r.json()}")
    assert r.json()["status"] == "OK"
    remove_time = time.time()
    logging.info(f"{zone}: removed in {r.elapsed.total_seconds()}s")

    logging.info(f"{zone}: waiting for appearance in listing")
    while True:
        try:
            r = requests.post(
                f"http://{address}/zones/list/json",
                json={
                    "service_type": "DNS-API",
                },
                timeout=60
            )

            r.raise_for_status()

            assert r.json()["status"] == "OK"
            if all(map(lambda zone: zone["config"]["Name"] != zone, r.json()["zones"])):
                disapeared_listing_time = time.time()
                logging.info(f"{zone}: not found in listing after {disapeared_listing_time - remove_time:.2f}s")
                break

        except Exception:
            logging.exception("Failed to list zones")
            pass


def create_zones(args):
    logging.info(f"About to create {args.num} zones")
    start_create_time = time.time()
    for i in range(args.num):
        zone_name = ZONE_NAME_TMPL.format(subdomain='zone{}'.format(i))
        create_zone(args.address, zone_name)
    end_create_time = time.time()
    logging.info(f"Created {args.num} zones in {end_create_time - start_create_time:.2f}s")


def remove_zones(args):
    logging.info(f"About to remove {args.num} zones")
    start_remove_time = time.time()
    for i in range(args.num):
        zone_name = ZONE_NAME_TMPL.format(subdomain='zone{}'.format(i))
        try:
            remove_zone(args.address, zone_name)
        except:
            logging.exception(f"{zone_name}: failed to remove zone")
    end_remove_time = time.time()
    logging.info(f"Removed {args.num} zones in {end_remove_time - start_remove_time:.2f}s")


def parse_args(argv):
    parser = argparse.ArgumentParser(description='Creates batch of zones via zone manager')

    parser.add_argument('-a', '--address', required=True,
                        help='Address of Zone Manager')
    parser.add_argument('-n', '--num', required=True, type=int,
                        help='Number of zones to create')
    parser.add_argument('--clean-up', action='store_true',
                        help='Clean up created zones after run')
    parser.add_argument('-d', '--dry-run', action='store_true')
    return parser.parse_args()


def main(argv):
    logging.basicConfig(level=logging.INFO, format='[%(asctime)s] [%(levelname)-5s] %(message)s')

    args = parse_args(argv)

    try:
        create_zones(args)
    except:
        logging.exception("Failed to create zones")
        remove_zones(args)
        raise

    if args.clean_up:
        time.sleep(10)
        remove_zones(args)


if __name__ == '__main__':
    main(sys.argv[1:])
