import argparse
import json
import logging

from yp.client import YpClient, find_token

from deepdiff import DeepDiff


def parse_args():
    parser = argparse.ArgumentParser(description='Finds diff in record sets in YP-clusters')

    parser.add_argument('-z', '--zone', required=True)
    parser.add_argument('-c', '--clusters', required=True, nargs='+',
                        choices=['sas-test', 'sas', 'man-pre', 'man', 'vla', 'myt', 'iva', 'xdc'],
                        help='YP cluster names')
    parser.add_argument('-f', '--filter', help='Additional filter for listing all records')
    return parser.parse_args()


def main():
    args = parse_args()

    logging.basicConfig(level=logging.DEBUG, format='[%(asctime)s] [%(levelname)-5s] %(message)s')

    filter_expr = ''
    if args.zone:
        filter_expr = "[/labels/zone] = '{}'".format(args.zone)
    if args.filter:
        filter_expr = '({}) and ({})'.format(filter_expr or '%true', args.filter)

    record_sets = {cluster: [] for cluster in args.clusters}

    for cluster in args.clusters:
        yp_client = YpClient(cluster, config={'token': find_token()})
        timestamp = yp_client.generate_timestamp()
        continuation_token = None
        while True:
            limit = 10000
            result = yp_client.select_objects(
                'dns_record_set',
                filter=filter_expr,
                selectors=['/meta/id', '/spec/records'],
                timestamp=timestamp,
                limit=limit,
                options={'continuation_token': continuation_token},
                enable_structured_response=True
            )
            continuation_token = result['continuation_token']

            record_sets[cluster].extend(result['results'])

            if len(result['results']) < limit:
                break

    record_sets_by_id = {cluster: {} for cluster in args.clusters}
    for cluster, record_sets in record_sets.items():
        for id, records in record_sets:
            record_sets_by_id[cluster][id['value']] = {
                'spec': {
                    'records': records['value'],
                },
            }

    keys = set()
    for cluster, record_sets in record_sets_by_id.items():
        keys |= set(record_sets.keys())

    diffs_total = 0
    for key in keys:
        rs = {}
        for cluster in record_sets_by_id.keys():
            rs[cluster] = record_sets_by_id[cluster].get(key)

        for i, cluster1 in enumerate(args.clusters):
            for j, cluster2 in enumerate(args.clusters):
                if j <= i:
                    continue
                record_set_1 = rs[cluster1]
                record_set_2 = rs[cluster2]
                diff = DeepDiff(record_set_1, record_set_2)
                if diff:
                    if 'type_changes' in diff:
                        if isinstance(None, diff['type_changes']['root']['old_type']):
                            continue
                    diff_json = json.loads(diff.to_json())
                    logging.info('Diff {} in YP-{} vs YP-{}:\n{}'.format(key, cluster1.upper(), cluster2.upper(), json.dumps(diff_json, indent=2)))
                    diffs_total += 1
    logging.info('Total diffs: {}'.format(diffs_total))

if __name__ == '__main__':
    main()
