#include <infra/yp_dns_api/bridge/client/client.h>

#include <infra/libs/service_iface/str_iface.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/datetime/cputimer.h>

namespace NInfra::NYpDnsApi {

struct TOptions {
    TString BridgeGrpcAddress = "dns-api-bridge.yp.yandex.net:8081";
    TString Zone;
    ui64 ListChunkSize = 200;
    bool Verbose = false;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    TOptions result;

    NLastGetopt::TOpts opts;

    opts
        .AddLongOption('a', "address", "Bridge gRPC address")
        .Optional()
        .RequiredArgument("HOST:GRPC_PORT")
        .DefaultValue(result.BridgeGrpcAddress)
        .StoreResult(&result.BridgeGrpcAddress);

    opts
        .AddLongOption('z', "zone", "Name of DNS zone to list")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.Zone);

    opts
        .AddLongOption('l', "list-chunk-size", "Max record sets number in each request chunk")
        .Optional()
        .RequiredArgument("UINT64")
        .DefaultValue(result.ListChunkSize)
        .StoreResult(&result.ListChunkSize);

    opts
        .AddLongOption('v', "verbose", "Enable verbose output")
        .Optional()
        .NoArgument()
        .StoreTrue(&result.Verbose);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

int RunListZone(int argc, const char* argv[]) {
    const TOptions options = ParseOptions(argc, argv);
    InitGlobalLog2Console(options.Verbose ? TLOG_DEBUG : TLOG_INFO);

    TClient client(options.BridgeGrpcAddress);

    NProtobufJson::TProto2JsonConfig proto2JsonConfig;
    proto2JsonConfig.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);

    NApi::TReqListZoneRecordSets request;
    request.set_zone(options.Zone);
    request.set_limit(options.ListChunkSize);

    TVector<NApi::TRecordSet> result;
    ui32 requestsNumber = 0;

    TTimer timer;
    while (true) {
        DEBUG_LOG << "Request: " << NProtobufJson::Proto2Json(request, proto2JsonConfig) << Endl;
        auto req = RequestPtr<TProtoRequest<NApi::TReqListZoneRecordSets>>("", request, NInfra::TAttributes());

        NApi::TRspListZoneRecordSets response;
        TAttributes responseAttributes;
        auto rsp = ReplyPtr<TProtoReply<NApi::TRspListZoneRecordSets>>(response, responseAttributes);

        ++requestsNumber;
        client.ListZoneRecordSets(req, rsp);

        DEBUG_LOG << "Chunk size: " << response.record_sets().size() << Endl;
        if (!response.record_sets().empty()) {
            DEBUG_LOG << "First record set id in chunk: " << response.record_sets().begin()->id() << Endl;
            DEBUG_LOG << "Last record set id in chunk:  " << response.record_sets().rbegin()->id() << Endl;
        }

        for (size_t i = 0; i < static_cast<size_t>(response.record_sets().size()); ++i) {
            if (i > 0) {
                Y_ENSURE(response.record_sets().at(i - 1).id() < response.record_sets().at(i).id(),
                    "Invalid at pos " << i << ": " << response.record_sets().at(i - 1).id() << " > " << response.record_sets().at(i).id());
            }

            Y_ENSURE(!response.record_sets().at(i).records().empty(), "Empty record set " << response.record_sets().at(i).id());
        }

        std::move(response.mutable_record_sets()->begin(), response.mutable_record_sets()->end(), std::back_inserter(result));

        if (request.limit() == 0 || static_cast<ui64>(response.record_sets().size()) < request.limit() / 2) {
            break;
        }

        request.set_continuation_token(response.continuation_token());
    }

    INFO_LOG << "Requests number: " << requestsNumber << Endl;
    INFO_LOG << "Record sets in zone " << options.Zone << ": " << result.size() << Endl;

    return EXIT_SUCCESS;
}

} // namespace NInfra::NYpDnsApi

int main(int argc, const char* argv[]) {
    return NInfra::NYpDnsApi::RunListZone(argc, argv);
}
