import argparse
import logging

from yp.client import YpClient, find_token

import dns.name


def in_zone(fqdn, zone):
    return dns.name.from_text(fqdn).is_subdomain(dns.name.from_text(zone))


def parse_args():
    parser = argparse.ArgumentParser(description='Marks all record sets in zone with zone label')

    parser.add_argument('-z', '--zone', required=True)
    parser.add_argument('-c', '--cluster', required=True,
                        choices=['sas-test', 'sas', 'man-pre', 'man', 'vla', 'myt', 'iva', 'xdc'],
                        help='YP cluster name')
    parser.add_argument('-f', '--filter', help='Additional filter for listing all records')
    parser.add_argument('-d', '--dry-run', action='store_true')
    return parser.parse_args()


def main():
    args = parse_args()

    logging.basicConfig(level=logging.DEBUG, format='[%(asctime)s] [%(levelname)-5s] %(message)s')

    yp_client = YpClient(address='{}.yp.yandex.net:8090'.format(args.cluster), config={'token': find_token()})

    filter_expr = "[/labels/zone] != '{}'".format(args.zone)
    if args.filter:
        filter_expr = '({}) and ({})'.format(filter_expr, args.filter)

    continuation_token = None
    update_requests = []
    while True:
        timestamp = yp_client.generate_timestamp()

        limit = 10000
        result = yp_client.select_objects('dns_record_set', filter=filter_expr,
                                          selectors=['/meta/id', '/labels/zone'], timestamp=timestamp, limit=limit,
                                          options={'continuation_token': continuation_token}, enable_structured_response=True)
        continuation_token = result['continuation_token']

        updates = []
        for id, zone in result['results']:
            if in_zone(id['value'], args.zone):
                updates.append(id['value'])

        for id in updates:
            update_requests.append({
                'object_type': 'dns_record_set',
                'object_id': id,
                'set_updates': [
                    {
                        'path': '/labels/zone',
                        'value': args.zone,
                    },
                ],
            })

        if len(result['results']) < limit:
            break

    logging.info('total updates: {}'.format(len(update_requests)))

    batch_size = 1000
    for i in range(0, len(update_requests), batch_size):
        reqs = update_requests[i:min(len(update_requests), i + batch_size)]
        logging.info('update batch of size {}'.format(len(reqs)))
        if not args.dry_run:
            yp_client.update_objects(reqs)


if __name__ == '__main__':
    main()
