import argparse
import logging

from yp.client import YpClient, find_token


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--zones-file',
                        help='Path to file with zones list')
    return parser.parse_args()


def read_zone_file(path):
    with open(path) as f:
        for line in f.readlines():
            yield line.split()[0]


def get_nameservers(record_set_spec):
    result = []
    for record in record_set_spec['records']:
        if record['type'] == 'NS':
            result.append((record['data'], record['ttl']))
    return result


def parse_soa(record_set_spec):
    for record in record_set_spec['records']:
        if record['type'] == 'SOA':
            primary, email, serial, refresh, retry, expire, neg_ttl = record['data'].split()
            serial, refresh, retry, expire, neg_ttl = map(int, [serial, refresh, retry, expire, neg_ttl])
            return (primary, email, serial, refresh, retry, expire, neg_ttl), record['ttl']


def sync_zones_configs_with_records():
    args = parse_args()
    zones = list(set(read_zone_file(args.zones_file)))

    yp_client_xdc = YpClient('xdc', config={'token': find_token()})
    yp_client_man = YpClient('man', config={'token': find_token()})

    batch_size = 100
    for start_idx in range(0, len(zones), batch_size):
        zones_batch = zones[start_idx:start_idx + batch_size]

        yp_zones = dict(yp_client_xdc.get_objects(
            'dns_zone',
            zones_batch,
            selectors=[
                '/meta/id',
                '/spec',
            ],
        ))

        record_sets = dict(filter(None, yp_client_man.get_objects(
            'dns_record_set',
            zones_batch,
            selectors=['/meta/id', '/spec'],
            options={"ignore_nonexistent": True},
        )))

        updates = []
        for zone_id, zone_config in yp_zones.items():
            if zone_id not in record_sets:
                logging.error(f"no record set for zone {zone_id}")
                continue
            record_set = record_sets[zone_id]
            nameservers = get_nameservers(record_set)
            soa = parse_soa(record_set)

            set_updates_for_zone = []
            if len(nameservers) > 0:
                new_value = [
                    {
                        "ttl": ttl,
                        "class": "IN",
                        "NS": {
                            "nameserver": nameserver,
                        }
                    }
                    for nameserver, ttl in nameservers
                ]
                if new_value != zone_config["config"]["default_ns_records"]:
                    set_updates_for_zone.append({
                        'path': '/spec/config/default_ns_records',
                        'value': new_value,
                    })
            else:
                logging.error(f"empty nameservers list for {zone_id}")

            if soa:
                (primary, email, serial, refresh, retry, expire, neg_ttl), ttl = soa
                new_value = {
                    "ttl": ttl,
                    "class": "IN",
                    "SOA": {
                        "primary_nameserver": primary,
                        "email": email,
                        "serial": serial,
                        "refresh": refresh,
                        "retry": retry,
                        "expire": expire,
                        "negative_ttl": neg_ttl,
                    }
                }
                if new_value != zone_config["config"]["default_soa_record"]:
                    set_updates_for_zone.append({
                        'path': '/spec/config/default_soa_record',
                        'value': new_value,
                    })
            else:
                logging.error(f"no soa record for {zone_id}")

            if set_updates_for_zone:
                logging.info(f"updates for zone {zone_id}: {set_updates_for_zone}")
                updates.append({
                    'object_type': 'dns_zone',
                    'object_id': zone_id,
                    'set_updates': set_updates_for_zone,
                })

        yp_client_xdc.update_objects(updates)


def main():
    logging.basicConfig(level=logging.INFO)
    sync_zones_configs_with_records()


if __name__ == '__main__':
    main()
