#include "update_timestamp.h"

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <yp/cpp/yp/client.h>
#include <yp/cpp/yp/token.h>

#include <library/cpp/getopt/small/last_getopt.h>

#include <util/generic/string.h>

namespace NYpDnsUpdatesCoordinator {

struct TOptions {
    TVector<TString> Clusters;
    TString Location;
    TString Service;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts = NLastGetopt::TOpts::Default();

    TOptions result;

    opts.AddLongOption('c', "cluster", "YP cluster")
        .Optional()
        .RequiredArgument("STRING")
        .DefaultValue("sas-test,man-pre,sas,man,vla,myt,iva,xdc")
        .SplitHandler(&result.Clusters, ',');

    opts.AddLongOption('l', "location", "DNS location")
        .Required()
        .RequiredArgument("STRING")
        .StoreResult(&result.Location);

    opts.AddLongOption('s', "service", "Service name")
        .Required()
        .RequiredArgument("STRING")
        .StoreResult(&result.Service);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

NYT::TYPath GetTargetPath(const TString& cluster, const TString& service, const TString& location) {
    return NYT::JoinYPaths("//home/yp-dns/yp_updates_coordinator", cluster, service, "coordinator/locations", location, "target");
}

void UpdateTimestamp(const TOptions& options) {
    for (const TString& cluster : options.Clusters) {
        NYP::NClient::TClientPtr ypClient = NYP::NClient::CreateClient(
            NYP::NClient::TClientOptions()
                .SetAddress(cluster + ".yp.yandex.net:8090")
                .SetToken(NYP::NClient::FindToken())
                .SetTimeout(TDuration::Seconds(10))
                .SetEnableBalancing(false)
        );

        NYT::IClientPtr ytClient = NYT::CreateClient("yp-" + cluster);

        const ui64 timestamp = ypClient->GenerateTimestamp().GetValue(ypClient->Options().Timeout() * 2);

        const NYT::TYPath targetPath = GetTargetPath(cluster, options.Service, options.Location);
        Cout << "Set [yp-" << cluster << "]" << targetPath << " = " << timestamp << Endl;
        ytClient->Set(targetPath, timestamp);
    }
}

int UpdateTimestamp(int argc, const char* argv[]) {
    const TOptions options = ParseOptions(argc, argv);
    UpdateTimestamp(options);
    return EXIT_SUCCESS;
}

} // namespace NYpDnsUpdatesCoordinator
