import os
import sys
import asyncio

import uvloop


asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())


import logging
import logging.handlers
import argparse
from functools import partial

from infra.yp_drcp.server import make_app, spawn_yp_updater

from aiohttp import web


def parse_args():
    parser = argparse.ArgumentParser(description='DynamicResource caching proxy for YP')
    parser.add_argument('--token', metavar='YP_TOKEN', default=os.getenv('YP_TOKEN'))
    parser.add_argument('--yp-address', metavar='YP_ADDRESS', default=os.getenv('YP_ADDRESS'))
    parser.add_argument('--yp-update-period', type=int, default=60)
    parser.add_argument('--yp-chunk-limit', type=int, default=200)
    parser.add_argument('--loglevel',
                        choices=('DEBUG', 'INFO', 'WARNING', 'ERROR'),
                        default='DEBUG',
                        help='Log level to use')
    parser.add_argument('--logfile', help='File to log into. If not specified, will log to stderr')
    parser.add_argument('--logsize', help='Log file size for rotating file handler', type=int, default=(1 << 22))
    parser.add_argument('--logcount', help='Log files to keep', type=int, default=14)

    result = parser.parse_args()
    if not result.yp_address:
        print('--yp-address or env var YP_ADDRESS must be present', file=sys.stderr)
        sys.exit(1)
    if not result.token:
        print('--token or env var YP_TOKEN must be present', file=sys.stderr)
        sys.exit(1)

    return result


def main():
    args = parse_args()
    if not args.logfile:
        logging.basicConfig(level=args.loglevel, format='%(asctime)s.%(msecs)003d [%(levelname)-1s] [%(name)-10s]   %(message)s')
    else:
        log = logging.getLogger('')
        handler = logging.handlers.RotatingFileHandler(
            args.logfile,
            maxBytes=args.logsize,
            backupCount=args.logcount,
        )
        handler.setFormatter(logging.Formatter(
            fmt='%(asctime)s.%(msecs)003d [%(levelname)-1s] [%(name)-10s]   %(message)s',
            datefmt='',
        ))
        log.setLevel(args.loglevel)
        log.addHandler(handler)

    app = asyncio.get_event_loop().run_until_complete(make_app())

    app.on_startup.append(partial(spawn_yp_updater, args=args))
    web.run_app(app, host='::')
