import logging
import asyncio
import hashlib
from functools import partial

import ujson
from aiohttp import web

from yp.client import YpClient
from yt.yson.convert import yson_to_json


class PodResources:
    __slots__ = ['resources', 'future', 'revision']

    def __init__(self, resources, revision):
        self.future = asyncio.Future()
        self.revision = revision
        self.resources = resources


async def handle(request: web.Request):
    log = logging.getLogger('http')
    pod_id = request.match_info['pod']
    revision = request.match_info.get('revision', None)

    pod = request.app['pods'].setdefault(pod_id, PodResources(None, None))
    log.debug("requested (%r, %r), found (%r)", pod_id, revision, pod.revision)
    if (
        (revision is None and pod.resources is not None)
        or (revision is not None and pod.revision is not None and pod.revision != revision)
    ):
        return web.json_response(
            {
                'resources': pod.resources,
                'revision': pod.revision,
            },
            dumps=ujson.dumps,
        )

    try:
        log.debug("will wait for future")
        result = await asyncio.wait_for(asyncio.shield(pod.future), 3600.)
    except asyncio.TimeoutError:
        log.debug("timed out")
        return web.json_response({}, dumps=ujson.dumps)
    else:
        return web.json_response(
            {
                'resources': result[0],
                'revision': result[1],
            },
            dumps=ujson.dumps,
        )


async def handle_post(request: web.Request):
    pod_id = request.match_info['pod']

    data = await request.json(loads=ujson.loads)
    pod = PodResources(data['resources'], data['revision'])

    await update_pod_info(request.app, pod_id, pod)
    return web.Response(status=204, reason='Updated successfully')


async def shutdown(app: web.Application):
    if 'yp_updater' in app:
        app['yp_updater'].cancel()
        del app['yp_updater']
    if 'yp_client' in app:
        app['yp_client'].close()
    for conn_info in app['sleeping_connections'].values():
        conn_info.close()


async def fetch_resources(client: YpClient, chunk_limit: int):
    log = logging.getLogger('yp')
    timestamp = client.generate_timestamp()
    offset = 0
    while True:
        objects = await asyncio.get_event_loop().run_in_executor(None, partial(
            client.select_objects,
            'pod',
            selectors=['/meta/id', '/spec/dynamic_resources'],
            timestamp=timestamp,
            offset=offset,
            limit=chunk_limit,
        ))
        if not objects:
            break
        for obj in objects:
            if obj:
                obj = tuple(yson_to_json(o) for o in obj)
                hasher = hashlib.new('md5')
                if obj[1]:
                    for dr in sorted(obj[1], key=lambda r: r.get('id')):
                        hasher.update(str(dr.get('id', '')).encode('ascii'))
                        hasher.update(str(dr.get('revision', '')).encode('ascii'))
                digest = hasher.hexdigest()
                if obj[1]:
                    log.info('loaded: %s -> %s', obj[0], obj[1])
                yield obj[0], PodResources(obj[1], digest)
        offset += len(objects)


async def update_pod_info(app: web.Application, pod_id: str, info: PodResources):
    if pod_id not in app['pods']:
        app['pods'][pod_id] = info
    elif app['pods'][pod_id].resources != info.resources:
        pod = app['pods'].pop(pod_id)
        app['pods'][pod_id] = info
        pod.future.set_result((info.resources, info.revision))


async def yp_updater_step(app: web.Application, client: YpClient):
    log = logging.getLogger('app')
    log.info("starting iteration with chunk size = %d", app['yp_chunk_limit'])
    new_pod_map = {pod_id: resources
                   async for pod_id, resources
                   in fetch_resources(client, app['yp_chunk_limit'])
                   }

    pods = app['pods']
    to_remove = []
    for pod_id, data in pods.items():
        if pod_id not in new_pod_map:
            to_remove.append(pod_id)

    for pod_id in to_remove:
        pod = pods.pop(pod_id)
        pod.future.set_result(([], None))

    for pod_id, data in new_pod_map.items():
        await update_pod_info(app, pod_id, data)

    log.info("loaded %d pods", len(new_pod_map))


async def yp_updater(app: web.Application):
    log = logging.getLogger('yp')
    client = app['yp_client']
    while True:
        log.debug("calling updater step")
        try:
            await yp_updater_step(app, client)
            log.debug("updater step finished, will sleep for %r", app['yp_update_period'])
        except Exception as e:
            log.warning(f"updater step failed: {e}")

        await asyncio.sleep(app['yp_update_period'])


async def spawn_yp_updater(app: web.Application, args):
    app['yp_update_period'] = args.yp_update_period
    app['yp_chunk_limit'] = args.yp_chunk_limit
    app['yp_client'] = YpClient(address=args.yp_address, config={'token': args.token, 'connect_timeout': 120, 'body_log_size_limit': 1})
    app['yp_updater'] = app.loop.create_task(yp_updater(app))


async def make_app():
    app = web.Application()
    app['sleeping_connections'] = {}
    app['pods'] = {}
    app.on_shutdown.append(shutdown)

    app.add_routes([
        web.get('/{pod}', handle),
        web.get('/{pod}/{revision:\w+}', handle),
    ])

    return app


async def make_test_app():
    app = await make_app()
    app.add_routes([
        web.post('/{pod}', handle_post)
    ])
    return app
