import codecs
import copy
import os

import yaml

_CONFIG = None
_UNDEFINED = object()


class MissingConfigOptionError(Exception):
    def __init__(self, name):
        super(MissingConfigOptionError, self).__init__(f"Configuration option '{name}' is missing.")
        self.name = name


def merge_config_from_env(config, prefix='DRP', env_getter=lambda: os.environ, ignore_errors=False):
    """
    Extracts config context from environment variables.
    Example:
    env = {
        'PATH': '/bin',
        'SEPELIB_port': '8080',
        'SEPELIB_a__b': '/usr/local/www/logs/sepelib-service.log'
    } =>
    config = {
        'port': '8080',
        'a': {
            'b': '/usr/local/www/logs/sepelib-service.log'
        },
    }

    :param config: precreated config
    :param prefix: environment variable prefix
    :param env_getter: function to get a dictionary with environment variables
    :param ignore_errors: should we skip key-value parsing errors
    """
    if not isinstance(prefix, str):
        raise ValueError("Prefix must a string")

    prefix_len = len(prefix) + 1  # SEPELIB + _ (underscore)
    for k, v in env_getter().items():
        if k.startswith(prefix):
            try:
                set_value(config, k[prefix_len:], v, delimiter='__')
            except Exception:
                if not ignore_errors:
                    raise


def load(path=None, defaults=None, env_prefix=None):
    """Loads the specified config file."""
    config = {} if path is None else _load(path)
    if defaults is not None:
        _merge(config, _load(defaults))

    if env_prefix is not None:
        merge_config_from_env(prefix=env_prefix, config=config)

    return config


def get_value(config, name, default=_UNDEFINED):
    """Returns the specified configuration value."""

    value = config

    for key in name.split("."):
        if not isinstance(value, dict):
            raise MissingConfigOptionError(name)

        try:
            value = value[key]
        except KeyError:
            if default is _UNDEFINED:
                raise MissingConfigOptionError(name)

            return default

    return value


def set_value(config, name, value, delimiter='.'):
    """Sets the specified configuration value."""

    keys = name.split(delimiter)

    for key_id, key in enumerate(keys):
        if not isinstance(config, dict):
            cfg_repr = delimiter.join(keys[:key_id]) or "[root]"
            raise Exception(f"Unable to set configuration value '{name}': '{cfg_repr}' is not a dictionary.")

        if key_id == len(keys) - 1:
            config[key] = value
        else:
            config = config.setdefault(key, {})


def _load(path):
    try:
        with codecs.open(path, encoding="utf-8") as config_file:
            file_content = config_file.read()
    except Exception as e:
        raise Exception(f"Error while reading configuration file '{path}': {e}.")
    try:
        return yaml.load(file_content)
    except Exception as e:
        raise Exception(f"Error while parsing configuration file '{path}': {e}.")


def _merge(config, other):
    for key, value in other.iteritems():
        if key in config:
            if isinstance(config[key], dict) and isinstance(value, dict):
                _merge(config[key], value)
        else:
            config[copy.deepcopy(key)] = copy.deepcopy(value)
